#include "CMfClassicKeysDialog.h"

#include <iostream>

#include "utils.h"

#define HAS_STYLE_PROVIDER_ADD_PROVIDER_FOR_DISPLAY GTKMM_CHECK_VERSION(4, 9, 1)

CMcKeysDialog::CMcKeysDialog() :
    m_nCurrentKey(-1),
    m_fAccept(false),
    m_fModifed(false),
    m_VBox(Gtk::Orientation::VERTICAL),
    m_pNewKeyLabel(nullptr),
    m_pCommentLabel(nullptr),
    m_SortButton("Сортировать"),
    m_CreateButton("Создать"),
    m_DeleteButton("Удалить"),
    m_CancelButton("Отмена"),
    m_OkButton("OK") {
    set_destroy_with_parent(true);

    set_title("Ключи аутентификации Mifare Classic");
    set_default_size(450, 300);

    m_VBox.set_margin(5);
    m_VBox.set_expand();
    set_child(m_VBox);

    m_VBox.append(m_ScrolledWindow);
    m_VBox.append(m_BottomBox);

    m_ScrolledWindow.set_child(m_ColumnView);
    m_ScrolledWindow.set_expand();
    m_ListStore = Gio::ListStore<ModelColumns>::create();

    // Set list model and selection model.
    auto selection_model = Gtk::MultiSelection::create(m_ListStore);
    m_ColumnView.set_model(selection_model);
    m_ColumnView.add_css_class("data-table");  // high density table

    // Add the ColumnView's columns:

    // Столбец "№"
    auto factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(sigc::mem_fun(*this, &CMcKeysDialog::on_setup_label));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMcKeysDialog::on_bind_idx));
    auto column = Gtk::ColumnViewColumn::create("№", factory);
    m_ColumnView.append_column(column);

    // Столбец "Ключ"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(sigc::mem_fun(*this, &CMcKeysDialog::on_setup_key));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMcKeysDialog::on_bind_key));
    factory->signal_unbind().connect(sigc::mem_fun(*this, &CMcKeysDialog::on_unbind_key));
    column = Gtk::ColumnViewColumn::create("Ключ", factory);
    m_ColumnView.append_column(column);

    // Столбец "Комментарий"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(sigc::mem_fun(*this, &CMcKeysDialog::on_setup_comment));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMcKeysDialog::on_bind_comment));
    factory->signal_unbind().connect(sigc::mem_fun(*this, &CMcKeysDialog::on_unbind_comment));
    column = Gtk::ColumnViewColumn::create("Комментарий", factory);
    m_ColumnView.append_column(column);

    m_refRClick = Gtk::GestureClick::create();
    m_refRClick->set_button(GDK_BUTTON_SECONDARY);
    m_refRClick->signal_pressed().connect(
        sigc::mem_fun(*this, &CMcKeysDialog::on_columnview_rclick));
    m_ColumnView.add_controller(m_refRClick);

    // Load extra CSS file.
    m_refCssProvider = Gtk::CssProvider::create();
#if HAS_STYLE_PROVIDER_ADD_PROVIDER_FOR_DISPLAY
    Gtk::StyleProvider::add_provider_for_display(get_display(), m_refCssProvider,
                                                 GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
#else
    Gtk::StyleContext::add_provider_for_display(get_display(), m_refCssProvider,
                                                GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
#endif
    std::string sCSS =
        ".key {"
        "  font-family:monospace; "
        "}";
    m_refCssProvider->load_from_data(sCSS);

    // Create actions:
    auto refActionGroup = Gio::SimpleActionGroup::create();
    refActionGroup->add_action("import", sigc::mem_fun(*this, &CMcKeysDialog::on_menu_import));
    insert_action_group("popup1", refActionGroup);

    m_refBuilder = Gtk::Builder::create();
    // Layout the actions in a menubar and toolbar:
    Glib::ustring ui_info =
        "<interface>"
        "  <menu id='menu-popup1'>"
        "    <section>"
        "      <item>"
        "        <attribute name='label' translatable='yes'>Импортировать ключи "
        "считывателя</attribute>"
        "        <attribute name='action'>popup1.import</attribute>"
        "      </item>"
        "    </section>"
        "  </menu>"
        "</interface>";

    try {
        m_refBuilder->add_from_string(ui_info);
    }
    catch (const Glib::Error& ex) {
        std::cerr << "building menus failed: " << ex.what();
    }

    // Get the menu:
    auto gmenu = m_refBuilder->get_object<Gio::Menu>("menu-popup1");
    if (!gmenu)
        g_warning("GMenu not found");

    m_MenuPopup.set_parent(m_ScrolledWindow);
    m_MenuPopup.set_menu_model(gmenu);
    m_MenuPopup.set_has_arrow(false);

    m_BottomBox.set_margin(5);
    m_BottomBox.set_spacing(5);
    m_BottomBox.set_hexpand();
    m_BottomBox.set_halign(Gtk::Align::END);
    m_BottomBox.append(m_SortButton);
    m_BottomBox.append(m_CreateButton);
    m_BottomBox.append(m_DeleteButton);
    m_BottomBox.append(m_CancelButton);
    m_BottomBox.append(m_OkButton);

    m_SortButton.signal_clicked().connect(sigc::mem_fun(*this, &CMcKeysDialog::on_button_sort));
    m_CreateButton.signal_clicked().connect(sigc::mem_fun(*this, &CMcKeysDialog::on_button_create));
    m_DeleteButton.signal_clicked().connect(sigc::mem_fun(*this, &CMcKeysDialog::on_button_delete));
    m_CancelButton.set_halign(Gtk::Align::END);
    m_OkButton.set_halign(Gtk::Align::END);
    m_CancelButton.signal_clicked().connect(sigc::mem_fun(*this, &CMcKeysDialog::on_button_cancel));
    m_OkButton.signal_clicked().connect(sigc::mem_fun(*this, &CMcKeysDialog::on_button_ok));

    set_default_widget(m_OkButton);
}

CMcKeysDialog::~CMcKeysDialog() {
    m_MenuPopup.unparent();
}

void CMcKeysDialog::Init(Glib::RefPtr<CMfClassicKeysSettings> refSets,
                         Glib::RefPtr<CMifareReaderSettings> refRdSets) {
    m_refSets = refSets;
    m_refRdSets = refRdSets;
    m_fModifed = false;
    UpdateView();
    if (m_nCurrentKey != -1) {
        auto pModel = dynamic_cast<Gtk::MultiSelection*>(m_ColumnView.get_model().get());
        auto nIdx = FindKey(m_nCurrentKey);
        if (nIdx != GTK_INVALID_LIST_POSITION)
            pModel->select_item(nIdx, true);
    }
}

void CMcKeysDialog::on_setup_label(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    list_item->set_child(*Gtk::make_managed<Gtk::Label>());
}

void CMcKeysDialog::on_setup_key(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto pLabel = Gtk::make_managed<Gtk::EditableLabel>();
    list_item->set_child(*pLabel);
    pLabel->add_css_class("key");
    pLabel->property_editing().signal_changed().connect(sigc::bind(
        sigc::mem_fun(*this, &CMcKeysDialog::on_key_label_editing_change), list_item.get()));
}

void CMcKeysDialog::on_setup_comment(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto pLabel = Gtk::make_managed<Gtk::EditableLabel>();
    list_item->set_child(*pLabel);
    pLabel->set_editable(false);
    pLabel->property_editing().signal_changed().connect(sigc::bind(
        sigc::mem_fun(*this, &CMcKeysDialog::on_comment_label_editing_change), list_item.get()));
}

void CMcKeysDialog::on_bind_idx(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_label(Glib::ustring::format(list_item->get_position()));
}

void CMcKeysDialog::on_bind_key(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::EditableLabel*>(list_item->get_child());
    if (!label)
        return;
    if (col->m_property_key.get_value() == -1)
        m_pNewKeyLabel = label;
    Glib::Binding::bind_property_value(col->m_property_key.get_proxy(), label->property_text(),
                                       Glib::Binding::Flags::SYNC_CREATE,
                                       sigc::mem_fun(*this, &CMcKeysDialog::key_transform_to),
                                       sigc::mem_fun(*this, &CMcKeysDialog::key_transform_from));
}

void CMcKeysDialog::on_unbind_key(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::EditableLabel*>(list_item->get_child());
    if (!label)
        return;
    if (label == m_pNewKeyLabel)
        m_pNewKeyLabel = nullptr;
}

bool CMcKeysDialog::key_transform_to(const GValue* a, GValue* b) {
    if (a->g_type != G_TYPE_LONG)
        return false;
    auto nSrc = a->data->v_long;
    if (b->g_type != G_TYPE_STRING)
        return false;
    if (nSrc != -1)
        g_value_set_string(b, ilr::CMifareClassicKey(nSrc).ToString().c_str());
    else
        g_value_set_string(b, "");
    return true;
}

bool CMcKeysDialog::key_transform_from(const GValue* a, GValue* b) {
    if (a->g_type != G_TYPE_STRING)
        return false;
    if (b->g_type != G_TYPE_LONG)
        return false;
    auto nDest = b->data->v_long;
    auto pValue = g_value_get_string(a);
    ilr::CMifareClassicKey key;
    if (key.TryParse(pValue)) {
        nDest = (glong)key.m_nKey;
        return true;
    }
    return false;
}

void CMcKeysDialog::on_key_label_editing_change(Gtk::ListItem* list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::EditableLabel*>(list_item->get_child());
    if (!label)
        return;
    if (!label->get_editing()) {
        ilr::CMifareClassicKey oNewKey;
        if (oNewKey.TryParse(label->get_text().c_str())) {
            if (col->m_property_key.get_value() != oNewKey) {
                col->m_property_key = oNewKey;
                m_fModifed = true;
                if (m_pCommentLabel != nullptr)
                    m_pCommentLabel->set_editable();
                auto nCount = m_ListStore->get_n_items();
                auto item = m_ListStore->get_item(nCount - 1);
                if (item->m_property_key.get_value() != -1) {
                    m_ListStore->append(ModelColumns::create(-1, ""));
                }
            }
        }
    }
}

void CMcKeysDialog::on_bind_comment(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::EditableLabel*>(list_item->get_child());
    if (!label)
        return;
    m_pCommentLabel = label;
    auto fValidKey = (col->m_property_key.get_value() != -1);
    label->set_editable(fValidKey);
    Glib::Binding::bind_property(col->m_property_comment.get_proxy(), label->property_text(),
                                 Glib::Binding::Flags::SYNC_CREATE);
}

void CMcKeysDialog::on_unbind_comment(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    m_pCommentLabel = nullptr;
}

void CMcKeysDialog::on_comment_label_editing_change(Gtk::ListItem* list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::EditableLabel*>(list_item->get_child());
    if (!label)
        return;
    if (!label->get_editing()) {
        auto text = label->get_text();
        if (col->m_property_comment.get_value() != text) {
            col->m_property_comment.set_value(text);
            m_fModifed = true;
        }
    }
}

void CMcKeysDialog::on_columnview_rclick(int n_press, double x, double y) {
    const Gdk::Rectangle rect(x, y, 1, 1);
    m_MenuPopup.set_pointing_to(rect);
    m_MenuPopup.popup();
}

void CMcKeysDialog::on_menu_import() {
    guint nInsertPos = m_ListStore->get_n_items() - 1;
    for (size_t i = 0; i < std::size(m_refRdSets->m_aMcRdKeys); i++) {
        for (size_t j = 0; j < std::size(m_refRdSets->m_aMcRdKeys[i]); j++) {
            if (GET_BIT(m_refRdSets->m_aMcValidRdKeys[i], j)) {
                auto& info = m_refRdSets->m_aMcRdKeys[i][j];
                // Если ключа ещё нет в списке, добавляем его в конец списка
                if (FindKey(info.m_Key) == GTK_INVALID_LIST_POSITION) {
                    m_ListStore->insert(nInsertPos++,
                                        ModelColumns::create(info.m_Key, info.m_sComment));
                    m_fModifed = true;
                }
            }
        }
    }
}

void CMcKeysDialog::on_button_sort() {
    m_ListStore->sort(sigc::mem_fun(*this, &CMcKeysDialog::on_sort_compare));
    m_fModifed = true;
}

int CMcKeysDialog::on_sort_compare(const Glib::RefPtr<const ModelColumns>& a,
                                   const Glib::RefPtr<const ModelColumns>& b) {
    auto nCmp = (a->m_property_key.get_value() - b->m_property_key.get_value());
    if (nCmp != 0) {
        if (a->m_property_key.get_value() == -1)
            nCmp = 1;
        else if (b->m_property_key.get_value() == -1)
            nCmp = -1;
        else
            nCmp = (nCmp < 0) ? -1 : 1;
    }
    return (int)nCmp;
}

void CMcKeysDialog::on_button_delete() {
    auto pModel = dynamic_cast<Gtk::MultiSelection*>(m_ColumnView.get_model().get());
    auto nCount = pModel->get_n_items();
    auto pSelection = pModel->get_selection(0, nCount);
    if (pSelection->is_empty())
        return;

    int nOffs = 0;
    for (auto it = pSelection->cbegin(); it != pSelection->cend(); it++) {
        auto nIdx = (*it - nOffs);
        auto item = m_ListStore->get_item(nIdx).get();
        if (item->m_property_key.get_value() == -1)
            continue;
        m_ListStore->remove(nIdx);
        ++nOffs;
    }
    m_fModifed = true;
}

void CMcKeysDialog::on_button_create() {
    if (m_pNewKeyLabel != nullptr)
        m_pNewKeyLabel->start_editing();
}

void CMcKeysDialog::on_button_cancel() {
    set_visible(false);
}

void CMcKeysDialog::on_button_ok() {
    SaveViewData();
    m_fAccept = true;
    set_visible(false);
}

void CMcKeysDialog::UpdateView() {
    m_ListStore->remove_all();

    for (auto& rInfo : m_refSets->m_oKeys)
        m_ListStore->append(ModelColumns::create(rInfo.m_Key, rInfo.m_sComment));

    m_ListStore->append(ModelColumns::create(-1, ""));
}

void CMcKeysDialog::SaveViewData() {
    if (m_fModifed) {
        CMcKeyList oKeys;
        CMcKeyInfo rInfo;
        auto nCount = m_ListStore->get_n_items();
        for (guint i = 0; i < nCount; i++) {
            auto item = m_ListStore->get_item(i);
            auto nKey = item->m_property_key.get_value();
            if (-1 == nKey)
                continue;
            rInfo.m_Key = nKey;
            rInfo.m_sComment = item->m_property_comment.get_value();
            oKeys.emplace_back(std::move(rInfo));
        }
        m_refSets->m_oKeys = std::move(oKeys);
        m_refSets->SetModifiedFlag();
        m_refSets->Save(GetMcKeysFilePath());
        m_fModifed = false;
    }
    auto pModel = dynamic_cast<Gtk::MultiSelection*>(m_ColumnView.get_model().get());
    auto pSelection = pModel->get_selection(0, pModel->get_n_items());
    m_nCurrentKey = pSelection->is_empty() ?
                        -1 :
                        m_ListStore->get_item(*pSelection->cbegin())->m_property_key.get_value();
}

guint CMcKeysDialog::FindKey(const ilr::CMifareClassicKey& key) const {
    auto nCount = m_ListStore->get_n_items();
    for (guint i = 0; i < nCount; i++)
        if (m_ListStore->get_item(i)->m_property_key.get_value() == key)
            return i;
    return GTK_INVALID_LIST_POSITION;
}
