/* gtkmm example Copyright (C) 2023 gtkmm development team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include "CMainWindow.h"

#include <fstream>  // для std::fstream
#include <iomanip>  // для std::put_time
#include <iostream>

#include "CAppSettings.h"

const char kLogLevelChars[] = {'-', 'E', 'W', 'I', 'D'};

CMainWindow::CMainWindow() :
    m_VBox(Gtk::Orientation::VERTICAL),
    m_Button_Rescan("Обновить"),
    m_Button_Open("Открыть..."),
    m_oCvtDlg(),
    m_oILG(false) {
    set_title("Demo SDK Guard");
    set_default_size(700, 250);

    m_sLogPath = GetLogFilePath();

    m_VBox.set_margin(5);
    set_child(m_VBox);

    // Add the ColumnView, inside a ScrolledWindow, with the button underneath:
    m_ScrolledWindow.set_child(m_ColumnView);

    // Only show the scrollbars when they are necessary:
    m_ScrolledWindow.set_policy(Gtk::PolicyType::AUTOMATIC, Gtk::PolicyType::AUTOMATIC);
    m_ScrolledWindow.set_expand();

    m_VBox.append(m_ScrolledWindow);
    m_VBox.append(m_ButtonBox);

    m_ButtonBox.append(m_Button_Rescan);
    m_ButtonBox.append(m_PortEntry);
    m_ButtonBox.append(m_oModelDropDown);
    m_ButtonBox.append(m_Button_Open);
    m_ButtonBox.set_margin(5);
    m_ButtonBox.set_expand(false);
    m_Button_Rescan.signal_clicked().connect(sigc::mem_fun(*this, &CMainWindow::on_button_rescan));
    m_PortEntry.set_expand();
    m_PortEntry.set_halign(Gtk::Align::END);
    m_PortEntry.set_placeholder_text("Имя порта");

    m_refModelStringList = Gtk::StringList::create({});
    m_refModelStringList->append("Авто");
    for (size_t i = 1; i < std::size(ilg::kConverterModelNames); i++)
        m_refModelStringList->append(ilg::kConverterModelNames[i]);
    m_oModelDropDown.set_model(m_refModelStringList);
    m_oModelDropDown.set_halign(Gtk::Align::END);

    m_Button_Open.set_hexpand(true);
    m_Button_Open.set_halign(Gtk::Align::END);
    m_Button_Open.signal_clicked().connect(sigc::mem_fun(*this, &CMainWindow::on_button_open));

    m_oDisp.connect(sigc::mem_fun(*this, &CMainWindow::on_ilg));

    // Create the List model:
    m_ListStore = Gio::ListStore<ModelColumns>::create();

    // Set list model and selection model.
    auto selection_model = Gtk::SingleSelection::create(m_ListStore);
    selection_model->set_autoselect(false);
    selection_model->set_can_unselect(true);
    m_ColumnView.set_model(selection_model);
    m_ColumnView.add_css_class("data-table");  // high density table
    selection_model->signal_selection_changed().connect(
        sigc::mem_fun(*this, &CMainWindow::on_selection_changed));

    // Make the columns reorderable.
    // This is not necessary, but it's nice to show the feature.
    m_ColumnView.set_reorderable(true);
    // m_ColumnView.set_enable_rubberband(true);

    // Add the ColumnView's columns:

    // Столбец "Порт"
    auto factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CMainWindow::on_setup_label), Gtk::Align::END));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMainWindow::on_bind_port));
    auto column = Gtk::ColumnViewColumn::create("Порт", factory);
    m_ColumnView.append_column(column);

    // Столбец "Модель"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CMainWindow::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMainWindow::on_bind_model));
    column = Gtk::ColumnViewColumn::create("Модель", factory);
    m_ColumnView.append_column(column);

    // Столбец "С/н"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CMainWindow::on_setup_label), Gtk::Align::END));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMainWindow::on_bind_sn));
    column = Gtk::ColumnViewColumn::create("С/н", factory);
    m_ColumnView.append_column(column);

    // Столбец "Версия прошивки"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CMainWindow::on_setup_label), Gtk::Align::END));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMainWindow::on_bind_fwversion));
    column = Gtk::ColumnViewColumn::create("Версия прошивки", factory);
    m_ColumnView.append_column(column);

    // Столбец "Сборка прошивки"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CMainWindow::on_setup_label), Gtk::Align::END));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMainWindow::on_bind_fwbuild));
    column = Gtk::ColumnViewColumn::create("Сборка прошивки", factory);
    m_ColumnView.append_column(column);

    // Столбец "Режим"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CMainWindow::on_setup_label), Gtk::Align::END));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CMainWindow::on_bind_mode));
    column = Gtk::ColumnViewColumn::create("Режим", factory);
    m_ColumnView.append_column(column);

    m_refRClick = Gtk::GestureClick::create();
    m_refRClick->set_button(GDK_BUTTON_SECONDARY);
    m_refRClick->signal_pressed().connect(sigc::mem_fun(*this, &CMainWindow::on_columnview_rclick));
    m_ColumnView.add_controller(m_refRClick);

    m_ColumnView.signal_activate().connect(
        sigc::mem_fun(*this, &CMainWindow::on_columnview_activate));

    m_oCvtDlg.set_transient_for(*this);
    m_oCvtDlg.set_modal();
    m_oCvtDlg.set_hide_on_close();
    m_oCvtDlg.signal_hide().connect(sigc::mem_fun(*this, &CMainWindow::on_converter_dialog_hide));

    // Create actions:
    auto refActionGroup = Gio::SimpleActionGroup::create();
    m_refILUsbAction = refActionGroup->add_action_bool(
        "ilusb", sigc::mem_fun(*this, &CMainWindow::on_menu_ilusb), false);
    m_refTPUsbAction = refActionGroup->add_action_bool(
        "tpusb", sigc::mem_fun(*this, &CMainWindow::on_menu_tpusb), false);
    m_refServerAction = refActionGroup->add_action_bool(
        "server", sigc::mem_fun(*this, &CMainWindow::on_menu_server), false);
    m_refClientAction = refActionGroup->add_action_bool(
        "client", sigc::mem_fun(*this, &CMainWindow::on_menu_client), false);

    m_refShowUnidentifiedAction = refActionGroup->add_action_bool(
        "showunidentified", sigc::mem_fun(*this, &CMainWindow::on_menu_showunidentified), false);
    m_refLogLevelAction = refActionGroup->add_action_radio_integer(
        "loglevel", sigc::mem_fun(*this, &CMainWindow::on_menu_loglevel), 1);
    m_refWriteLogAction = refActionGroup->add_action_bool(
        "writelog", sigc::mem_fun(*this, &CMainWindow::on_menu_writelog), false);
    m_refClearLogAtStartupAction = refActionGroup->add_action_bool(
        "clearLogatstartup", sigc::mem_fun(*this, &CMainWindow::on_menu_clearlogatstartup), false);
    refActionGroup->add_action("clearlog", sigc::mem_fun(*this, &CMainWindow::on_menu_clearlog));

    insert_action_group("popup1", refActionGroup);

    m_refBuilder = Gtk::Builder::create();
    // Layout the actions in a menubar and toolbar:
    Glib::ustring ui_info =
        "<interface>"
        "  <menu id='menu-popup1'>"
        "    <submenu>"
        "      <attribute name='label' translatable='yes'>Типы конвертеров</attribute>"
        "      <section>"
        "        <item>"
        "          <attribute name='label' translatable='yes'>USB конвертеры Iron Logic</attribute>"
        "          <attribute name='action'>popup1.ilusb</attribute>"
        "        </item>"
        "        <item>"
        "          <attribute name='label' translatable='yes'>USB конвертеры сторонних "
        "производителей</attribute>"
        "          <attribute name='action'>popup1.tpusb</attribute>"
        "        </item>"
        "        <item>"
        "          <attribute name='label' translatable='yes'>IP конвертеры в режиме "
        "\"Сервер\"</attribute>"
        "          <attribute name='action'>popup1.server</attribute>"
        "        </item>"
        "        <item>"
        "          <attribute name='label' translatable='yes'>IP конвертеры в режиме "
        "\"Клиент\"</attribute>"
        "          <attribute name='action'>popup1.client</attribute>"
        "        </item>"
        "      </section>"
        "    </submenu>"
        "    <section>"
        "      <item>"
        "        <attribute name='label' translatable='yes'>Показывать неопознанные "
        "устройства</attribute>"
        "        <attribute name='action'>popup1.showunidentified</attribute>"
        "      </item>"
        "    </section>"
        "    <submenu>"
        "      <attribute name='label' translatable='yes'>Уровень лога</attribute>"
        "      <section>"
        "        <item>"
        "          <attribute name='label' translatable='yes'>Ошибки</attribute>"
        "          <attribute name='action'>popup1.loglevel</attribute>"
        "          <attribute name='target' type='i'>1</attribute>"
        "        </item>"
        "        <item>"
        "          <attribute name='label' translatable='yes'>Предупреждение</attribute>"
        "          <attribute name='action'>popup1.loglevel</attribute>"
        "          <attribute name='target' type='i'>2</attribute>"
        "        </item>"
        "        <item>"
        "          <attribute name='label' translatable='yes'>Информация</attribute>"
        "          <attribute name='action'>popup1.loglevel</attribute>"
        "          <attribute name='target' type='i'>3</attribute>"
        "        </item>"
        "        <item>"
        "          <attribute name='label' translatable='yes'>Отладка</attribute>"
        "          <attribute name='action'>popup1.loglevel</attribute>"
        "          <attribute name='target' type='i'>4</attribute>"
        "        </item>"
        "      </section>"
        "    </submenu>"
        "    <section>"
        "      <item>"
        "        <attribute name='label' translatable='yes'>Писать лог файл</attribute>"
        "        <attribute name='action'>popup1.writelog</attribute>"
        "      </item>"
        "      <item>"
        "        <attribute name='label' translatable='yes'>Очищать лог при запуске</attribute>"
        "        <attribute name='action'>popup1.clearLogatstartup</attribute>"
        "      </item>"
        "      <item>"
        "        <attribute name='label' translatable='yes'>Очистить лог файл</attribute>"
        "        <attribute name='action'>popup1.clearlog</attribute>"
        "      </item>"
        "    </section>"
        "  </menu>"
        "</interface>";

    try {
        m_refBuilder->add_from_string(ui_info);
    }
    catch (const Glib::Error& ex) {
        std::cerr << "building menus failed: " << ex.what();
    }

    // Get the menu:
    auto gmenu = m_refBuilder->get_object<Gio::Menu>("menu-popup1");
    if (!gmenu)
        g_warning("GMenu not found");

    m_MenuPopup.set_parent(m_ScrolledWindow);
    m_MenuPopup.set_menu_model(gmenu);
    m_MenuPopup.set_has_arrow(false);

    signal_destroy().connect(sigc::mem_fun(*this, &CMainWindow::on_form_destroy));

    g_AppSet.Load();
    m_refILUsbAction->change_state((g_AppSet.m_nConverterTypes & ILG_CONVERTER_ILUSB) != 0);
    m_refTPUsbAction->change_state((g_AppSet.m_nConverterTypes & ILG_CONVERTER_TPUSB) != 0);
    m_refServerAction->change_state((g_AppSet.m_nConverterTypes & ILG_CONVERTER_SERVER) != 0);
    m_refClientAction->change_state((g_AppSet.m_nConverterTypes & ILG_CONVERTER_CLIENT) != 0);
    m_refShowUnidentifiedAction->change_state(g_AppSet.m_fShowUnidentified);
    m_refWriteLogAction->change_state(g_AppSet.m_fDebugLog);
    m_refClearLogAtStartupAction->change_state(g_AppSet.m_fClearLogAtStartup);
    m_refLogLevelAction->change_state((int)g_AppSet.m_nLogLevel);

    if (g_AppSet.m_fClearLogAtStartup)
        ClearLog();
    UpdateDebugLog();

    m_oILG.Init();
    m_oSearch = m_oILG.GetSearch();
    m_oSearch.SetMessageCallback(&on_ilg_message, this);
    m_oSearch.EnableMessageQueue();

    UpdateCvtTypes();
    UpdateListView();
    UpdateProperties();
    m_oSearch.SetAutoScan(true, false);
}

CMainWindow::~CMainWindow() {
    m_MenuPopup.unparent();
}

void CMainWindow::UpdateListView() {
    auto nCount = m_oSearch.GetConverterCount();
    m_ListStore->remove_all();
    ilg_converter_info rInfo;
    for (size_t i = 0; i < nCount; i++) {
        m_oSearch.GetConverterInfo(i, rInfo);
        if ((rInfo.nModel != ILG_CONVERTER_MODEL_UNKNOWN) || g_AppSet.m_fShowUnidentified)
            m_ListStore->append(ModelColumns::create(rInfo.nPortType, rInfo.pszPortName,
                                                     rInfo.nModel, rInfo.nSn, rInfo.nFwVersion,
                                                     rInfo.nFwBuildDate, rInfo.nMode));
    }
}

bool CMainWindow::DoOpen() {
    auto sPortName = m_PortEntry.get_text();
    if (sPortName.empty())
        return false;

    ilg_port_type nPortType = ILG_PORT_TYPE_UNKNOWN;
    ilg_converter_model nModel = ILG_CONVERTER_MODEL_UNKNOWN;

    ilg::CConverter oConverter;
    auto nCount = m_ListStore->get_n_items();
    for (size_t i = 0; i < nCount; i++) {
        auto p = m_ListStore->get_item(i);
        if (p->m_sPortName == sPortName) {
            nPortType = p->m_nPortType;
            nModel = p->m_nModel;
            break;
        }
    }

    if (ILG_PORT_TYPE_UNKNOWN == nPortType)
        nPortType = GetPortTypeByName(sPortName.c_str());
    if (ILG_CONVERTER_MODEL_UNKNOWN == nModel) {
        auto nIdx = m_oModelDropDown.get_selected();
        if (nIdx != GTK_INVALID_LIST_POSITION)
            nModel = (ilg_converter_model)nIdx;
    }

    oConverter = m_oILG.GetConverter(nPortType, sPortName.c_str());
    if (nModel != ILG_CONVERTER_MODEL_UNKNOWN) {
        ilg_converter_options rParams;
        oConverter.GetOptions(rParams);
        rParams.nConnectModel = nModel;
        oConverter.SetOptions(rParams);
    }
    try {
        oConverter.Connect();
    }
    catch (const std::exception& e) {
        // std::cerr << e.what() << '\n';
        ShowMessage(e.what(), Gtk::MessageType::ERROR);
        return false;
    }

    m_oCvtDlg.SetConverter(oConverter);
    m_oCvtDlg.set_visible(true);

    return true;
}

ilg_port_type CMainWindow::GetPortTypeByName(const char* pszPortName) {
    if (*pszPortName == '\0')
        return ILG_PORT_TYPE_UNKNOWN;
    if (strncmp(pszPortName, "/dev/tty", 8) == 0)
        return ILG_PORT_TYPE_COMPORT;
    auto p = strchr(pszPortName, '@');
    if (p != nullptr) {
        int nSn;
        uint nTcpPort;
        if (sscanf(pszPortName, "%d@%u", &nSn, &nTcpPort) == 2)
            return ILG_PORT_TYPE_CLIENT;
        // 30:00000000@192.168.1.98:25001
        return ILG_PORT_TYPE_PROXY;
    }
    uint a[4];
    if (sscanf(pszPortName, "%u.%u.%u.%u", &a[0], &a[1], &a[2], &a[3]) == 4)
        return ILG_PORT_TYPE_SERVER;
    return ILG_PORT_TYPE_UNKNOWN;
}

void CMainWindow::UpdateProperties() {
    auto pModel = dynamic_cast<Gtk::SingleSelection*>(m_ColumnView.get_model().get());
    auto item = std::dynamic_pointer_cast<ModelColumns>(pModel->get_selected_item());
    if (item) {
        m_PortEntry.set_text(item->m_sPortName);
        m_oModelDropDown.set_selected((guint)item->m_nModel);
        m_Button_Open.set_sensitive(true);
    }
    else {
        m_PortEntry.set_text("");
        m_oModelDropDown.set_selected(0);
        m_Button_Open.set_sensitive(false);
    }
}

void CMainWindow::UpdateDebugLog() {
    if (g_AppSet.m_fDebugLog) {
        m_oILG.SetLogLevel(g_AppSet.m_nLogLevel);
        m_oILG.SetLogCallback(&on_ilg_log, this);
    }
    else
        m_oILG.SetLogCallback(nullptr);
}

void CMainWindow::ClearLog() {
    // Очищаем лог файл
    if (std::filesystem::exists(m_sLogPath)) {
        std::ofstream file(m_sLogPath.c_str(), std::ios_base::out | std::ios_base::trunc);
        file.close();
    }
}

std::filesystem::path CMainWindow::GetLogFilePath() const {
    char szCfgPath[PATH_MAX];
    auto count = readlink("/proc/self/exe", szCfgPath, std::size(szCfgPath) - 1);
    szCfgPath[count] = '\0';
    std::stringstream ss;
    ss << szCfgPath << ".log";
    return ss.str();
}

void CMainWindow::UpdateCvtTypes() {
    ilg_search_options rOptions;
    m_oSearch.GetOptions(rOptions);
    rOptions.nConverterTypes = g_AppSet.m_nConverterTypes;
    m_oSearch.SetOptions(rOptions);
}

void CMainWindow::ShowMessage(const std::string& sMessage, Gtk::MessageType nType) {
    if (nullptr == m_refDialog)
        m_refDialog.reset(new Gtk::MessageDialog("Демо", false, nType));
    m_refDialog->set_transient_for(*this);
    m_refDialog->set_message(sMessage);
    m_refDialog->show();
    m_refDialog->signal_response().connect(
        sigc::bind(sigc::mem_fun(*this, &CMainWindow::on_dialog_response)));
}

void CMainWindow::on_button_rescan() {
    m_ListStore->remove_all();

    // m_oSearch.Scan();
    m_oCommand = m_oSearch.Begin_Scan(true);
    m_refProgressDialog.reset(new CProgressDialog());
    m_refProgressDialog->set_transient_for(*this);
    m_refProgressDialog->set_modal();
    m_refProgressDialog->set_hide_on_close();
    m_refProgressDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CMainWindow::on_progress_dialog_hide));
    m_refProgressDialog->signal_cancel().connect(
        sigc::mem_fun(*this, &CMainWindow::on_progress_dialog_cancel));
    m_refProgressDialog->signal_progress().connect(
        sigc::mem_fun(*this, &CMainWindow::on_progress_dialog_progress));
    m_refProgressDialog->set_visible(true);
    UpdateListView();
}

void CMainWindow::on_progress_dialog_cancel() {
    m_oCommand.Cancel();
    m_oCommand = nullptr;
    m_refProgressDialog = nullptr;
}

bool CMainWindow::on_progress_dialog_progress(size_t& nCurrent, size_t& nTotal) {
    m_oCommand.GetProgress(nCurrent, nTotal);

    auto nStatus = m_oCommand.GetStatus();
    if (nStatus != ILG_E_PENDING) {
        m_oCommand = nullptr;
        m_refProgressDialog = nullptr;
        if (ILG_FAILED(nStatus))
            ShowMessage(ilg_get_error_text(nStatus), Gtk::MessageType::ERROR);
        return false;
    }
    return true;
}

void CMainWindow::on_progress_dialog_hide() {
    m_oCommand = nullptr;
    m_refProgressDialog = nullptr;
}

void CMainWindow::on_button_open() {
    DoOpen();
}

void CMainWindow::on_setup_label(const Glib::RefPtr<Gtk::ListItem>& list_item, Gtk::Align halign) {
    list_item->set_child(*Gtk::make_managed<Gtk::Label>("", halign));
}

void CMainWindow::on_bind_port(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(col->m_sPortName);
}

void CMainWindow::on_bind_model(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(ilg::kConverterModelNames[col->m_nModel]);
}

void CMainWindow::on_bind_sn(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    if (col->m_nSn != -1)
        label->set_text(Glib::ustring::sprintf("%d", col->m_nSn));
    else
        label->set_text("");
}

void CMainWindow::on_bind_fwversion(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(ilg::VersionToStr(col->m_nFwVersion));
}

void CMainWindow::on_bind_fwbuild(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(col->m_tFwBuild ? ilg::TimeToStr(col->m_tFwBuild) : "");
}

void CMainWindow::on_bind_mode(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(ilg::kConverterModeNames[col->m_nMode]);
}

void CMainWindow::on_selection_changed(guint, guint) {
    UpdateProperties();
}

void CMainWindow::on_converter_dialog_hide() {
    ilg::CConverter oConverter;
    m_oCvtDlg.SetConverter(oConverter);
    m_oCvtDlg.set_visible(false);
}

void CMainWindow::on_menu_ilusb() {
    bool fActive = false;
    m_refILUsbAction->get_state(fActive);
    fActive = !fActive;
    m_refILUsbAction->change_state(fActive);
    g_AppSet.SetConverterType(ILG_CONVERTER_ILUSB, fActive);
    UpdateCvtTypes();
}

void CMainWindow::on_menu_tpusb() {
    bool fActive = false;
    m_refTPUsbAction->get_state(fActive);
    fActive = !fActive;
    m_refTPUsbAction->change_state(fActive);
    g_AppSet.SetConverterType(ILG_CONVERTER_TPUSB, fActive);
    UpdateCvtTypes();
}

void CMainWindow::on_menu_server() {
    bool fActive = false;
    m_refServerAction->get_state(fActive);
    fActive = !fActive;
    m_refServerAction->change_state(fActive);
    g_AppSet.SetConverterType(ILG_CONVERTER_SERVER, fActive);
    UpdateCvtTypes();
}

void CMainWindow::on_menu_client() {
    bool fActive = false;
    m_refClientAction->get_state(fActive);
    fActive = !fActive;
    m_refClientAction->change_state(fActive);
    g_AppSet.SetConverterType(ILG_CONVERTER_CLIENT, fActive);
    UpdateCvtTypes();
}

void CMainWindow::on_menu_showunidentified() {
    bool fActive = false;
    m_refShowUnidentifiedAction->get_state(fActive);
    fActive = !fActive;
    m_refShowUnidentifiedAction->change_state(fActive);
    g_AppSet.SetShowUnidentified(fActive);
    UpdateListView();
}

void CMainWindow::on_menu_loglevel(int parameter) {
    m_refLogLevelAction->change_state(parameter);

    if ((parameter > 0) && (parameter < ILG_LOG_LEVEL_SIZE)) {
        g_AppSet.SetLogLevel((ilg_log_level)parameter);
        m_oILG.SetLogLevel(g_AppSet.m_nLogLevel);
    }
}

void CMainWindow::on_menu_writelog() {
    bool fActive = false;
    m_refWriteLogAction->get_state(fActive);
    fActive = !fActive;
    m_refWriteLogAction->change_state(fActive);
    g_AppSet.SetDebugLog(fActive);
    UpdateDebugLog();
}

void CMainWindow::on_menu_clearlogatstartup() {
    bool fActive = false;
    m_refClearLogAtStartupAction->get_state(fActive);
    fActive = !fActive;
    m_refClearLogAtStartupAction->change_state(fActive);
    g_AppSet.SetClearLogAtStartup(fActive);
}

void CMainWindow::on_menu_clearlog() {
    ClearLog();
}

void CMainWindow::on_columnview_rclick(int n_press, double x, double y) {
    const Gdk::Rectangle rect(x, y, 1, 1);
    m_MenuPopup.set_pointing_to(rect);
    m_MenuPopup.popup();
}

void CMainWindow::on_columnview_activate(guint n) {
    DoOpen();
}

void CMainWindow::on_form_destroy() {
    m_oILG.SetLogCallback(nullptr);
    if (g_AppSet.IsModified())
        g_AppSet.Save();
}

void CMainWindow::on_dialog_response(int) {
    m_refDialog = nullptr;
}

void ILG_CALL CMainWindow::on_ilg_log(ilg_log_level nLevel, const char* pContext,
                                      const char* pMessage, void* pUserData) {
    try {
        auto p = (CMainWindow*)pUserData;
        std::fstream file(p->m_sLogPath.c_str(), std::ios_base::out | std::ios_base::app);
        auto t = std::time(nullptr);
        auto tmb = std::localtime(&t);
        file << std::put_time(tmb, "%d-%m-%Y %H:%M:%S") << " [" << kLogLevelChars[nLevel] << ' '
             << pContext << "] " << pMessage << std::endl;
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << '\n';
    }
}

void ILG_CALL CMainWindow::on_ilg_message(ilg_search_msg nMsg, const void* pMsgData,
                                          void* pUserData) {
    auto p = (CMainWindow*)pUserData;
    p->m_oDisp.emit();
}

void CMainWindow::on_ilg() {
    bool fUpdateListView = false;
    ilg_search_msg nMsg;
    const void* pMsgData;
    while (m_oSearch.GetMessage(nMsg, pMsgData)) {
        switch (nMsg) {
        case ILG_SEARCH_MSG_COMMAND_FINISH:
            if (m_refProgressDialog != nullptr) {
                auto nStatus = m_oCommand.GetStatus();
                m_refProgressDialog = nullptr;
                m_oCommand = nullptr;
                if (ILG_FAILED(nStatus))
                    ShowMessage(ilg_get_error_text(nStatus), Gtk::MessageType::ERROR);
            }
            break;

        case ILG_SEARCH_MSG_LIST_CHANGED:
            fUpdateListView = true;
            break;
        }
    }
    if (fUpdateListView)
        UpdateListView();
}
