#include "CEcDialog.h"

#include <codecvt>  // для std::wstring_convert
#include <iomanip>  // для std::put_time
#include <iostream>
#include <sstream>  // для std::stringstream

#include "utils.h"

CEcDialog::CEcDialog() :
    m_Box(Gtk::Orientation::VERTICAL),
    m_EnableCheckButton("Активировать управление питанием"),
    m_ExtReaderCheckButton("Внешний контрольный считыватель"),
    m_CardOpenCheckButton("Не блокировать функцию открывания двери ключом"),
    m_ExitOffCheckButton("По датчику двери выключать немедленно при выходе из комнаты"),
    m_DelayLabel("Задержка на выключение (сек):"),
    m_refDelayAdjustment(Gtk::Adjustment::create(0, 0, 255.0)),
    m_DelaySpinButton(m_refDelayAdjustment),
    m_InvertCheckButton("Включение питания нулевым уровнем"),
    m_ScheduleCheckButton("Использовать расписание временной зоны #7"),
    m_ReadButton("Читать"),
    m_WriteButton("Записать") {
    set_destroy_with_parent(true);
    set_title("Режим ЭлектроКонтроль");
    set_default_size(660, 285);

    m_Box.set_margin(5);
    set_child(m_Box);

    m_Box.append(m_EnableCheckButton);
    m_Box.append(m_ExtReaderCheckButton);
    m_Box.append(m_CardOpenCheckButton);
    m_Box.append(m_ExitOffCheckButton);
    m_Box.append(m_DelayLabel);
    m_Box.append(m_DelaySpinButton);
    m_Box.append(m_InvertCheckButton);
    m_Box.append(m_ScheduleCheckButton);
    m_Box.append(m_ScheduleLabel);
    m_Box.append(m_BottomBox);

    m_EnableCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CEcDialog::on_button_enabled));
    m_ScheduleCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CEcDialog::on_button_schedule));

    m_DelayLabel.set_halign(Gtk::Align::START);
    m_DelaySpinButton.set_margin_start(21);
    m_DelaySpinButton.set_hexpand(false);
    m_DelaySpinButton.set_size_request(100);
    m_ScheduleLabel.set_halign(Gtk::Align::START);
    m_ScheduleLabel.set_margin_start(21);
    m_ScheduleLabel.signal_activate_link().connect(
        sigc::mem_fun(*this, &CEcDialog::on_schedule_label_link), false);
    m_BottomBox.set_margin(5);
    m_BottomBox.set_spacing(5);
    m_BottomBox.set_expand();
    m_BottomBox.set_valign(Gtk::Align::END);
    m_BottomBox.append(m_ReadButton);
    m_BottomBox.append(m_WriteButton);

    m_ReadButton.signal_clicked().connect(sigc::mem_fun(*this, &CEcDialog::on_button_read));
    m_WriteButton.signal_clicked().connect(sigc::mem_fun(*this, &CEcDialog::on_button_write));
}

CEcDialog::~CEcDialog() {
}

void CEcDialog::Init(const ilg::CController& oController) {
    m_oController = oController.Clone();
    DoRead();
}

void CEcDialog::on_button_enabled() {
    auto fNewEnabled = m_EnableCheckButton.get_active();
    bool fOldEnabled = ((m_rTz.nPowerConfig & ILG_ECPC_F_ENABLED) != 0);
    if (fOldEnabled != fNewEnabled) {
        if (fNewEnabled)
            m_rTz.nPowerConfig |= ILG_ECPC_F_ENABLED;
        else
            m_rTz.nPowerConfig &= ~ILG_ECPC_F_ENABLED;
        UpdateCtrlState();
    }
}

void CEcDialog::on_button_schedule() {
    auto fNewEnabled = m_ScheduleCheckButton.get_active();
    bool fOldEnabled = ((m_rTz.nPowerConfig & ILG_ECPC_F_SCHEDULE) != 0);
    if (fOldEnabled != fNewEnabled) {
        if (fNewEnabled)
            m_rTz.nPowerConfig |= ILG_ECPC_F_SCHEDULE;
        else
            m_rTz.nPowerConfig &= ~ILG_ECPC_F_SCHEDULE;
        UpdateCtrlState();
    }
}

bool CEcDialog::on_schedule_label_link(const Glib::ustring& uri) {
    DoChangeSchedule();
    return true;
}

void CEcDialog::on_tz_dialog_hide() {
    if (m_refTimeZoneDialog->m_fAccept) {
        m_rTz = m_refTimeZoneDialog->m_rTz;
        UpdateScheduleLabel();
    }
    m_refTimeZoneDialog = nullptr;
}

void CEcDialog::on_button_read() {
    DoRead();
}

void CEcDialog::on_button_write() {
    DoWrite();
}

void CEcDialog::DoRead() {
    // Читает 7-ую временную зону с индексом 6 (считая от 0)
    m_oController.ReadTimeZones(0, 6, &m_rTz, 1);
    UpdateCtrlData(false);
}

void CEcDialog::DoWrite() {
    UpdateCtrlData(true);
    m_oController.WriteTimeZones(0, 6, &m_rTz, 1);
}

void CEcDialog::DoChangeSchedule() {
    if (nullptr == m_refTimeZoneDialog)
        m_refTimeZoneDialog.reset(new CTimeZoneDialog());
    m_refTimeZoneDialog->set_transient_for(*this);
    m_refTimeZoneDialog->set_modal();
    m_refTimeZoneDialog->set_hide_on_close();
    m_refTimeZoneDialog->signal_hide().connect(sigc::mem_fun(*this, &CEcDialog::on_tz_dialog_hide));

    m_refTimeZoneDialog->m_rTz = m_rTz;
    m_refTimeZoneDialog->SetTzIdx(false, false, 6);

    m_refTimeZoneDialog->set_visible(true);
    // TODO: Дописать
}

void CEcDialog::UpdateCtrlData(bool fSave) {
    if (fSave) {
        m_rTz.nPowerConfig = 0;
        if (m_EnableCheckButton.get_active())
            m_rTz.nPowerConfig |= ILG_ECPC_F_ENABLED;
        if (m_ExtReaderCheckButton.get_active())
            m_rTz.nPowerConfig |= ILG_ECPC_F_EXT_READER;
        if (m_CardOpenCheckButton.get_active())
            m_rTz.nPowerConfig |= ILG_ECPC_F_CARD_OPEN;
        if (m_ExitOffCheckButton.get_active())
            m_rTz.nPowerConfig |= ILG_ECPC_F_EXIT_OFF;
        if (m_InvertCheckButton.get_active())
            m_rTz.nPowerConfig |= ILG_ECPC_F_INVERT;
        if (m_ScheduleCheckButton.get_active())
            m_rTz.nPowerConfig |= ILG_ECPC_F_SCHEDULE;

        m_rTz.nEcDelaySec = static_cast<uint8_t>(m_DelaySpinButton.get_value_as_int());
    }
    else {
        m_EnableCheckButton.set_active((m_rTz.nPowerConfig & ILG_ECPC_F_ENABLED) != 0);
        m_ExtReaderCheckButton.set_active((m_rTz.nPowerConfig & ILG_ECPC_F_EXT_READER) != 0);
        m_CardOpenCheckButton.set_active((m_rTz.nPowerConfig & ILG_ECPC_F_CARD_OPEN) != 0);
        m_ExitOffCheckButton.set_active((m_rTz.nPowerConfig & ILG_ECPC_F_EXIT_OFF) != 0);
        m_InvertCheckButton.set_active((m_rTz.nPowerConfig & ILG_ECPC_F_INVERT) != 0);
        m_ScheduleCheckButton.set_active((m_rTz.nPowerConfig & ILG_ECPC_F_SCHEDULE) != 0);

        m_DelaySpinButton.set_value(m_rTz.nEcDelaySec);
        UpdateScheduleLabel();
        UpdateCtrlState();
    }
}

void CEcDialog::UpdateScheduleLabel() {
    const char32_t kDowTitles[] = U"ПВСЧПСВ";
    size_t nPos = 0;
    char32_t szDows[15] = {0};
    for (size_t k = 0; k < 7; k++) {
        szDows[nPos++] = (GET_BIT(m_rTz.nDaysOfWeek, k) ? kDowTitles[k] : '-');
        szDows[nPos++] = ' ';
    }
    std::wstring_convert<std::codecvt_utf8<char32_t>, char32_t> cv;
    std::stringstream ss;
    ss << "<a href=\"#1\">" << cv.to_bytes(szDows) << "  " << std::setfill('0') << std::setw(2)
       << (uint)m_rTz.nBeginHour << ':' << std::setw(2) << (uint)m_rTz.nBeginMinute << '-'
       << std::setw(2) << (uint)m_rTz.nEndHour << ':' << std::setw(2) << (uint)m_rTz.nEndMinute
       << "</a>";
    m_ScheduleLabel.set_text(ss.str());
    m_ScheduleLabel.set_use_markup();
}

void CEcDialog::UpdateCtrlState() {
    auto f = (m_rTz.nPowerConfig & ILG_ECPC_F_ENABLED) != 0;
    m_ExtReaderCheckButton.set_sensitive(f);
    m_CardOpenCheckButton.set_sensitive(f);
    m_ExitOffCheckButton.set_sensitive(f);
    m_DelaySpinButton.set_sensitive(f);
    m_InvertCheckButton.set_sensitive(f);
    m_ScheduleCheckButton.set_sensitive(f);
    f = f & (m_rTz.nPowerConfig & ILG_ECPC_F_SCHEDULE) != 0;
    m_ScheduleLabel.set_sensitive(f);
}
