#include "CEventsDialog.h"

#include <iomanip>  // для std::put_time
#include <sstream>  // для std::stringstream

#include "utils.h"

#define HAS_STYLE_PROVIDER_ADD_PROVIDER_FOR_DISPLAY GTKMM_CHECK_VERSION(4, 9, 1)

CEventsDialog::CEventsDialog() :
    m_fWiegand(false),
    m_fX2(false),
    m_nMaxEvents(0),
    m_nMaxKeys(0),
    m_nEventWriteIdx(-1),
    m_nEventReadIdx(-1),
    m_nMode(MODE_UNDEF),
    m_nReadBankIdx(0),
    m_nReadIdx(0),
    m_nReadCount(0),
    m_VBox(Gtk::Orientation::VERTICAL),
    m_ReadAllButton("Читать все"),
    m_ReadNewButton("Читать новые"),
    m_ReadIdxButton("Указатель чтения"),
    m_WriteIdxButton("Указатель Записи") {
    set_destroy_with_parent(true);
    set_title("События контроллера");
    set_default_size(900, 300);

    m_VBox.set_margin(5);
    set_child(m_VBox);

    m_VBox.append(m_ScrolledWindow);
    m_VBox.append(m_BottomBox);

    // Add the ColumnView, inside a ScrolledWindow, with the button underneath:
    m_ScrolledWindow.set_child(m_ColumnView);
    // Only show the scrollbars when they are necessary:
    m_ScrolledWindow.set_policy(Gtk::PolicyType::AUTOMATIC, Gtk::PolicyType::AUTOMATIC);
    m_ScrolledWindow.set_expand();

    // Create the List model:
    m_ListStore = Gio::ListStore<CModelColumns>::create();

    // Set list model and selection model.
    auto selection_model = Gtk::SingleSelection::create(m_ListStore);
    selection_model->set_autoselect(false);
    selection_model->set_can_unselect(true);
    m_ColumnView.set_model(selection_model);
    m_ColumnView.add_css_class("data-table");  // high density table

    // Add the ColumnView's columns:

    // Столбец "№"
    auto factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CEventsDialog::on_setup_label), Gtk::Align::END));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CEventsDialog::on_bind_idx));
    auto column = Gtk::ColumnViewColumn::create("№", factory);
    m_ColumnView.append_column(column);

    // Столбец "Дата и время"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CEventsDialog::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CEventsDialog::on_bind_time));
    column = Gtk::ColumnViewColumn::create("Дата и время", factory);
    m_ColumnView.append_column(column);

    // Столбец "Событие"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CEventsDialog::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CEventsDialog::on_bind_event));
    column = Gtk::ColumnViewColumn::create("Событие", factory);
    m_ColumnView.append_column(column);

    // Столбец "Направление"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CEventsDialog::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CEventsDialog::on_bind_direction));
    column = Gtk::ColumnViewColumn::create("Направление", factory);
    m_ColumnView.append_column(column);

    // Столбец "Ключ"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CEventsDialog::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CEventsDialog::on_bind_key));
    column = Gtk::ColumnViewColumn::create("Ключ", factory);
    m_ColumnView.append_column(column);

    // Столбец "Детали"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::bind(sigc::mem_fun(*this, &CEventsDialog::on_setup_label), Gtk::Align::START));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CEventsDialog::on_bind_details));
    column = Gtk::ColumnViewColumn::create("Детали", factory);
    m_ColumnView.append_column(column);

    m_BottomBox.set_margin(5);
    m_BottomBox.set_spacing(5);
    m_BottomBox.append(m_ReadAllButton);
    m_BottomBox.append(m_ReadNewButton);
    m_BottomBox.append(m_ReadIdxButton);
    m_BottomBox.append(m_WriteIdxButton);

    m_ReadAllButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CEventsDialog::on_button_read_all));
    m_ReadNewButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CEventsDialog::on_button_read_new));
    m_ReadIdxButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CEventsDialog::on_button_read_idx));
    m_WriteIdxButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CEventsDialog::on_button_write_idx));

    m_oDisp.connect(sigc::mem_fun(*this, &CEventsDialog::on_ilg));

    signal_hide().connect(sigc::mem_fun(*this, &CEventsDialog::on_hide));

    // Load extra CSS file.
    m_refCssProvider = Gtk::CssProvider::create();
#if HAS_STYLE_PROVIDER_ADD_PROVIDER_FOR_DISPLAY
    Gtk::StyleProvider::add_provider_for_display(get_display(), m_refCssProvider,
                                                 GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
#else
    Gtk::StyleContext::add_provider_for_display(get_display(), m_refCssProvider,
                                                GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
#endif
    std::string sCSS =
        ".read_idx {"
        "  color: green; "
        "  font-weight: bold;"
        "}"
        ".write_idx {"
        "  color: red;"
        "  font-weight: bold;"
        "}";

    m_refCssProvider->load_from_data(sCSS);
}

CEventsDialog::~CEventsDialog() {
    m_oController = nullptr;
}

void CEventsDialog::Init(const ilg::CController& oController) {
    m_oController = oController.Clone();

    ilg_controller_info rInfo;
    m_oController.GetControllerInfo(rInfo);
    m_fWiegand = (rInfo.nCtrFlags & ILG_CTR_F_WIEGAND) != 0;
    m_fX2 = (rInfo.nCtrFlags & ILG_CTR_F_X2) != 0;
    m_nMaxEvents = (rInfo.nBankSize / 8);
    m_nMaxKeys = (m_nMaxEvents - 24);
    if (m_fX2)
        m_nMaxKeys *= 2;
    m_oController.EnableMessageQueue();
    m_oController.SetMessageCallback(&on_ilg_message, this);
    UpdateCtrlState();
}

void CEventsDialog::on_setup_label(const Glib::RefPtr<Gtk::ListItem>& list_item,
                                   Gtk::Align halign) {
    list_item->set_child(*Gtk::make_managed<Gtk::Label>("", halign));
}

void CEventsDialog::on_bind_idx(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(Glib::ustring::sprintf("%u", col->m_nIdx));
    if (col->m_nIdx == m_nEventWriteIdx) {
        if (!label->has_css_class("write_idx"))
            label->add_css_class("write_idx");
    }
    else if (col->m_nIdx == m_nEventReadIdx) {
        if (!label->has_css_class("read_idx"))
            label->add_css_class("read_idx");
    }
}

void CEventsDialog::on_bind_time(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    std::stringstream ss;
    if (col->m_tTime != 0) {
        auto tmb = std::localtime(&col->m_tTime);
        ss << std::put_time(tmb, "%d-%m-%Y %H:%M:%S");
    }
    label->set_text(ss.str());
}

void CEventsDialog::on_bind_event(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(kEventTypeNames[col->m_nType]);
}

void CEventsDialog::on_bind_direction(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_text(kDirectionNames[col->m_nDirection]);
    label->set_halign(Gtk::Align::CENTER);
}

void CEventsDialog::on_bind_key(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    if (col->m_fKeyInitialized) {
        auto key = col->m_property_number.get_value();
        if (key.m_fEmpty)
            label->set_text("");
        else
            label->set_text(ilg::KeyNumberToStr(key.m_Number, key.m_nFlags, m_fWiegand));
    }
    else
        label->set_text("");
    Glib::Binding::bind_property_value(col->m_property_number.get_proxy(), label->property_label(),
                                       Glib::Binding::Flags::DEFAULT,
                                       sigc::mem_fun(*this, &CEventsDialog::number_transform_to));
}

bool CEventsDialog::number_transform_to(const GValue* a, GValue* b) {
    auto pSrc = (const MyKeyNumber*)a->data->v_pointer;
    if (b->g_type != G_TYPE_STRING)
        return false;
    if (pSrc->m_fEmpty)
        g_value_set_string(b, "");
    else
        g_value_set_string(b,
                           ilg::KeyNumberToStr(pSrc->m_Number, pSrc->m_nFlags, m_fWiegand).c_str());
    return true;
}

void CEventsDialog::on_bind_details(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<CModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    std::stringstream ss;
    switch (col->m_nFormat) {
    case ILG_EVENT_FORMAT_PASSAGE:
        if (col->m_nKeyIdx != -1)
            ss << "Позиция ключа " << std::dec << col->m_nKeyIdx << " (банк " << col->m_nKeyBankN
               << ")";
        break;
    case ILG_EVENT_FORMAT_CONTROLLER_MODE:
        ss << kControllerModeNames[col->m_nMode] << "; Флаги: " << std::hex << std::showbase
           << (uint)col->m_nModeFlags << ", триггер: " << std::dec << (uint)col->m_nModeTrigger;
        break;
    case ILG_EVENT_FORMAT_STATE:
        {
            bool fAddSemicolon = true;
            switch (col->m_nType) {
            case ILG_EVENT_ELECTROCONTROL:
                if (col->m_nModeFlags & 1)
                    ss << "Питание включено";
                else
                    ss << "Питание выключено";
                break;

            case ILG_EVENT_FIRE:
                if (col->m_nModeFlags & 1)
                    ss << "Пожарный режим включен";
                else
                    ss << "Пожарный режим выключен";
                break;

            case ILG_EVENT_SECURITY:
                if (col->m_nModeFlags & 1)
                    ss << "Охранный режим включен";
                else
                    ss << "Охранный режим выключен";
                ss << "; ";
                if (col->m_nModeFlags & 2)
                    ss << "Тревога включена";
                else
                    ss << "Тревога выключена";
                break;

            default:
                fAddSemicolon = false;
                break;
            }
            if (fAddSemicolon)
                ss << "; ";
            ss << "Флаги: " << std::hex << std::showbase << (uint)col->m_nModeFlags
               << ", триггер: " << std::dec << (uint)col->m_nModeTrigger;
            break;
        }
    }
    label->set_text(ss.str());
}

void CEventsDialog::on_button_read_all() {
    ilg_rtc_params rRtc;
    m_oController.ReadRtcParams(rRtc);
    m_nEventReadIdx = rRtc.nEventReadIdx;
    m_nEventWriteIdx = rRtc.nEventWriteIdx;
    StartReadEvents(0, m_nMaxEvents);
}

void CEventsDialog::on_button_read_new() {
    ilg_rtc_params rRtc;
    m_oController.ReadRtcParams(rRtc);
    m_nEventReadIdx = rRtc.nEventReadIdx;
    m_nEventWriteIdx = rRtc.nEventWriteIdx;
    if ((-1 == m_nEventReadIdx) || (-1 == m_nEventWriteIdx)) {
        ShowMessage("Некорректные указатели событий в контроллере", Gtk::MessageType::ERROR);
        return;
    }
    ssize_t nNewCount;
    if (m_nEventWriteIdx >= m_nEventReadIdx)
        nNewCount = (m_nEventWriteIdx - m_nEventReadIdx);
    else
        nNewCount = (m_nMaxEvents - m_nEventReadIdx + m_nEventWriteIdx);

    m_ListStore->remove_all();
    if (0 == nNewCount) {
        m_nReadCount = 0;
        return;
    }
    StartReadEvents(m_nEventReadIdx, nNewCount);
}

void CEventsDialog::on_button_read_idx() {
    auto pModel = dynamic_cast<Gtk::SingleSelection*>(m_ColumnView.get_model().get());
    auto nCount = pModel->get_n_items();
    if ((nCount != m_nMaxEvents) || (-1 == m_nEventReadIdx))
        return;
    pModel->select_item(m_nEventReadIdx, true);
}

void CEventsDialog::on_button_write_idx() {
    auto pModel = dynamic_cast<Gtk::SingleSelection*>(m_ColumnView.get_model().get());
    auto nCount = pModel->get_n_items();
    if ((nCount != m_nMaxEvents) || (-1 == m_nEventWriteIdx))
        return;
    pModel->select_item(m_nEventWriteIdx, true);
}

void CEventsDialog::on_progress_dialog_cancel() {
    m_oCommand.Cancel();
    m_oCommand = nullptr;
    m_refProgressDialog = nullptr;
}

bool CEventsDialog::on_progress_dialog_progress(size_t& nCurrent, size_t& nTotal) {
    switch (m_nMode) {
    case MODE_READ_EVENTS:
        m_oCommand.GetProgress(nCurrent, nTotal);
        break;

    case MODE_READ_KEYS:
        nCurrent = 0;
        nTotal = 0;
        break;
    }
    return true;
}

void CEventsDialog::on_progress_dialog_hide() {
    m_oCommand = nullptr;
    m_refProgressDialog = nullptr;
}

void CEventsDialog::on_dialog_response(int) {
    m_refDialog = nullptr;
}

void CEventsDialog::on_hide() {
    m_oController.Close();
}

void ILG_CALL CEventsDialog::on_ilg_message(ilg_controller_msg nMsg, const void* pMsgData,
                                            void* pUserData) {
    auto p = (CEventsDialog*)pUserData;
    p->m_oDisp.emit();
}

void CEventsDialog::on_ilg() {
    ilg_controller_msg nMsg;
    const void* pMsgData;
    while (m_oController.GetMessage(nMsg, pMsgData)) {
        if (ILG_CONTROLLER_MSG_COMMAND_FINISH == nMsg)
            on_ilg_command_finish((ilg_handle)pMsgData);
    }
}

void CEventsDialog::on_ilg_command_finish(ilg_handle hCommand) {
    if (hCommand != m_oCommand.Get())
        return;
    ilg::CAsyncCommand oCommand(std::move(m_oCommand));
    switch (m_nMode) {
    case MODE_READ_EVENTS:
        {
            m_nMode = MODE_UNDEF;
            m_refProgressDialog = nullptr;

            auto nStatus = oCommand.GetStatus();
            if (ILG_SUCCEEDED(nStatus)) {
                const uint64_t* pList;
                size_t nCount;
                m_oController.End_ReadEvents(oCommand, pList, nCount);

                if (m_nReadCount != m_nMaxEvents) {
                    m_oController.WriteEventReadIdx(m_nEventWriteIdx);
                    m_nEventReadIdx = m_nEventWriteIdx;
                }
                SetEvents(m_nReadIdx, pList, nCount);
            }
            else
                ShowMessage(ilg_get_error_text(nStatus), Gtk::MessageType::ERROR);
            break;
        }

    case MODE_READ_KEYS:
        {
            auto nStatus = oCommand.GetStatus();
            if (ILG_SUCCEEDED(nStatus)) {
                const ilg_key* pList;
                size_t nCount;
                m_oController.End_ReadKeys(oCommand, pList, nCount);
                auto fReadKeys = false;
                uint8_t nNextReadBankIdx = 0;
                size_t nNextReadIdx = 0;
                // Применяем прочитанные ключи
                auto c = m_ListStore->get_n_items();
                CModelColumns* pCol;
                MyKeyNumber rNumber;
                const ilg_key* pKey;
                for (size_t i = 0; i < c; i++) {
                    pCol = m_ListStore->get_item(i).get();
                    if (!pCol->m_fKeyInitialized && (ILG_EVENT_FORMAT_PASSAGE == pCol->m_nFormat)) {
                        if ((pCol->m_nKeyBankN == m_nReadBankIdx) &&
                            ((pCol->m_nKeyIdx >= m_nReadIdx) ||
                             (pCol->m_nKeyIdx < (m_nReadIdx + nCount)))) {
                            pKey = &pList[pCol->m_nKeyIdx - m_nReadIdx];
                            rNumber.m_Number = pKey->rNumber;
                            rNumber.m_nFlags = pKey->nFlags;
                            rNumber.m_fEmpty = pKey->fErased;
                            pCol->m_property_number = rNumber;
                            pCol->m_fKeyInitialized = true;
                        }
                        else if (!fReadKeys) {
                            nNextReadBankIdx = pCol->m_nKeyBankN;
                            nNextReadIdx = pCol->m_nKeyIdx;
                            fReadKeys = true;
                        }
                    }
                }
                if (fReadKeys) {
                    m_nReadBankIdx = nNextReadBankIdx;
                    m_nReadIdx = nNextReadIdx;
                    size_t nMaxRead = m_fX2 ? 12 : 6;
                    m_nReadCount = std::min(nMaxRead, static_cast<size_t>(m_nMaxKeys - m_nReadIdx));
                    m_oCommand =
                        m_oController.Begin_ReadKeys(m_nReadBankIdx, m_nReadIdx, m_nReadCount);
                }
                else {
                    m_nMode = MODE_UNDEF;
                    m_refProgressDialog = nullptr;
                }
            }
            else {
                m_nMode = MODE_UNDEF;
                m_refProgressDialog = nullptr;
                if (ILG_FAILED(nStatus))
                    ShowMessage(ilg_get_error_text(nStatus), Gtk::MessageType::ERROR);
            }
            break;
        }
    }
}

void CEventsDialog::StartReadEvents(size_t nIdx, size_t nCount) {
    if (m_oCommand != nullptr) {
        m_oCommand.Cancel();
        m_oCommand = nullptr;
        m_refProgressDialog = nullptr;
    }
    m_nMode = MODE_READ_EVENTS;
    m_nReadIdx = nIdx;
    m_nReadCount = nCount;
    m_oCommand = m_oController.Begin_ReadEvents(nIdx, nCount);
    m_refProgressDialog.reset(new CProgressDialog());
    m_refProgressDialog->set_transient_for(*this);
    m_refProgressDialog->set_modal();
    m_refProgressDialog->set_hide_on_close();
    m_refProgressDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CEventsDialog::on_progress_dialog_hide));
    m_refProgressDialog->signal_cancel().connect(
        sigc::mem_fun(*this, &CEventsDialog::on_progress_dialog_cancel));
    m_refProgressDialog->signal_progress().connect(
        sigc::mem_fun(*this, &CEventsDialog::on_progress_dialog_progress));
    m_refProgressDialog->SetStatusText("Чтение событий...");
    m_refProgressDialog->set_visible(true);
}

void CEventsDialog::SetEvents(size_t nStart, const uint64_t* pList, size_t nCount) {
    m_ListStore->remove_all();
    if (pList != nullptr) {
        time_t tEventTime;          // Дата и время события
        ilg_event_type nEventType;  // Тип события
        ilg_event_format nFormat;   // Формат записи события
        ilg_direction nDirection;   // Направление
        bool fKeyEmpty;             // true, номер ключа пустой (нет ключа)
        ilg_key_number rKeyNumber;  // Номер ключа
        uint8_t nKeyFlags;          // Флаги ключа
        uint8_t nModeFlags;         // Флаги режима
        uint8_t nModeTrigger;       // Триггер режима
        ilg_controller_mode nMode;  // Режим контроллера
        bool fKeyInitialized;  // true, ключ инициализирован (загружен из банка ключей)
        uint8_t nKeyBankN;  // Номер банка ключей
        ssize_t nKeyIdx;    // Позиция в банке ключей
        ilg_controller_time rTime;
        auto fReadKeys = false;
        auto nEventIdx = nStart;  // Позиция события в контроллере
        time_t tNow = time(nullptr);
        auto tmb = std::localtime(&tNow);

        auto pEnd = (pList + nCount);
        for (auto p = pList; p != pEnd; p++) {
            auto& nRawEvent = *p;
            nEventType = m_oController.DecodeEventType(nRawEvent, &nFormat);
            tEventTime = 0;
            nDirection = ILG_DIRECTION_UNKNOWN;
            memset(&rTime, 0, sizeof(rTime));
            fKeyInitialized = false;
            fKeyEmpty = true;

            switch (nFormat) {
            case ILG_EVENT_FORMAT_PASSAGE:
                m_oController.DecodePassageEvent(nRawEvent, rTime, nDirection, nKeyBankN, nKeyIdx);
                if (nKeyIdx != -1) {
                    fKeyEmpty = false;
                    fReadKeys = true;
                }
                else
                    fKeyInitialized = true;
                break;

            case ILG_EVENT_FORMAT_TIME:
                m_oController.DecodeTimeEvent(nRawEvent, rTime);
                break;

            case ILG_EVENT_FORMAT_CONTROLLER_MODE:
                m_oController.DecodeControllerModeEvent(nRawEvent, rTime, nMode, nModeFlags,
                                                        nModeTrigger);
                break;

            case ILG_EVENT_FORMAT_KEY_NUMBER:
                m_oController.DecodeKeyNumber(nRawEvent, rKeyNumber);
                fKeyEmpty = false;
                fKeyInitialized = true;
                break;
            }
            if (rTime.nMonth != 0) {
                struct tm atm;
                atm.tm_sec = rTime.nSecond;
                atm.tm_min = rTime.nMinute;
                atm.tm_hour = rTime.nHour;
                atm.tm_mday = rTime.nDay;
                atm.tm_mon = rTime.nMonth - 1;  // tm_mon is 0 based
                atm.tm_year = tmb->tm_year;     // tm_year is 1900 based
                atm.tm_isdst = -1;
                tEventTime = mktime(&atm);
                if (-1 == tEventTime)
                    tEventTime = 0;
                if (tEventTime != 0) {
                    // Если (дата события + 30 дней) > текущей даты,
                    if ((tEventTime + (60 * 60 * 24 * 30)) > tNow) {
                        // Вычитаем 1 год
                        tEventTime -=
                            60 * 60 * 24 * (365 + (IsLeapYear(1900 + atm.tm_year) ? 1 : 0));
                    }
                }
            }
            m_ListStore->append(CModelColumns::create(
                nEventIdx, tEventTime, nEventType, nFormat, nDirection, fKeyInitialized, fKeyEmpty,
                rKeyNumber, nKeyFlags, nModeFlags, nModeTrigger, nMode, nKeyBankN, nKeyIdx));

            if (++nEventIdx == m_nMaxEvents)
                nEventIdx = 0;
        }
        if (fReadKeys)
            StartReadKeys();
    }
    UpdateCtrlState();
}

void CEventsDialog::StartReadKeys() {
    if (m_oCommand != nullptr) {
        m_oCommand.Cancel();
        m_oCommand = nullptr;
        m_refProgressDialog = nullptr;
    }
    auto nCount = m_ListStore->get_n_items();
    CModelColumns* pCol;
    size_t nMaxRead = m_fX2 ? 12 : 6;

    for (size_t i = 0; i < nCount; i++) {
        pCol = m_ListStore->get_item(i).get();
        if ((ILG_EVENT_FORMAT_PASSAGE == pCol->m_nFormat) && !pCol->m_fKeyInitialized) {
            m_nMode = MODE_READ_KEYS;
            m_nReadBankIdx = pCol->m_nKeyBankN;
            m_nReadIdx = pCol->m_nKeyIdx;
            m_nReadCount = std::min(nMaxRead, static_cast<size_t>(m_nMaxKeys - m_nReadIdx));
            m_oCommand = m_oController.Begin_ReadKeys(m_nReadBankIdx, m_nReadIdx, m_nReadCount);
            m_refProgressDialog.reset(new CProgressDialog());
            m_refProgressDialog->set_transient_for(*this);
            m_refProgressDialog->set_modal();
            m_refProgressDialog->set_hide_on_close();
            m_refProgressDialog->signal_hide().connect(
                sigc::mem_fun(*this, &CEventsDialog::on_progress_dialog_hide));
            m_refProgressDialog->signal_cancel().connect(
                sigc::mem_fun(*this, &CEventsDialog::on_progress_dialog_cancel));
            m_refProgressDialog->signal_progress().connect(
                sigc::mem_fun(*this, &CEventsDialog::on_progress_dialog_progress));
            m_refProgressDialog->SetStatusText("Чтение ключей для событий...");
            m_refProgressDialog->set_visible(true);
            break;
        }
    }
}

void CEventsDialog::UpdateCtrlState() {
    auto f = (m_ListStore->get_n_items() == m_nMaxEvents);
    m_ReadIdxButton.set_sensitive(f);
    m_WriteIdxButton.set_sensitive(f);
}

void CEventsDialog::ShowMessage(const std::string& sMessage, Gtk::MessageType nType) {
    if (nullptr == m_refDialog)
        m_refDialog.reset(new Gtk::MessageDialog("Демо", false, nType));
    m_refDialog->set_transient_for(*this);
    m_refDialog->set_message(sMessage);
    m_refDialog->show();
    m_refDialog->signal_response().connect(
        sigc::bind(sigc::mem_fun(*this, &CEventsDialog::on_dialog_response)));
}
