/**
 * @file ilguard.h
 * @brief Заголовочный файл API SDK Guard.
 *
 * Данный файл содержит в себе определение API SDK Guard.
 * SDK Guard - библиотека для связи с сетевыми контроллерами Iron Logic через конвертеры.
 * Библиотека предназначена для ОС Linux.
 * [Сайт SDK](https://ironlogic.ru/il_new.nsf/htm/ru_sdk-guard)
 */
/**
 * @defgroup init Инициализация, версия и коды ошибок
 * @brief Функции и типы, связанные с инициализацией и обработкой ошибок.
 *
 * Ссылки на инфо о инициализации и завершении библиотеки, проверке версии и обработке ошибок.
 */
/**
 * @defgroup search Поиск конвертеров
 * @brief Функции и типы, связанные с поиском конвертеров.
 *
 * Ссылки на инфо о создании дескриптора поиска, поиске конвертеров, настройке поиска.
 */
/**
 * @defgroup converter Связь с конвертером
 * @brief Функции и типы, связанные с связью с конвертером.
 *
 * Ссылки на инфо о создании дескриптора конвертера, настройке параметров связи с конвертером,
 * подключение к конвертеру.
 */
/**
 * @defgroup ctr_search Поиск контроллеров
 * @brief Функции и типы, связанные с поиском контроллеров.
 *
 * Ссылки на инфо о создании дескриптора поиска, поиске контроллеров, настройке поиска.
 */
/**
 * @defgroup controller Связь с контроллером
 * @brief Функции и типы, связанные с связью с контроллером.
 *
 * Ссылки на инфо о создании дескриптора контроллера, настройке параметров связи с контроллером,
 * подключение к контроллеру.
 */
#ifndef ILGUARD_H_INCLUDED
#define ILGUARD_H_INCLUDED
#include <stdint.h>  // для uint8_t, uint32_t
#include <string.h>  // для memcpy, memset и memcmp

/// Переключатель между статическим и динамическим связыванием.
// #define ILGUARD_LINKONREQUEST

#ifndef ILGUARD_LINKONREQUEST
#ifdef ILG_STATIC_LIB
#define ILG_API
#elif defined(_WIN32) || defined(__CYGWIN__)
#ifdef ILGUARD_EXPORTS
#define ILG_API __declspec(dllexport)
#else
#define ILG_API __declspec(dllimport)
#endif
#else  // non Windows
#define ILG_API
#endif  // ILG_STATIC_LIB
#endif  // !ILGUARD_LINKONREQUEST

/// Макрос, определяющий соглашение о вызове функций.
#if defined(_WIN32) || defined(__CYGWIN__)
#define ILG_CALL WINAPI
#else
#define ILG_CALL
#endif /* _WIN32 || __CYGWIN__ */

/** @name Макросы версии SDK Guard.
 * @{ */
/**
 * @brief Основной номер версии библиотеки SDK Guard.
 *
 * Увеличивается, когда API становится не совместимым с предыдущими версиями.
 * @ingroup init
 */
#define ILG_VERSION_MAJOR 1
/**
 * @brief Минорный номер версии библиотеки SDK Guard.
 *
 * Увеличивается, когда в API добавляются новые возможности,
 * но при этом сохраняется обратная совместимость.
 * @ingroup init
 */
#define ILG_VERSION_MINOR 0
/**
 * @brief Номер патча библиотеки SDK Guard.
 *
 * Увеличивается, когда выпускается исправление ошибки,
 * не содержащее никаких изменений в API.
 * @ingroup init
 */
#define ILG_VERSION_PATCH 2
/**
 * @brief Полный номер версии библиотеки SDK Guard.
 * @ingroup init
 */
#define ILG_VERSION \
    ((ILG_VERSION_MAJOR << 24) | (ILG_VERSION_MINOR << 16) | (ILG_VERSION_PATCH << 8) | 0xC0)

/** Извлекает основной номер из полной версии. */
#define ILG_VERSION_GET_MAJOR(v) ((v) >> 24)
/** Извлекает минорный номер из полной версии. */
#define ILG_VERSION_GET_MINOR(v) (((v) >> 16) & 0xff)
/** Извлекает номер патча из полной версии. */
#define ILG_VERSION_GET_PATCH(v) (((v) >> 8) & 0xff)
/** @} */

/**
 * @brief Константа, указывающая на необходимость выделить память автоматически.
 *
 * Используется в функциях, которые возвращают строки и массивы.
 * Для освобождения памяти нужно использовать функцию @ref ilg_free_memory.
 */
#define ILG_AUTOALLOCATE 0xffffffff

/** Тип дескриптора (ссылка на объект библиотеки). */
typedef void* ilg_handle;

/** Флаг. */
typedef enum : int {
    ILG_FALSE,  ///< Ложь. =0
    ILG_TRUE  ///< Истина. Может быть любое число отличное от 0.
} ilg_bool;

/**
 * @brief Коды ошибок.
 * @ingroup init
 * @{
 */
typedef enum {
    /// Операция успешно выполнена.
    ILG_OK = 0,
    /// Неизвестная ошибка.
    ILG_E_FAIL,
    /// Операция выполняется асинхронно.
    ILG_E_PENDING,
    /// Индекс вне диапазона.
    ILG_E_BOUNDS,
    /// Команда не поддерживается конвертером.
    ILG_E_NOTIMPL,
    /// Неправильный указатель.
    ILG_E_POINTER,
    /// Операция прервана.
    ILG_E_ABORT,
    /// Неправильный дескриптор.
    ILG_E_HANDLE,
    /// Недостаточно памяти.
    ILG_E_OUTOFMEMORY,
    /// Неправильные параметры.
    ILG_E_INVALIDARG,
    /// Неправильная версия Sdk Guard передана в функцию @ref ilg_init.
    ILG_E_WRONG_SDK_VERSION,
    /// Библиотека не инициализирована с помощью @ref ilg_init.
    ILG_E_NOT_INITIALIZED,
    /// Размер буфера слишком мал.
    ILG_E_BUFFER_TOO_SMALL,
    /// Недостаточно ресурсов для завершения операции.
    ILG_E_OUT_OF_RESOURCES,
    /// Блокирующий вызов функции из обработчика не разрешен.
    ILG_E_BLOCKING_CALL_NOT_ALLOWED,
    /// Порт не существует.
    ILG_E_PORT_NOT_EXIST,
    /// Нет доступа к порту.
    ILG_E_PORT_ACCESS_DENIED,
    /// Неизвестная ошибка открытия порта.
    ILG_E_PORT_OPEN_FAILED,
    /// Ошибка подключения.
    ILG_E_CONNECTION_ERROR,
    /// Связь с устройством была потеряна.
    ILG_E_CONNECTION_DOWN,
    /// Тайм-аут запроса к конвертеру.
    ILG_E_REQUEST_TIMEOUT,
    /// Не распознан ответ конвертера.
    ILG_E_BAD_RESPONSE,
    /// Неизвестная ошибка конвертера.
    ILG_E_CONVERTER_ERROR,
    /// Контроллер не существует по указанному адресу.
    ILG_E_CTR_NOT_EXIST,
    /// Неизвестная ошибка лицензии.
    ILG_E_LICENSE_ERROR,
    /// Лицензия не найдена.
    ILG_E_LICENSE_NOT_FOUND,
    /// Лицензия просрочена.
    ILG_E_LICENSE_EXPIRED,
    /// Ограничение лицензии на количество контроллеров.
    ILG_E_LICENSE_CONTROLLERS,
    /// Ограничение лицензии на количество считываемых ключей.
    ILG_E_LICENSE_READ_KEYS,
    /// Ограничение лицензии на количество записываемых ключей.
    ILG_E_LICENSE_WRITE_KEYS,
    /// Команда не поддерживается.
    ILG_E_NOT_SUPPORTED,
    /// Размер файла не соответствует запросу.
    ILG_E_FIRMWARE_FILESIZE,
    /// Не удалось активировать режим прошивки.
    ILG_E_BOOTLOADER_NOSTART,
    /// Не подходит для данной модели устройства.
    ILG_E_NO_COMPATIBLE,
    /// Не подходит для данного номера устройства.
    ILG_E_INVALID_DEV_NUM,
    /// Слишком большой размер данных.
    ILG_E_TOOLARGE_FIRMWARE,
    /// Нарушена последовательность передачи данных.
    ILG_E_SEQUENCE_DATA,
    /// Нарушена целостность данных.
    ILG_E_DATA_INTEGRITY,
    /// Другая ошибка прошивки.
    ILG_E_FIRMWARE_OTHER,
    /// Не найдена запущенная программа контроллера. Попробуйте перезапустить устройство.
    ILG_E_RUN_FIRMWARE_FAIL,
    /// Неправильный образ прошивки.
    ILG_E_FIRMWARE_IMAGE_INVALID,
    /// Контроллер отказал в выполнении команды.
    ILG_E_CTR_NACK,
    /// Размер списка
    ILG_STATUS_SIZE
} ilg_status;

/**
 * @brief Проверяет код возврата SDK на успех.
 *
 * @param[in] status Код возврата.
 * @ingroup init
 */
#define ILG_SUCCEEDED(status) ((status) == ILG_OK)

/**
 * @brief Проверяет код возврата SDK на ошибку.
 *
 * @param[in] status Код возврата.
 * @ingroup init
 */
#define ILG_FAILED(status) ((status) != ILG_OK)

/** @} */

/** Тип порта конвертера. */
typedef enum {
    /** Не известно. */
    ILG_PORT_TYPE_UNKNOWN,
    /** Имя последовательного порта (например "/dev/ttyACM0"). */
    ILG_PORT_TYPE_COMPORT,
    /**
     * @brief Адрес конвертера в режиме "Сервер" (например "10.0.0.2:1000").
     *
     * Конвертер после соединения с сетью и получения IP адреса ожидает установки соединения.
     * Соединение устанавливает компьютер, расположенный в локальной или внешней сети. Этот режим
     * удобно использовать, когда известен IP адрес конвертера. Также преимуществом данного режима
     * является возможность подключения к конвертору с разных компьютеров, как находящихся в
     * локальной сети, так и по сети Internet. Соединение устанавливается по протоколу TCP.
     */
    ILG_PORT_TYPE_SERVER,
    /**
     * @brief Адрес конвертера в режиме "Клиент" (например "40200@25000").
     *
     * В режиме клиента, после получения IP адреса, конвертер пытается установить соединение с
     * локальным или удалённым компьютером – сервером. При невозможности установления соединения
     * попытка повторяется. В данном режиме работы нет необходимости знать IP адреса всех
     * конвертеров, входящих в систему. Все они будут соединяться с одним сервером самостоятельно.
     * При этом нет возможности перенести сервер на другой компьютер без переконфигурирования всех
     * конверторов. Соединение устанавливается по протоколу TCP.
     */
    ILG_PORT_TYPE_CLIENT,
    /**
     * @brief Адрес конвертера в режиме "Прокси" (например "40200:36D782FB@127.0.0.1:25001").
     *
     * Конвертер активно пытается соединиться с прокси-сервером. На этот же прокси-сервер обращается
     * компьютер. Поиск происходит по кодовому слову, заданному при конфигурировании конвертера.
     * Этот метод используется для связи, когда конвертер и компьютер работают в разных сетях и
     * установление прямого соединения невозможно. Соединение устанавливается по протоколу TCP.
     */
    ILG_PORT_TYPE_PROXY,
    /** Размер списка. */
    ILG_PORT_TYPE_SIZE
} ilg_port_type;

/**
 * @brief Модель конвертера.
 *
 * @sa @ref converters "Список моделей конвертеров"
 */
typedef enum {
    /** Неизвестная модель. */
    ILG_CONVERTER_MODEL_UNKNOWN,
    /** Z-397 (мод. USB)/Z-397 */
    ILG_CONVERTER_MODEL_Z397,
    /** Z-397 СтражЪ */
    ILG_CONVERTER_MODEL_STRAG,
    /** Z-397 СтражЪ 115 */
    ILG_CONVERTER_MODEL_STRAG115,
    /** Z-397 (мод. USB Guard)/Z-397 Guard */
    ILG_CONVERTER_MODEL_Z397_GUARD,
    /** Z-397 IP */
    ILG_CONVERTER_MODEL_Z397_IP,
    /** Z-397 (мод. Web) */
    ILG_CONVERTER_MODEL_Z397_WEB,
    /** Z-5R Web */
    ILG_CONVERTER_MODEL_Z5R_WEB,
    /** Matrix II Wi-Fi */
    ILG_CONVERTER_MODEL_MATRIX2_WIFI,
    /** Matrix VI Wi-Fi */
    ILG_CONVERTER_MODEL_MATRIX6_WIFI,
    /** Z-5R Web BT */
    ILG_CONVERTER_MODEL_Z5R_WEB_BT,
    /** Z-5R Wi-Fi */
    ILG_CONVERTER_MODEL_Z5R_WIFI,
    /** Matrix-II EH Web */
    ILG_CONVERTER_MODEL_MATRIX2_EH_WEB,
    /** Matrix-VI EH Web */
    ILG_CONVERTER_MODEL_MATRIX6_EH_WEB,
    /** Z-5R Web mini */
    ILG_CONVERTER_MODEL_Z5R_WEB_MINI,
    /** Matrix VI NFC WiFi */
    ILG_CONVERTER_MODEL_MATRIX6_NFC_WIFI,
    /** Размер списка. */
    ILG_CONVERTER_MODEL_SIZE
} ilg_converter_model;

/**
 * @brief Режим конвертера.
 */
typedef enum {
    /** Неизвестный режим. */
    ILG_CONVERTER_MODE_UNKNOWN,
    /**
     * @brief Режим "NORMAL".
     *
     * Режим простого конвертера RS-232 > RS-485. Этот режим поддерживают все конвертеры.
     */
    ILG_CONVERTER_MODE_NORMAL,
    /**
     * @brief Режим "ADVANCED".
     *
     * Режим для работы конвертера под управление специального ПО. В этом режиме конвертером
     * реализуется ряд функций повышающих надежность ПО и обеспечивается лицензионная защита.
     * Лицензионные ограничения касаются количества одновременно подключенных и обслуживаемых
     * конвертером контроллеров, а также количество карт в системе.
     * Этот режим поддерживает [Z-397 (мод. USB Guard)/Z-397 Guard](@ref z397_guard) и все IP
     * конвертеры/контроллеры.
     */
    ILG_CONVERTER_MODE_ADVANCED,
    /**
     * @brief Режим "TEST".
     *
     * Режим проверки сети контроллеров.
     * SDK не работает с этим режимом.
     */
    ILG_CONVERTER_MODE_TEST,
    /**
     * Режим "ACCEPT".
     *
     * Режим для быстрого запуска установленной сети без установки ПО на компьютере.
     * SDK не работает с этим режимом.
     */
    ILG_CONVERTER_MODE_ACCEPT,
    /** Размер списка. */
    ILG_CONVERTER_MODE_SIZE
} ilg_converter_mode;

/**
 * @brief Состояние переключателя LOCK у IP конвертера.
 */
typedef enum {
    /** Не известно. */
    ILG_LOCK_UNKNOWN,
    /** Переключатель LOCK включен. */
    ILG_LOCK_ON,
    /** Переключатель LOCK выключен." */
    ILG_LOCK_OFF,
    /** Размер списка. */
    ILG_LOCK_SIZE
} ilg_lock_state;

/**
 * @brief Модель контроллера.
 *
 * @sa @ref controllers "Список моделей контроллеров".
 */
typedef enum {
    /** Неизвестная модель. */
    ILG_CONTROLLER_MODEL_UNKNOWN,
    /** Gate 2000 */
    ILG_CONTROLLER_MODEL_GATE2000,
    /** Matrix-II (мод. E K Net)/Matrix-II Net */
    ILG_CONTROLLER_MODEL_MATRIX2_NET,
    /** Z-5R (мод. Net) */
    ILG_CONTROLLER_MODEL_Z5R_NET,
    /** Z-5R Net 8k */
    ILG_CONTROLLER_MODEL_Z5R_NET_8K,
    /** Z-5R (мод. Net 16000) */
    ILG_CONTROLLER_MODEL_Z5R_NET_16000,
    /** Matrix-III (мод. MF K Net) / Matrix-III Net */
    ILG_CONTROLLER_MODEL_MATRIX3_NET,
    /** Guard (мод. Net) */
    ILG_CONTROLLER_MODEL_GUARD_NET,
    /** Z-5R (мод. Web) */
    ILG_CONTROLLER_MODEL_Z5R_WEB,
    /** Matrix-II (мод. E K Wi-Fi)/Matrix-II Wi-Fi */
    ILG_CONTROLLER_MODEL_MATRIX2_WIFI,
    /** Z-Eurolock / Eurolock EHT net */
    ILG_CONTROLLER_MODEL_ZEUROLOCK,
    /** Z-9 EHT net v2 */
    ILG_CONTROLLER_MODEL_Z9_EHT_NET,
    /** [Matrix-VI (мод. NFC K Net)](@ref matrix6_nfc_k_net) */
    ILG_CONTROLLER_MODEL_MATRIX6_NFC_K_NET,
    /** Matrix-VI (мод. EH K Net) */
    ILG_CONTROLLER_MODEL_MATRIX6_EH_K_NET,
    /** Matrix VI Wi-Fi */
    ILG_CONTROLLER_MODEL_MATRIX6_WIFI,
    /** Z-5R Web BT */
    ILG_CONTROLLER_MODEL_Z5R_WEB_BT,
    /** Z-5R Wi-Fi */
    ILG_CONTROLLER_MODEL_Z5R_WIFI,
    /** Matrix-II EH Web */
    ILG_CONTROLLER_MODEL_MATRIX2_EH_WEB,
    /** Matrix-VI EH Web */
    ILG_CONTROLLER_MODEL_MATRIX6_EH_WEB,
    /** Z-5R Web mini */
    ILG_CONTROLLER_MODEL_Z5R_WEB_MINI,
    /** Matrix VI NFC WiFi */
    ILG_CONTROLLER_MODEL_MATRIX6_NFC_WIFI,
    /** CP Z-2 EH K Relay */
    ILG_CONTROLLER_MODEL_CP_Z2_EH_K_RELAY,
    /** CP Z-2 EH K */
    ILG_CONTROLLER_MODEL_CP_Z2_EH_K,
    /** Размер списка. */
    ILG_CONTROLLER_MODEL_SIZE
} ilg_controller_model;

/// Уровень лога.
typedef enum {
    /**
     * @brief Лог выключен.
     * @details Лог сообщения не показываются.
     */
    ILG_LOG_LEVEL_NONE,

    /**
     * @brief Ошибки.
     * @details Показывает любые ошибки, которые являются фатальными для операции, но не для службы
     * или приложения.
     */
    ILG_LOG_LEVEL_ERROR,

    /**
     * @brief Предупреждения.
     * @details Показывает возможные проблемы, которые не являются ошибками.
     */
    ILG_LOG_LEVEL_WARNING,

    /**
     * @brief Уведомления.
     * @details Показывает полезную информацию, в основном успехи.
     */
    ILG_LOG_LEVEL_INFO,

    /**
     * @brief Отладочные сообщения.
     * @details Показывает шаги программы, получаемые и отправляемые данные.
     */
    ILG_LOG_LEVEL_DEBUG,
    /** Размер списка. */
    ILG_LOG_LEVEL_SIZE
} ilg_log_level;

/**
 * @brief Тип функции обратного вызова для получения сообщений лога отладки библиотеки.
 *
 * Функция имеет следующий вид:
 * @code
 * void logging_callback(ilg_log_level level, const char *pContext, const char *pMessage, void
 * *pUserData)
 * @endcode
 *
 * @param[in] nLevel    Уровень лога.
 * @param[in] pContext  Контекст.
 * @param[in] pMessage  Текст сообщения.
 * @param[in] pUserData Указатель на данные пользователя.
 *
 * @ingroup init
 */
typedef void(ILG_CALL* ilg_logging_callback)(ilg_log_level nLevel, const char* pContext,
                                             const char* pMessage, void* pUserData);

/**
 * @brief Настройки библиотеки.
 *
 * @ingroup init
 */
typedef struct {
    /**
     * ILG_TRUE, посылать два стоповых бита, иначе один (для COM-порта). По умолчанию ILG_FALSE.
     */
    ilg_bool fTwoStopBits;
    /**
     * Тайм-аут запроса в миллисекундах. По умолчанию 3000.
     */
    int nRequestTimeout;
    /**
     * Количество попыток запроса. По умолчанию 2.
     */
    int nRequestAttempts;
} ilg_options;

/**
 * @brief Тип функции обратного вызова для фильтрации портов при поиске конвертеров.
 *
 * @param[in] nPortType   Тип порта.
 * @param[in] pszPortName Имя порта.
 * @param[in] pUserData   Указатель на данные пользователя.
 *
 * @return ILG_TRUE, порт нужно исключить.
 * @ingroup search
 */
typedef ilg_bool(ILG_CALL* ilg_filter_port_callback)(ilg_port_type nPortType,
                                                     const char* pszPortName, void* pUserData);

/**
 * @struct ilg_converter_info
 * @brief Информация о конвертере.
 *
 * Описывает основную информацию о конвертере.
 */
typedef struct {
    /** Тип порта. */
    ilg_port_type nPortType;
    /** Имя порта. */
    const char* pszPortName;
    /** Для IP конвертера: IP или имя сервера, к которому подключён конвертер. Если пустая строка,
     * то порт свободен. */
    const char* pszConnect;
    /** Модель конвертера. */
    ilg_converter_model nModel;
    /** Серийный номер конвертера. Если =-1, то не известно. */
    int nSn;
    /** Версия прошивки конвертера. Если =0, то не известно. Каждый байт представляет собой номер
     * версии, начиная от старшего байта.
     */
    uint32_t nFwVersion;
    /** Дата и время сборки прошивки. Если =0, то не известно. Эквивалентно типу __time64_t в C++ -
     * количество секунд с 00:00:00 1 января 1970 года. */
    int64_t nFwBuildDate;
    /** Режим работы конвертера. */
    ilg_converter_mode nMode;
    /** Ключ аутентификации. Если =-1, не известно. */
    int64_t nAuthKey;
    /** Состояние переключателя LOCK у IP конвертера. */
    ilg_lock_state nLock;
    /** Статус идентификации конвертера. */
    ilg_status nError;
} ilg_converter_info;

/**
 * Информация об ошибке прослушки TCP портов конвертеров в режиме "Клиент".
 */
typedef struct {
    /** Номер TCP порта. */
    uint16_t nTcpPort;
    /** Код ошибки. */
    ilg_status nError;
} ilg_listen_error_info;

/**
 * @brief Сообщение поиска конвертеров.
 * @ingroup search
 */
typedef enum {
    /**
     * Завершилась асинхронная команда, созданная функцией с префиксом "ilg_search_begin_".
     * Параметр - дескриптор команды (дескриптор действителен до следующего сообщения этого типа или
     * до следующего вызова GetMessage), его не нужно закрывать функцией @ref ilg_close_handle.
     */
    ILG_SEARCH_MSG_COMMAND_FINISH,
    /**
     * Конвертер найден, параметр @ref ilg_converter_info*.
     */
    ILG_SEARCH_MSG_CONVERTER_FOUND,
    /**
     * Конвертер потерян, параметр @ref ilg_converter_info*.
     */
    ILG_SEARCH_MSG_CONVERTER_LOST,
    /**
     * Список конвертеров изменён.
     * @remark Это сообщение посылается при добавлении/удалении конвертера в список найденных и при
     * изменении параметра инфо конвертера @ref ilg_converter_info.
     */
    ILG_SEARCH_MSG_LIST_CHANGED,
    /**
     * TCP порт для прослушки открыт, параметр uint16_t номер TCP порта.
     */
    ILG_SEARCH_MSG_LISTEN_PORT_OPEN,
    /**
     * TCP порт для прослушки закрыт, параметр uint16_t номер TCP порта.
     */
    ILG_SEARCH_MSG_LISTEN_PORT_CLOSED,
    /**
     * Ошибка TCP порта для прослушки, параметр @ref ilg_listen_error_info*.
     */
    ILG_SEARCH_MSG_LISTEN_ERROR
} ilg_search_msg;

/**
 * @brief Тип функции обратного вызова для получения уведомлений от дескриптора поиска.
 *
 * @param[in] nMsg      Тип сообщения.
 * @param[in] pMsgData  Указатель на данные сообщения. Тип данных зависит от типа сообщения.
 * @param[in] pUserData Указатель на данные пользователя.
 *
 * @warning Указатель @p pMsgData действителен до выхода из этой функции.
 *
 * @ingroup search
 */
typedef void(ILG_CALL* ilg_search_message_callback)(ilg_search_msg nMsg, const void* pMsgData,
                                                    void* pUserData);

/**
 * @brief Флаги типов конвертеров.
 * @ingroup search
 */
typedef enum : uint32_t {
    /** USB конвертеры Iron Logic. */
    ILG_CONVERTER_ILUSB = (1 << 0),
    /**
     * USB конвертеры сторонних производителей.
     * Успешная работа с такими конвертерами не гарантируется.
     */
    ILG_CONVERTER_TPUSB = (1 << 1),
    /** IP конвертеры в режиме "Сервер" (поиск по UDP). */
    ILG_CONVERTER_SERVER = (1 << 2),
    /** IP конвертеры в режиме "Клиент" (прослушка TCP). */
    ILG_CONVERTER_CLIENT = (1 << 3)
} ilg_converter_types;

/**
 * @brief Настройки поиска конвертеров.
 * @ingroup search
 */
typedef struct {
    /**
     * Типы конвертеров, которые нужно искать.
     * По умолчанию (ILG_CONVERTER_ILUSB | ILG_CONVERTER_SERVER | ILG_CONVERTER_CLIENT).
     */
    uint32_t nConverterTypes;
    /** Период между попытками открыть порт в миллисекундах. По умолчанию 1000. */
    int nOpenPortPeriod;
    /** Период между попытками опросить порт в миллисекундах. По умолчанию 60000. */
    int nPollPeriod;
    /** Период опроса IP конвертеров по UDP в миллисекундах. По умолчанию 5000. */
    int nUdpScanPeriod;
    /** Тайм-аут запроса по UDP в миллисекундах. По умолчанию 1500. */
    int nUdpRequestTimeout;
    /** Количество попыток запроса по UDP. По умолчанию 2. */
    int nUdpRequestAttempts;
    /** Период между попытками открыть TCP порт для прослушки конвертеров в режиме "Клиент" в
     * миллисекундах. По умолчанию 3000. */
    int nOpenListenerPeriod;
    /** Тайм-аут ожидания подключения конвертера в режиме "Клиент" после открытия порта прослушки.
     * Z-397 Web делает попытки примерно 1 раз в секунду. По умолчанию 1800. */
    int nAcceptTimeout;
} ilg_search_options;

/**
 * @brief Состояние подключения к конвертеру/контроллеру.
 * @ingroup converter
 */
typedef enum {
    ILG_CONNECTION_DISCONNECTED,  ////< Отключён.
    ILG_CONNECTION_CONNECTED,     ///< Подключён.
    ILG_CONNECTION_CONNECTING,    ///< Подключение.
    ILG_CONNECTION_SIZE           ///< Размер списка.
} ilg_connection_status;

/**
 * @brief Информация о лицензии конвертера.
 * @ingroup converter
 */
typedef struct {
    /** Номер лицензии. */
    uint8_t nLicenseN;
    /** Максимальное количество контроллеров. Если равно 0xFF, не ограничено. */
    uint8_t nControllers;
    /** Максимальное количество ключей. Если равно 0xFFFF, не ограничено. */
    uint16_t nKeys;
    /** Максимальная дата. Если равно 0xFFFF, не ограничено,
     * Используйте макросы ILG_EXPIREDATE_DAY, ILG_EXPIDEDATE_MONTH, ILG_EXPIDEDATE_YEAR для
     * декодирования даты.
     */
    uint16_t nExpireDate;
    /** Оставшееся время жизни лицензии в минутах (шаг - 2 мин). Если равно 0xFFFF, не ограничено.
     */
    uint16_t nCountdown;
} ilg_license_info;

/** @name Макросы для декодирования даты лицензии.
 * @{ */
/**
 * Декодирует день из даты лицензии.
 * @param[in] date         Дата ilg_license_info.nExpireDate.
 * @return День, в котором истекает срок лицензии.
 */
#define ILG_EXPIREDATE_DAY(date) ((date)&0x1F)
/**
 * Декодирует месяц из даты лицензии.
 * @param[in] date         Дата ilg_license_info.nExpireDate.
 * @return Месяц, в котором истекает срок лицензии.
 */
#define ILG_EXPIDEDATE_MONTH(date) (((date) >> 5) & 0xF)
/**
 * Декодирует год из даты лицензии.
 * @param[in] date         Дата ilg_license_info.nExpireDate.
 * @param[in] current_year Текущий год.
 * @return Год, в котором истекает срок лицензии.
 */
#define ILG_EXPIDEDATE_YEAR(date, current_year) (((current_year) / 100) * 100 + ((date) >> 9))

/** @} */

/// Номер лицензии конвертера для SDK Guard.
#define ILG_LICENSE_N 5

/**
 * @brief Сообщение конвертера.
 * @ingroup converter
 */
typedef enum {
    /**
     * Завершилась асинхронная команда, созданная функцией с префиксом "ilg_converter_begin_".
     * Параметр - дескриптор команды (дескриптор действителен до следующего сообщения этого типа или
     * до следующего вызова GetMessage), его не нужно закрывать функцией @ref ilg_close_handle.
     */
    ILG_CONVERTER_MSG_COMMAND_FINISH,
    /**
     * Изменилось состояние подключения к конвертеру.
     * Узнать текущее состояние можно функцией @ref ilg_converter_get_connection_status.
     */
    ILG_CONVERTER_MSG_CONNECTION_CHANGED,
    /**
     * Контроллер найден, параметр @ref ilg_controller_info*.
     */
    ILG_CONVERTER_MSG_CONTROLLER_FOUND,
    /**
     * Контроллер потерян, параметр @ref ilg_controller_info*.
     */
    ILG_CONVERTER_MSG_CONTROLLER_LOST,
    /**
     * Изменён список контроллеров.
     */
    ILG_CONVERTER_MSG_CONTROLLER_LIST_CHANGED,
    /**
     * Изменён адрес контроллера.
     */
    ILG_CONVERTER_MSG_CONTROLLER_ADDRESS_CHAMGED
} ilg_converter_msg;

/**
 * @brief Тип функции обратного вызова для получения уведомлений от дескриптора конвертера.
 *
 * @param[in] nMsg      Тип сообщения.
 * @param[in] pMsgData  Указатель на данные сообщения. Тип данных зависит от типа сообщения.
 * @param[in] pUserData Указатель на данные пользователя.
 *
 * @warning Указатель @p pMsgData действителен до выхода из этой функции.
 *
 * @ingroup converter
 */
typedef void(ILG_CALL* ilg_converter_message_callback)(ilg_converter_msg nMsg, const void* pMsgData,
                                                       void* pUserData);

/**
 * @brief Флаги типов контроллеров.
 * @ingroup controller
 */
typedef enum : uint32_t {
    /** Сетевой контроллер. */
    ILG_CTR_TYPE_F_NETCTR = (1 << 0),
    /** Электронный замок (Eurolock EHT net, Z-9 EHT Net). */
    ILG_CTR_TYPE_F_ELOCK = (1 << 1)
} ilg_ctr_type_flags;

/**
 * Способ сканирования контроллеров в режиме Normal.
 */
typedef enum {
    /** Автоматический. */
    ILG_CTR_SCAN_METHOD_AUTO,
    /** Сканирование диапазона. */
    ILG_CTR_SCAN_METHOD_RANGE,
    /** Последовательный опрос адресов. */
    ILG_CTR_SCAN_METHOD_POLL,
    /** Размер списка. */
    ILG_CTR_SCAN_METHOD_SIZE
} ilg_ctr_scan_method;

/**
 * @brief Настройки конвертера.
 * @ingroup converter
 */
typedef struct {
    /** Модель конвертера для подключения. По умолчанию ILG_CONVERTER_MODEL_UNKNOWN. */
    ilg_converter_model nConnectModel;

    /** Скорость конвертера. По умолчанию 57600. */
    uint32_t nSpeed;

    /** Устанавливает режим Normal для IP конвертера. По умолчанию ILG_FALSE. */
    ilg_bool fNormalMode;

    /** Период авто подключения к конвертеру в миллисекундах. По умолчанию -1.
     * @remark Это общий параметр для всех конвертеров.
     */
    int nReconnectPeriod;

    /** Период сканирования контроллеров в миллисекундах. По умолчанию 3000. */
    int nScanPeriod;

    /**
     * Типы сканируемых устройств в режиме Normal конвертера.
     * По умолчанию (ILG_CTR_TYPE_F_NETCTR | ILG_CTR_TYPE_F_ELOCK).
     */
    uint32_t nScanDeviceTypes;

    /**
     * Способ сканирования контроллеров в режиме Normal конвертера.
     * По умолчанию ILG_CTR_SCAN_METHOD_AUTO.
     */
    ilg_ctr_scan_method nScanMethod;

    /** Последний опрашиваемый адрес 1-254 в режиме Normal. По умолчанию 69. */
    uint32_t nLastPolledAddress;

    /** Тайм-аут опроса при последовательном сканировании в режиме Normal. По умолчанию 500. */
    int nPollTimeout;

    /** Автоматическое переназначение адресов 0 и 1 в режиме Normal. По умолчанию ILG_FALSE. */
    ilg_bool fReassignAddress01;
} ilg_converter_options;

/**
 * @brief Флаги контроллера.
 * @ingroup controller
 */
typedef enum : uint32_t {
    /** Два банка ключей, иначе - один. */
    ILG_CTR_F_TWOBANKS = (1 << 0),
    /** Режим считывателя Wiegand, иначе - iButton. */
    ILG_CTR_F_WIEGAND = (1 << 1),
    /** Режим объединения банков ключей Join (2 физ. банка -> 1 логический). */
    ILG_CTR_F_JOIN = (1 << 2),
    /** Режим удвоения ключей X2 (8 байт записи ключа -> 4). */
    ILG_CTR_F_X2 = (1 << 3),
    /** Поддержка режима управления электропитанием ElectroControl. */
    ILG_CTR_F_EC = (1 << 4),
    /** Поддержка режимов контроллера (blk режимы блокировки) */
    ILG_CTR_F_MODES = (1 << 5),
    /** Дополнительный набор временных зон. */
    ILG_CTR_F_DUALZONE = (1 << 6),
    /** Поддержка старой команды аварийного открывания дверей. */
    ILG_CTR_F_OLDEMERGOPEN = (1 << 7),
    /** Поддержка управления конфигурацией контроллера. */
    ILG_CTR_F_CONFIG = (1 << 8),
    /** Поддержка запроса состояния ExtAsk и команд пожарного и охранного режима. */
    ILG_CTR_F_EXTASK = (1 << 9),
    /** Поддержка антипассбэк. */
    ILG_CTR_F_APB = (1 << 10),
    /** Поддержка большого времени замка BigTime. */
    ILG_CTR_F_BIGTIME = (1 << 11),
    /** Поддержка режим AntiCovid. */
    ILG_CTR_F_ANTICOVID = (1 << 12),
    /** Поддержка двойных карт. */
    ILG_CTR_F_DUALCARD = (1 << 13)
} ilg_controller_flags;

/** Тип точки прохода контроллера. @ref pass_point_types */
typedef enum {
    /** Не известно. */
    ILG_PASS_POINT_UNKNOWN,
    /** [01] Дверь. */
    ILG_PASS_POINT_DOOR,
    /** [02] Турникет. */
    ILG_PASS_POINT_TURNSTILE,
    /** [03] Шлюз. */
    ILG_PASS_POINT_GATEWAY,
    /** [04] Шлагбаум. */
    ILG_PASS_POINT_BARRIER,
    /** [17] Электромагнитный замок. */
    ILG_PASS_POINT_EMAGNETIC,
    /** [18] Электромеханический замок. */
    ILG_PASS_POINT_EMECHANICAL,
    /** [19] Турникет. */
    ILG_PASS_POINT_TURNSTILE2,
    /** [20] Моторный через два реле. */
    ILG_PASS_POINT_MOTOR_TWO_RELAYS,
    /** [21] Моторный через одно реле. */
    ILG_PASS_POINT_MOTOR_ONE_RELAYS,
    /** [22] Электроконтроль. */
    ILG_PASS_POINT_ELECTROCONTROL,
    /** [23] Автоконтроль. */
    ILG_PASS_POINT_AUTOCONTROL,
    /** Размер списка. */
    ILG_PASS_POINT_SIZE
} ilg_pass_point_type;

/** Режим контроллера */
typedef enum {
    /** Неактивный. */
    ILG_CONTROLLER_MODE_INACTIVE,
    /** Обычный режим работы. */
    ILG_CONTROLLER_MODE_NORMAL,
    /**
     * @brief Блокировка.
     * @details Проходить могут только "блокирующие" карты.
     */
    ILG_CONTROLLER_MODE_BLOCK,
    /**
     * @brief Свободный.
     * @details Замок обесточен, при поднесении карты регистрируются.
     */
    ILG_CONTROLLER_MODE_FREE,
    /**
     * @brief Ожидание.
     * @details Обычный режим работы, при поднесении допустимой карты переход в режим "Free".
     */
    ILG_CONTROLLER_MODE_WAIT,
    /** Размер списка. */
    ILG_CONTROLLER_MODE_SIZE
} ilg_controller_mode;

/**
 * @brief Информация о контроллере.
 *
 * Информацию о контроллере можно получить функцией @ref ilg_converter_get_controller_info и
 * функцией @ref ilg_controller_get_info.
 */
typedef struct {
    /** Модель контроллера. */
    ilg_controller_model nModel;
    /** Код модели контроллера. */
    uint8_t nModelCode;
    /** Сетевой адрес контроллера. */
    uint8_t nAddress;
    /** Версия прошивки. */
    uint32_t nFwVersion;
    /** Серийный номер контроллера. */
    int nSn;
    /** Размер банка событий/ключей. */
    uint32_t nBankSize;
    /** Тип точки прохода. */
    ilg_pass_point_type nPassPoint;
    /** Флаги контроллера. @ref ilg_controller_flags */
    uint32_t nCtrFlags;
    /**
     * Маска инициализированных флагов контроллера. @ref ilg_controller_flags
     *
     * Позволяет определить какие биты действительны в @p nCtrFlags. Функция поиска контроллеров
     * @ref ilg_converter_scan определяет не все флаги контроллера, в отличие от функции подключения
     * контроллеру.
     * Если программа хранит флаги контроллера, то чтобы их обновить нужно сделать так:
     * @code
     * AppCtrFlags = (AppCtrFlags & ~nInitCtrFlags) | nCtrFlags;
     * @endcode
     */
    uint32_t nInitCtrFlags;
} ilg_controller_info;

/**
 * @brief Сообщение контроллера.
 * @ingroup controller
 */
typedef enum {
    /**
     * Завершилась асинхронная команда, созданная функцией с префиксом "ilg_controller_begin_".
     * Параметр - дескриптор команды (дескриптор действителен до следующего сообщения этого типа или
     * до следующего вызова GetMessage), его не нужно закрывать функцией @ref ilg_close_handle.
     */
    ILG_CONTROLLER_MSG_COMMAND_FINISH,
    /**
     * Изменилось состояние подключения к конвертеру.
     * Узнать текущее состояние можно функцией @ref ilg_controller_get_connection_status.
     */
    ILG_CONTROLLER_MSG_CONNECTION_CHANGED,
    /**
     * Изменён адрес контроллера.
     */
    ILG_CONTROLLER_MSG_ADDRESS_CHANGED,
    /**
     * Часы рассинхронизированы.
     */
    ILG_CONTROLLER_MSG_CLOCK_OUT_OF_SYNC,
    /**
     * Изменились параметры RTC.
     */
    ILG_CONTROLLER_MSG_RTC_CHANGED,
    /**
     * Изменились параметры Ext Ask.
     */
    ILG_CONTROLLER_MSG_EXTASK_CHANGED
} ilg_controller_msg;

/**
 * @brief Тип функции обратного вызова для получения уведомлений от дескриптора контроллера.
 *
 * @param[in] nMsg      Тип сообщения.
 * @param[in] pMsgData  Указатель на данные сообщения. Тип данных зависит от типа сообщения.
 * @param[in] pUserData Указатель на данные пользователя.
 *
 * @warning Указатель @p pMsgData действителен до выхода из этой функции.
 *
 * @ingroup controller
 */
typedef void(ILG_CALL* ilg_controller_message_callback)(ilg_controller_msg nMsg,
                                                        const void* pMsgData, void* pUserData);

/**
 * @brief Номер ключа.
 *
 * Номер ключа в контроллере занимает 3 или 6 байт памяти.
 */
typedef struct {
    union {
        uint64_t nData;      ///< Номер ключа в формате целого числа.
        uint8_t aDallas[6];  ///< Номер ключа в формате массива байт.

        struct {                   ///< Номер ключа в формате в Em-Marine
            uint16_t nNumber;      ///< Номер
            uint8_t nSeries;       ///< Серия
            uint8_t aFacility[3];  ///< Код производителя
        } em_marine;

        struct {
            uint8_t aDallas1[3];
            uint8_t aDallas2[3];
        } dual_dallas;

        struct {
            uint16_t nNumber1;  ///< Номер
            uint8_t nSeries1;   ///< Серия
            uint16_t nNumber2;  ///< Номер
            uint8_t nSeries2;   ///< Серия
        } dual_em_marine;
    };
} ilg_key_number;

/**
 * Временная зона контроллера.
 */
typedef struct {
    /** Дни недели. Бит 0 - понедельник, бит 6 - воскресенье. */
    uint8_t nDaysOfWeek;

    /** Начало: час. */
    uint8_t nBeginHour;

    /** Начало: минута. */
    uint8_t nBeginMinute;

    /** Конец: час. */
    uint8_t nEndHour;

    /** Конец: минута. */
    uint8_t nEndMinute;

    /**
     * Флаги временной зоны режима ElectroControl @ref ilg_ecpc_flags
     * Бит 0 (ILG_ECPC_F_ENABLED) Задействовать управление питанием.
     * Бит 1 (ILG_ECPC_F_SCHEDULE) Использовать временную зону 6 для включения питания.
     * Бит 2 (ILG_ECPC_F_EXT_READER) Контрольный считыватель: «0» Matrix-II Net, «1» внешний
     * считыватель. Бит 3 (ILG_ECPC_F_INVERT) Инвертировать управляющий выход. Бит 4
     * (ILG_ECPC_F_EXIT_OFF) Задействовать датчик двери. Бит 5 (ILG_ECPC_F_CARD_OPEN) Не блокировать
     * функцию открывания для контрольного считывателя.
     */
    uint8_t nPowerConfig;

    /**
     * @brief Задержка на выключение в секундах.
     *
     * Период времени в секундах, в течение которого должно оставаться включённым силовое реле после
     * снятия карты с контрольного считывателя.
     * */
    uint8_t nEcDelaySec;
} ilg_time_zone;

/**
 * Временная зона контроллера для переключения режима контроллера.
 */
typedef struct {
    /** Дни недели. Бит 0 - понедельник, бит 6 - воскресенье. */
    uint8_t nDaysOfWeek;
    /** Начало: час. */
    uint8_t nBeginHour;
    /** Начало: минута. */
    uint8_t nBeginMinute;
    /** Конец: час. */
    uint8_t nEndHour;
    /** Конец: минута. */
    uint8_t nEndMinute;
    /** Режим контроллера. */
    ilg_controller_mode nMode;
} ilg_mode_time_zone;

/** Тип ключа контроллера. */
typedef enum {
    /**
     * @brief Простой.
     *
     * Проходы по такому ключу осуществляются в обычном режиме.
     */
    ILG_KEY_NORMAL,
    /**
     * @brief Блокирующий.
     *
     * Такой ключ предназначен для проходов и перевода контроллера в режим «Блокировка». Срабатывает
     * такая карта не при прикладывании к считывателю, а при снятии со считывателя. После поднесения
     * блокирующей карты к считывателю с удержанием около 3-х секунд, контроллер переходит в режим
     * блокировки и пропускает только блокирующие карты. Проход по обычным картам запрещен. Выход из
     * режима блокировки осуществляется повторным поднесением блокирующей карты к считывателю с
     * удержанием около 3-х секунд или функциональной карты.
     */
    ILG_KEY_BLOCKING,
    /** Мастер. */
    ILG_KEY_MASTER,
    /** Размер списка. */
    ILG_KEY_TYPE_SIZE
} ilg_key_type;

/** Флаги ключа. */
typedef enum : uint32_t {
    /**
     * @brief Функциональная карта.
     *
     * Такая карта предназначена только для переключения режимов («блокировка» или «норма») при
     * поднесении к считывателю. Каждое поднесение карты к считывателю меняет режим контроллера.
     * Проход по такой карте невозможен.
     */
    ILG_KEY_F_FUNC = (1 << 1),
    /**
     * @brief Двойная карта.
     *
     * Номер состоит из двух ID по 3 байта. Первый ID должен быть уникальным, то есть первая карта
     * или код может использоваться только один раз в контроллере. Каждый ID может быть номером
     * карты или кодом.
     */
    ILG_KEY_F_DUAL = (1 << 2),
    /**
     * @brief Короткий номер.
     *
     * Указывает контроллеру, что номер ключа имеет размер 3 байта. При сравнении с таким номером
     * контроллер проверяет только 3 младших байта.
     */
    ILG_KEY_F_SHORT = (1 << 5)
} ilg_key_flags;

/** Ключ. */
typedef struct {
    /** True, ключ стёрт. */
    uint8_t fErased;
    /** Флаги ключа. @see @ref ilg_key_flags. */
    uint8_t nFlags;
    /**
     * Доступ.
     *
     * 0x01-0x7F, флаги временных зон. Каждый из разрядов 0-6 соответствует зоне, когда проход
     * разрешен.
     * 0, проход запрещен.
     * 0xFF, проход разрешен всегда, вне зависимости от расписания зон.
     */
    uint8_t nAccess;
    /** True, выделено для WriteKeys(Selected=True). */
    uint8_t fSelected;
    /** Тип ключа. */
    ilg_key_type nType;
    /** Номер ключа. */
    ilg_key_number rNumber;
} ilg_key;

/**
 * Тип события контроллера.
 * @ingroup controller
 * @{
 */
typedef enum {
    /** Неизвестное событие. */
    ILG_EVENT_UNKNOWN,
    /** Открыто кнопкой изнутри. */
    ILG_EVENT_BUTTON_OPEN,
    /** Ключ не найден в банке ключей. */
    ILG_EVENT_KEY_NOT_FOUND,
    /** Ключ найден, дверь открыта. */
    ILG_EVENT_KEY_OPEN,
    /** Ключ найден, доступ не разрешен. */
    ILG_EVENT_KEY_ACCESS,
    /** Открыто оператором по сети. */
    ILG_EVENT_REMOTE_OPEN,
    /** Ключ найден, дверь заблокирована. */
    ILG_EVENT_KEY_DOOR_BLOCK,
    /** Попытка открыть заблокированную дверь кнопкой. */
    ILG_EVENT_BUTTON_DOOR_BLOCK,
    /** Дверь взломана. */
    ILG_EVENT_NO_OPEN,
    /** Дверь оставлена открытой (тайм-аут). */
    ILG_EVENT_NO_CLOSE,
    /** Проход состоялся. */
    ILG_EVENT_PASSAGE,
    /** Перезагрузка контроллера. */
    ILG_EVENT_REBOOT,
    /** Заблокирована кнопка открывания. */
    ILG_EVENT_BUTTON_BLOCK,
    /** Попытка двойного прохода. */
    ILG_EVENT_DBL_PASSAGE,
    /** Дверь открыта. */
    ILG_EVENT_OPEN,
    /** Дверь закрыта. */
    ILG_EVENT_CLOSE,
    /** Изменение состояния питания: пропало питание. */
    ILG_EVENT_POWER_OFF,
    /** Изменение состояния питания: появилось. */
    ILG_EVENT_POWER_ON,
    /** Включение замка (триггер). */
    ILG_EVENT_LOCK_CONNECT,
    /** Отключение замка (триггер). */
    ILG_EVENT_LOCK_DISCONNECT,
    /** Изменение состояния электропитания. */
    ILG_EVENT_ELECTROCONTROL,
    /** Переключение режима контроллера. */
    ILG_EVENT_CONTROLLER_MODE,
    /** Изменение состояния Пожара. */
    ILG_EVENT_FIRE,
    /** Изменение состояния Охраны. */
    ILG_EVENT_SECURITY,
    /** Совершен вход в шлюз. */
    ILG_EVENT_GATEWAY_PASS,
    /** Заблокирован вход в шлюз (занят). */
    ILG_EVENT_GATEWAY_BLOCK,
    /** Разрешен вход в шлюз. */
    ILG_EVENT_GATEWAY_ALLOWED,
    /** Заблокирован проход (Антипассбек). */
    ILG_EVENT_ANTIPASSBACK,
    /** Hotel (Изменение режима работы). */
    ILG_EVENT_HOTEL40,
    /** Hotel (Отработка карт). */
    ILG_EVENT_HOTEL41,
    /** Номер ключа. */
    ILG_EVENT_KEY_NUMBER,
    /** Размер списка. */
    ILG_EVENT_TYPE_SIZE
} ilg_event_type;

/** Формат записи события контроллера. */
typedef enum {
    /** Неизвестный формат события. */
    ILG_EVENT_FORMAT_UNKNOWN,
    /** События прохода, декодировать функцией ilg_decode_passage_event. */
    ILG_EVENT_FORMAT_PASSAGE,
    /** Событие с датой и временем, декодировать функцией ilg_decode_time_event. */
    ILG_EVENT_FORMAT_TIME,
    /** Событие переключения режима, декодировать функцией ilg_decode_controller_mode_event. */
    ILG_EVENT_FORMAT_CONTROLLER_MODE,
    /** Событие изменения состояния, декодировать функцией ilg_decode_state_event. */
    ILG_EVENT_FORMAT_STATE,
    /** Событие "Номер ключа", декодировать функцией ilg_decode_key_number_event. */
    ILG_EVENT_FORMAT_KEY_NUMBER
} ilg_event_format;

/**
 * Направление прохода.
 * @ingroup controller
 */
typedef enum {
    /** Неизвестное направление. */
    ILG_DIRECTION_UNKNOWN,
    /** Вход. */
    ILG_DIRECTION_IN,
    /** Выход. */
    ILG_DIRECTION_OUT,
    /** Размер списка. */
    ILG_DIRECTION_SIZE
} ilg_direction;

/** @} */

/**
 * Флаги конфигурации электропитания (ElectroControl).
 * @ingroup controller
 */
typedef enum : uint32_t {
    /** Задействовать управление питанием. */
    ILG_ECPC_F_ENABLED = (1 << 0),
    /** Использовать временную зону 6 для включения питания. */
    ILG_ECPC_F_SCHEDULE = (1 << 1),
    /** Контрольный считыватель: «0» Matrix-II Net, «1» внешний считыватель. */
    ILG_ECPC_F_EXT_READER = (1 << 2),
    /** Инвертировать управляющий выход. */
    ILG_ECPC_F_INVERT = (1 << 3),
    /** Задействовать датчик двери. */
    ILG_ECPC_F_EXIT_OFF = (1 << 4),
    /** Не блокировать функцию открывания для контрольного считывателя. */
    ILG_ECPC_F_CARD_OPEN = (1 << 5)
} ilg_ecpc_flags;

/**
 * Параметры запроса RTC.
 * @ingroup controller
 */
typedef struct {
    /** Рассинхронизация часов в секундах. */
    int64_t nDiffSec;
    /** Указатель записи событий. Если =-1, то адрес указателя не корректный. */
    ssize_t nEventWriteIdx;
    /** Указатель чтения событий. Если =-1, то адрес указателя не корректный. */
    ssize_t nEventReadIdx;
    /** Номер последнего ключа, поднесённого к считывателю. */
    ilg_key_number rLastKey;
    /** True, аварийное открывание дверей (старая команда). */
    ilg_bool fEmergencyOpen;
} ilg_rtc_params;

/**
 * Время контроллера.
 * @ingroup controller
 */
typedef struct {
    uint16_t nYear;      ///< Год.
    uint8_t nMonth;      ///< Месяц.
    uint8_t nDayOfWeek;  ///< День недели.
    uint8_t nDay;        ///< День.
    uint8_t nHour;       ///< Час.
    uint8_t nMinute;     ///< Минута.
    uint8_t nSecond;     ///< Секунда.
} ilg_controller_time;

/**
 * Параметры запроса Ext Ask.
 * @ingroup controller
 */
typedef struct {
    /** Режим контроллера. */
    ilg_controller_mode nMode;
    /** Секунды часов контроллера. Для определения разлёта часов. */
    uint8_t nClockSec;
    /** Питание 12 В, умноженное на 10 (=120 (12.0V)). */
    uint8_t n12V;
    /** Температура, прибавлено 64. «0» – неизвестно (=88 (24С)). */
    uint8_t nT;
    /** Питание CR2032, умноженное на 50. «0» – неизвестно (=150 (3V)). */
    uint8_t nCR2032;
    /**
     * Флаги состояния.
     * Бит 0 - Состояние двери «0» - закрыта.
     * Бит 1 - Состояние кнопки «1» - нажата.
     * Бит 2 - Состоянии блокировок «1» - активна блокировка двери.
     * Бит 3 - Состояние охраны «1» - активен режим охраны.
     * Биты 6 - Замок обесточен из-за пожара.
     * Биты 7 - Замок обесточен старой командой аварийного открытия.
     */
    uint8_t nState;
} ilg_extask_params;

/**
 * @brief Настройки контроллера.
 * @ingroup controller
 */
typedef struct {
    /** Авто синхронизацию часов контроллера. По умолчанию ILG_FALSE. */
    ilg_bool fAutoSyncClock;
    /** Период проверки параметров RTC в миллисекундах. По умолчанию 3000. */
    int nCheckRtcPeriod;
    /** Период опроса Ext Ask в миллисекундах. По умолчанию -1. */
    int nExtAskPeriod;
} ilg_controller_options;

/**
 * Конфигурация Z5-R Web, Z-5R Web BT.
 * @ingroup controller
 * @{
 */
typedef struct {
    /** Контрольная сумма CRC-8. */
    uint8_t nCRC;
    /** Версия конфигурации. */
    uint8_t nVersion;

    union {
        /** Системные флаги. */
        uint16_t nFlags;

        struct {
            uint16_t ctCardMode : 1,  ///< 0 CardMode (для памяти).
                ctDual_zone : 1,      ///< 1 Использование двух зон.
                ctReaderMode : 1,     ///< 2 Режим считывателей (Dallas/Wiegand).
                ctAPB : 1,            ///< 3 Антипассбэк.
                blk_oxr : 1,          ///< 4 Блокировка включает охрану.
                flgUnused5 : 1,       ///< 5 Не занято.
                flgUnused6 : 1,       ///< 6 Не занято.
                ctAltWieg : 1,        ///< 7 Альтернативный Wiegand (на SND).
                ctMemSize : 3,        ///< 8 Размер конфигурации памяти.
                ctUniBank : 1,        ///< 11 Один банк карт.
                joinbank : 1;         ///< 12 Режим JOIN.
        } sys_flags;
    };

    /** Индикация. */
    uint16_t nGrn_mask;  ///< Зелёный.
    uint16_t nRed_mask;  ///< Красный.
    uint16_t nSnd_mask;  ///< Звук.
    /** Специальные. */
    uint16_t nAlarm_mask;  ///< Тревога.
    uint16_t nPower_mask;  ///< Электроконтроль.
    uint16_t nGate_msk;    ///< Шлюз занят.
    uint16_t nBell_mask;   ///< Звонок.
    /** Команды от считывателей */
    uint8_t nD0_dn;  ///< DATA0.
    uint8_t nD0_up;  ///< DATA0.
    uint8_t nD1_dn;  ///< DATA1.
    uint8_t nD1_up;  ///< DATA1.
    /** Команды входов. */
    uint8_t nGRN_dn;  ///< Вход Green.
    uint8_t nGRN_up;  ///< Вход Green.
    uint8_t nRED_dn;  ///< Вход Red.
    uint8_t nRED_up;  ///< Вход Red.
    uint8_t nSND_dn;  ///< Вход BEEP.
    uint8_t nSND_up;  ///< Вход BEEP.
    /** Команды LOCK. */
    uint8_t nLOCK0_dn;  ///< LOCK.
    uint8_t nLOCK0_up;  ///< LOCK.
    uint8_t nLOCK1_dn;  ///< LOCK 1k8.
    uint8_t nLOCK1_up;  ///< LOCK 1k8.
    uint8_t nLOCK2_dn;  ///< LOCK 3k6.
    uint8_t nLOCK2_up;  ///< LOCK 3k6.
    /** Команды BUTT. */
    uint8_t nBUTT0_dn;  ///< BUTT.
    uint8_t nBUTT0_up;  ///< BUTT.
    uint8_t nBUTT1_dn;  ///< BUTT 1k8.
    uint8_t nBUTT1_up;  ///< BUTT 1k8.
    uint8_t nBUTT2_dn;  ///< BUTT 3k6.
    uint8_t nBUTT2_up;  ///< BUTT 3k6.
    /** Команды DOOR. */
    uint8_t nDOOR0_dn;  ///< DOOR.
    uint8_t nDOOR0_up;  ///< DOOR.
    uint8_t nDOOR1_dn;  ///< DOOR 1k8.
    uint8_t nDOOR1_up;  ///< DOOR 1k8.
    uint8_t nDOOR2_dn;  ///< DOOR 3k6.
    uint8_t nDOOR2_up;  ///< DOOR 3k6.
    /** Команды EXT. */
    uint8_t nEXT0_dn;  ///< EXT.
    uint8_t nEXT0_up;  ///< EXT.
    uint8_t nEXT1_dn;  ///< EXT 1k8.
    uint8_t nEXT1_up;  ///< EXT 1k8.
    uint8_t nEXT2_dn;  ///< EXT 3k6.
    uint8_t nEXT2_up;  ///< EXT 3k6.

    /** Замок. */
    union {
        uint32_t nLock_flags;  ///< Флаги замка.

        struct {
            uint32_t activator : 1;     ///< Вместо free - passive.
            uint32_t trigmode : 1;      ///< Режим триггер.
            uint32_t duallock : 1;      ///< Режим "два выхода" (турникет).
            uint32_t no_wait_door : 1;  ///< Не ждать закрытия двери.
            uint32_t unblock_door : 1;  ///< Дверь не открыли.
            uint32_t nightlock : 1;     ///< Режим "ночной замок".
            uint32_t nightconf : 1;  ///< Есть подтверждение переключения ночного.
            uint32_t no_long_pwr : 1;  ///< Нельзя долго включать (э.механика).
        } lock_flags;
    };

    uint16_t aLock_mask[4];  ///< Маски.
    uint8_t aLock_cmd[16];   ///< Логика.
} ilg_z5rw_config;

/**
 * Конфигурация Matrix II Wi-Fi, Matrix-VI Wi-Fi, Z-5R WiFi, Matrix-VI EHK Net.
 * @ingroup controller
 */
typedef struct {
    /** Контрольная сумма CRC-8. */
    uint8_t nCRC;
    /** Версия конфигурации. */
    uint8_t nVersion;

    union {
        /** Системные флаги. */
        uint16_t nFlags;

        struct {
            uint16_t ctCardMode : 1,  // 0 CardMode (для памяти).
                ctDual_zone : 1,      // 1 Использование двух зон.
                ctReaderMode : 1,     // 2 Режим считывателей (Dallas/Wiegand).
                ctAPB : 1,            // 3 Антипассбэк.
                blk_oxr : 1,          // 4 Блокировка включает охрану.
                ctFireButt : 1,       // 5 Пожарная тревога по кнопке.
                flgUnused6 : 1,       // 6 Не занято.
                flgUnused7 : 1,       // 7 Не занято.

                ctMemSize : 3,  // 8 Размер конфигурации памяти.
                ctUniBank : 1,  // 11 Один банк карт.
                joinbank : 1;   // 12 Режим JOIN.
        } sys_flags;
    };

    /** Индикация. */
    uint16_t nGrn_mask;  ///< Зелёный.
    uint16_t nRed_mask;  ///< Красный.
    uint16_t nSnd_mask;  ///< Звук.
    /** Специальные. */
    uint16_t nAlarm_mask;  ///< Тревога.
    uint16_t nPower_mask;  ///< Электроконтроль.
    uint16_t nGate_msk;    ///< Шлюз занят.
    uint16_t nBell_mask;   ///< Звонок.
    /** Команды от считывателей */
    uint8_t nD0_dn;  ///< DATA0.
    uint8_t nD0_up;  ///< DATA0.
    uint8_t nD1_dn;  ///< DATA1.
    uint8_t nD1_up;  ///< DATA1.
    /** Команды ZP. */
    uint8_t nZP0_dn;  ///< ZP.
    uint8_t nZP0_up;  ///< ZP.
    uint8_t nZP1_dn;  ///< ZP
    uint8_t nZP1_up;  ///< ZP.
    uint8_t nZP2_dn;  ///< ZP.
    uint8_t nZP2_up;  ///< ZP.
    /** Команды LED. */
    uint8_t nLED0_dn;  ///< LED.
    uint8_t nLED0_up;  ///< LED.
    uint8_t nLED1_dn;  ///< LED.
    uint8_t nLED1_up;  ///< LED.
    uint8_t nLED2_dn;  ///< LED.
    uint8_t nLED2_up;  ///< LED.
    /** Команды DOOR. */
    uint8_t nDOOR0_dn;  ///< DOOR.
    uint8_t nDOOR0_up;  ///< DOOR.
    uint8_t nDOOR1_dn;  ///< DOOR 1k8.
    uint8_t nDOOR1_up;  ///< DOOR 1k8.
    uint8_t nDOOR2_dn;  ///< DOOR 3k6.
    uint8_t nDOOR2_up;  ///< DOOR 3k6.

    /** Замок. */
    union {
        uint32_t nLock_flags;  ///< Флаги замка.

        struct {
            uint32_t activator : 1;     ///< Вместо free - passive.
            uint32_t trigmode : 1;      ///< Режим триггер.
            uint32_t duallock : 1;      ///< Режим "два выхода" (турникет).
            uint32_t no_wait_door : 1;  ///< Не ждать закрытия двери.
            uint32_t unblock_door : 1;  ///< Дверь не открыли.
            uint32_t nightlock : 1;     ///< Режим "ночной замок".
            uint32_t nightconf : 1;  ///< Есть подтверждение переключения ночного.
            uint32_t no_long_pwr : 1;  ///< Нельзя долго включать (э.механика).
        } lock_flags;
    };

    uint16_t aLock_mask[4];  ///< Маски.
    uint8_t aLock_cmd[16];   ///< Логика.
} ilg_m2wf_config;

/**
 * Конфигурация Z5-R Net, Z-5R Net 16k.
 * @ingroup controller
 */
typedef struct {
    /** Контрольная сумма CRC-8. */
    uint8_t nCRC;
    /** Версия конфигурации. */
    uint8_t nVersion;

    union {
        /** Системные флаги. */
        uint16_t nFlags;

        struct {
            uint16_t ctCardMode : 1,  // 0 CardMode (для памяти).
                ctDual_zone : 1,      // 1 Использование двух зон.
                led_act : 1,          // 2 led принимать как аналог.
                zp_act : 1,           // 3 zp принимать как аналог.
                blk_oxr : 1,          // 4 Блокировка включает охрану.
                ctReaderMode : 1,     // 5 Режим считывателей (Dallas/Wiegand).
                flgUnused6 : 1,       // 6 Не занято.
                flgUnused7 : 1;       // 7 Не занято.
        } sys_flags;
    };

    /** Индикация. */
    uint8_t nGrn_mask;  ///< Зелёный.
    uint8_t nRed_mask;  ///< Красный.
    uint8_t nSnd_mask;  ///< Звук.
    /** Специальные. */
    uint8_t nAlarm_mask;  ///< Тревога.
    uint8_t nPower_mask;  ///< Электроконтроль.
    uint8_t nGate_msk;    ///< Шлюз занят.
    uint8_t nBell_mask;   ///< Звонок.
    /** Команды от считывателей */
    uint8_t nD0_dn;  ///< DATA0.
    uint8_t nD0_up;  ///< DATA0.
    uint8_t nD1_dn;  ///< DATA1.
    uint8_t nD1_up;  ///< DATA1.
    /** Команды ZP. */
    uint8_t nZP0_dn;  ///< ZP.
    uint8_t nZP0_up;  ///< ZP.
    uint8_t nZP1_dn;  ///< ZP.
    uint8_t nZP1_up;  ///< ZP.
    uint8_t nZP2_dn;  ///< ZP.
    uint8_t nZP2_up;  ///< ZP.
    /** Команды LED. */
    uint8_t nLED0_dn;  ///< LED.
    uint8_t nLED0_up;  ///< LED.
    uint8_t nLED1_dn;  ///< LED.
    uint8_t nLED1_up;  ///< LED.
    uint8_t nLED2_dn;  ///< LED.
    uint8_t nLED2_up;  ///< LED.
    /** Команды DOOR. */
    uint8_t nDOOR0_dn;  ///< DOOR.
    uint8_t nDOOR0_up;  ///< DOOR.
    uint8_t nDOOR1_dn;  ///< DOOR 1k8.
    uint8_t nDOOR1_up;  ///< DOOR 1k8.
    uint8_t nDOOR2_dn;  ///< DOOR 3k6.
    uint8_t nDOOR2_up;  ///< DOOR 3k6.

    /** Замок. */
    union {
        uint8_t nLock_flags;  ///< Флаги замка.

        struct {
            uint8_t activator : 1;     ///< Вместо free - passive.
            uint8_t trigmode : 1;      ///< Режим триггер.
            uint8_t duallock : 1;      ///< Режим "два выхода" (турникет).
            uint8_t no_wait_door : 1;  ///< Не ждать закрытия двери.
            uint8_t unblock_door : 1;  ///< Дверь не открыли.
            uint8_t nightlock : 1;     ///< Режим "ночной замок".
            uint8_t nightconf : 1;  ///< Есть подтверждение переключения ночного.
            uint8_t no_long_pwr : 1;  ///< Нельзя долго включать (э.механика).
        } lock_flags;
    };

    uint8_t aLock_mask[4];  ///< Маски.
    uint8_t aLock_cmd[16];  ///< Логика.
} ilg_z5rnet_config;

/** @} */

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @brief Возвращает номер версии библиотеки.
 *
 * @remark Эта функция не требует инициализации библиотеки функцией @ref ilg_init.
 *
 * @ingroup init
 */
ILG_API uint32_t ILG_CALL ilg_get_version();

/**
 * @brief Возвращает описание ошибки по её коду.
 *
 * @remark Эта функция не требует инициализации библиотеки функцией @ref ilg_init.
 *
 * @param[in] nErrorCode Код ошибки.
 *
 * @ingroup init
 */
ILG_API const char* ILG_CALL ilg_get_error_text(ilg_status nErrorCode);

/**
 * @brief Устанавливает уровень лога отладки.
 *
 * @remark Эта функция не требует инициализации библиотеки функцией @ref ilg_init.
 *
 * @param[in] nLevel Уровень лога отладка.
 *
 * @return Всегда код успеха `ILG_OK`.
 *
 * @ingroup init
 */
ILG_API ilg_status ILG_CALL ilg_set_log_level(ilg_log_level nLevel);

/**
 * @brief Устанавливает функцию обратного вызова для лога отладки.
 *
 * @remark Эта функция не требует инициализации библиотеки функцией @ref ilg_init.
 *
 * @param[in] pCallback Указатель на функцию, которую библиотека будет вызывать для передачи
 * сообщений лога отладки.
 * @param[in] pUserData Указатель на данные пользователя, который будет передаваться в функцию.
 *
 * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в try catch.
 * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
 * из которого вызвана эта callback-функция, иначе вернёт ошибку `ILG_E_BLOCKING_CALL_NOT_ALLOWED`.
 *
 * @return Всегда код успеха `ILG_OK`.
 *
 * @ingroup init
 */
ILG_API ilg_status ILG_CALL ilg_set_log_callback(ilg_logging_callback pCallback,
                                                 void* pUserData = nullptr);

/**
 * @brief Освобождает память, которую выделила библиотека.
 *
 * Память могут выделять функции: @ref ilg_search_get_listen_ports.
 *
 * @remark Эта функция не требует инициализации библиотеки функцией @ref ilg_init.
 *
 * @param[in] p Указатель на память.
 *
 * @return Всегда код успеха `ILG_OK`.
 */
ILG_API ilg_status ILG_CALL ilg_free_memory(void* p);

/**
 * @brief Проверяет версию SDK.
 *
 * Возвращает `true` если версия библиотеки совместима с этим заголовочным файлом.
 * @ingroup init
 */
inline bool ilg_check_version() {
    auto v = ilg_get_version();
    return ((((v >> 24) & 0xff) == ILG_VERSION_MAJOR) &&
            ((int)((v >> 16) & 0xff) >= ILG_VERSION_MINOR));
}

/**
 * @brief Инициализирует библиотеку SDK Guard.
 *
 * Эта функция инициализирует библиотеку SDK Guard. Функция ilg_init должна быть вызвана первой
 * перед вызовом других функций, кроме @ref ilg_get_version, @ref ilg_get_error_text,
 * @ref ilg_set_log_level, @ref ilg_set_log_callback и @ref ilg_free_memory. Завершив использование
 * библиотеки приложение должно вызвать @ref ilg_cleanup, чтобы библиотека освободила внутренние
 * ресурсы. Приложение должно вызвать @ref ilg_cleanup при каждом успешном вызове ilg_init, т.к.
 * используется внутренний счётчик инициализаций.
 *
 * @remark Эта функция сама проверяет версию SDK, поэтому вызывать @ref ilg_check_version не нужно.
 *
 * @param[in] nVersionRequested Номер версии API SDK Guard, всегда должно быть равен
 * @ref ILG_VERSION.
 *
 * @return `ILG_OK` в случае успеха, иначе код ошибки.
 * @retval `ILG_E_WRONG_SDK_VERSION` запрошенная версия SDK не поддерживается.
 *
 * @sa @ref ilg_cleanup
 * @ingroup init
 */
ILG_API ilg_status ILG_CALL ilg_init(uint32_t nVersionRequested = ILG_VERSION);

/**
 * @brief Освобождает ресурсы библиотеки.
 *
 * Освобождает память и закрывает все дескрипторы библиотеки.
 *
 * @warning Эта функция не потокобезопасна. Её нельзя вызывать одновременно с функциями, которым
 * требуется инициализация библиотеки. Её нельзя вызывать из функции обратного вызова.
 *
 * @return Всегда код успеха `ILG_OK`.
 *
 * @sa @ref ilg_init
 * @ingroup init
 */
ILG_API ilg_status ILG_CALL ilg_cleanup();

/**
 * @brief Закрывает дескриптор библиотеки.
 *
 * @param[in] h Дескриптор библиотеки.
 */
ILG_API ilg_status ILG_CALL ilg_close_handle(ilg_handle h);

/**
 * @brief Клонирует дескриптор поиска конвертеров, дескриптор конвертера или дескриптор контроллера.
 *
 * @param[in]  h           Исходный дескриптор.
 * @param[out] pNewHandle  Новый дескриптор.
 */
ILG_API ilg_status ILG_CALL ilg_clone_handle(ilg_handle h, ilg_handle* pNewHandle);

/**
 * @brief Устанавливает функцию обратного вызова для фильтрации портов при поиске конвертеров.
 *
 * @param[in] pCallback Указатель на функцию, которую поток поиска будет вызывать при нахождении
 * порта.
 * @param[in] pUserData Указатель на данные пользователя, который будет передаваться в функцию.
 *
 * @warning Не позволяйте исключениям выйти из callback-функции, оберните тело функции в try catch.
 * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
 * из которого вызвана эта callback-функция, иначе вернёт ошибку `ILG_E_BLOCKING_CALL_NOT_ALLOWED`.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_set_filter_port_callback(ilg_filter_port_callback pCallback,
                                                         void* pUserData = nullptr);

/**
 * @brief Устанавливает глобальные настройки библиотеки.
 *
 * @param[in] pOptions Опции библиотеки.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup init
 */
ILG_API ilg_status ILG_CALL ilg_set_options(const ilg_options* pOptions);

/**
 * @brief Возвращает глобальные настройки библиотеки.
 *
 * @param[out] pOptions Опции библиотеки.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup init
 */
ILG_API ilg_status ILG_CALL ilg_get_options(ilg_options* pOptions);

/**
 * @brief Создаёт дескриптор поиска конвертеров.
 *
 * @param[out] pSearch Дескриптор поиска конвертеров.
 *
 * @remark Эта функция создаёт поток поиска конвертеров, если ещё не создан (один поток на
 * библиотеку). Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
 * конвертеров.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_get_search(ilg_handle* pSearch);

/**
 * @brief Создаёт дескриптор конвертера.
 *
 * @param[in]  nPortType   Тип порта.
 * @param[in]  pszPortName Имя порта.
 * @param[out] pConverter  Дескриптор конвертера.
 *
 * @remark Эта функция создаёт поток конвертера, если ещё не создан (один поток на порт).
 * Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 * Функция позволяет создать более одного дескриптора для одного порта.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_get_converter(ilg_port_type nPortType, const char* pszPortName,
                                              ilg_handle* pConverter);

/**
 * @brief Устанавливает функцию обратного вызова для уведомлений поиска конвертеров.
 *
 * Устанавливает функцию для получения сообщений от дескриптора поиска конвертеров.
 *
 * @param[in] hSearch    Дескриптор поиска конвертеров.
 * @param[in] pCallback  Указатель на функцию, которую библиотека будет вызывать при возникновении
 * события поиска конвертеров.
 * @param[in] pUserData  Указатель на данные пользователя, который будет передаваться в функцию.
 *
 * @warning
 * - Не позволяйте исключениям выйти из callback-функции, оберните тело функции в try catch.
 * - Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
 * из которого вызвана эта callback-функция, иначе будет ошибка `ILG_E_BLOCKING_CALL_NOT_ALLOWED`.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_search_set_message_callback(ilg_handle hSearch,
                                                            ilg_search_message_callback pCallback,
                                                            void* pUserData = nullptr);

/**
 * @brief Включает/выключает очередь сообщений.
 *
 * Эта функция устанавливает/снимает флаг "очередь включена" в дескрипторе поиска.
 * Очередь сообщений предназначена для синхронизации обработки сообщений поиска.
 *
 * @remark Алгоритм синхронизации: при возникновении события в очередь добавляется сообщение и
 * вызывается функция обратного вызова, установленная функцией @ref ilg_search_set_message_callback,
 * из которой посылается сигнал потоку, обрабатывающему сообщения, этот поток при получении сигнала
 * циклично вызывает @ref ilg_search_get_message, чтобы получить и обработать все сообщения.
 *
 * @param[in] hSearch  Дескриптор поиска конвертеров.
 * @param[in] fEnable  `ILG_TRUE`, включить очередь, иначе - выключить.
 *
 * @warning Если не извлекать сообщения из очереди функцией @ref ilg_search_get_message, то
 * она будет расти пока не закончится память.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_search_enable_message_queue(ilg_handle hSearch,
                                                            ilg_bool fEnable = ILG_TRUE);

/**
 * @brief Извлекает следующее сообщение из очереди.
 *
 * @param[in]  hSearch  Дескриптор поиска конвертеров.
 * @param[out] pMsg     Тип сообщения.
 * @param[out] pMsgData Указатель на данные пользователя, которые были установлены функцией
 * @ref ilg_search_set_message_callback.
 * @param[out] pFound   `ILG_TRUE`, если сообщение успешно извлечено, иначе - очередь пуста.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_search_get_message(ilg_handle hSearch, ilg_search_msg* pMsg,
                                                   const void** pMsgData, ilg_bool* pFound);

/**
 * @brief Устанавливает параметры поиска конвертеров.
 *
 * @remark Параметры поиска конвертеров общие для всех дескрипторов.
 * Параметры не теряются при закрытии всех дескрипторов поиска.
 *
 * @param[in] hSearch   Дескриптор поиска конвертеров.
 * @param[in] pOptions  Опции поиска конвертеров.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_search_set_options(ilg_handle hSearch,
                                                   const ilg_search_options* pOptions);

/**
 * @brief Возвращает параметры поиска конвертеров.
 *
 * @param[in]  hSearch   Дескриптор поиска конвертеров.
 * @param[out] pOptions  Опции поиска конвертеров.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_search_get_options(ilg_handle hSearch,
                                                   ilg_search_options* pOptions);

/**
 * @brief Устанавливает список портов для прослушки конвертеров к режиме "Клиент".
 *
 * Устанавливает список TCP-портов, к которым будут подключаться IP-конвертеры
 * в режиме "Клиент". Получить список можно функцией @ref ilg_search_get_listen_ports.
 *
 * @remark Список портов общий для всех дескрипторов.
 *
 * @param[in] hSearch Дескриптор поиска конвертеров.
 * @param[in] pPorts  Массив портов. Если равно `nullptr`, то очищает список.
 * @param[in] nCount  Количество элементов массива портов.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_search_set_listen_ports(ilg_handle hSearch, const uint16_t* pPorts,
                                                        size_t nCount);

/**
 * @brief Возвращает список портов для прослушки конвертеров к режиме "Клиент".
 *
 * Возвращает список TCP-портов, которые открываются для прослушки, к этим портам подключаются
 * IP-конвертеры в режиме "Клиент". Установить список можно с помощью функциеи @ref
 * ilg_search_set_listen_ports.
 *
 * @param[in]     hSearch Дескриптор поиска конвертеров.
 * @param[in out] pBuf    Указатель на буфер для списка портов. Может быть равен `nullptr`.
 * @param[in out] pSize   Размер буфера в словах `uint16_t`.
 *
 * @remarks
 * - Список портов общий для всех дескрипторов.
 * - Если @p pBuf равен `nullptr`, то буфер не используется, возвращается только требуемый размер
 * для буфера @p pBuf.
 * - При завершении функции в @p pSize устанавливается требуемый размер для буфера @p pBuf.
 * - Если @p pSize равно @ref ILG_AUTOALLOCATE, то библиотека сама выделяет память. @p pBuf должен
 * ссылаться на указатель, в который будет записан адрес выделенной памяти. После использования
 * памяти нужно освободить функцией @ref ilg_free_memory
 * @code
 *  uint16_t* pBuf = nullptr;
 *  size_t nBufSize = ILG_AUTOALLOCATE;
 *  if (ilg_search_get_listen_ports(hSearch, (uint16_t*)&pBuf, &nBufSize) == ILG_OK) {
 *      uint16_t* p = pBuf;
 *      for (size_t i = 0; i < nBufSize; i++, p++)
 *          printf("%u\n", *p);
 *      ilg_free_memory(pBuf);
 *  }
 * @endcode
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_POINTER Необходимый выходной параметр равен `nullptr`.
 * @retval ILG_E_BUFFER_TOO_SMALL Размер буфера не достаточный.
 * @retval ILG_E_OUTOFMEMORY Не удалось выделить память.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_search_get_listen_ports(ilg_handle hSearch, uint16_t* pBuf,
                                                        size_t* pSize);

/**
 * @brief Возвращает состояние Tcp-порта, открытого для прослушки конвертеров в режиме "Клиент".
 *
 * @param[in]  hSearch   Дескриптор поиска конвертеров.
 * @param[in]  nTcpPort  Номер TCP порта.
 * @param[out] pStatus   Состояние порта. `ILG_OK` порт открыт успешно, `ILG_E_PENDING` порт в
 * процессе открытия или его нет в списке портов для прослушки, иначе не удалось открыть.
 *
 * @remark Чтобы открыть порт для прослушки нужно добавить его в список портов с помощью функции
 * @ref ilg_search_set_listen_ports, и включить поиск конвертеров в режиме "Клиент" с помощью
 * @ref ilg_search_set_options (установить `pOptions->nConverterTypes |= ILG_CONVERTER_CLIENT`).
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_POINTER Необходимый выходной параметр равен `nullptr`.
 */
ILG_API ilg_status ILG_CALL ilg_search_get_listen_status(ilg_handle hSearch, uint16_t nTcpPort,
                                                         ilg_status* pStatus);

/**
 * @brief Ищет конвертеры.
 *
 * @param[in] hSearch  Дескриптор поиска конвертеров.
 * @param[in] fReset   `ILG_TRUE`, очищает список найденных конвертеров перед началом поиска.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
 * конвертеров.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_search_scan(ilg_handle hSearch, ilg_bool fReset = ILG_FALSE);

/**
 * @brief Запускает асинхронную команду поиска конвертеров.
 *
 * @param[in]  hSearch   Дескриптор поиска конвертеров.
 * @param[in]  fReset    `ILG_TRUE`, очистить список найденных перед поиском.
 * @param[out] pCommand  Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_search_begin_scan(ilg_handle hSearch, ilg_bool fReset,
                                                  ilg_handle* pCommand);

/**
 * @brief Возвращает количество найденных конвертеров.
 *
 * @param[in]  hSearch Дескриптор поиска конвертеров.
 * @param[out] pCount  Количество найденных конвертеров.
 *
 * @warning Эта функция копирует список из потока поиска, поэтому её лучше не вызывать в цикле, а
 * сохранить значение @p *pCount в переменной, и уже её использовать в цикле.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_search_get_converter_count(ilg_handle hSearch, size_t* pCount);

/**
 * @brief Возвращает информацию о найденном конвертере.
 *
 * @param[in]  hSearch Дескриптор поиска конвертеров.
 * @param[in]  nIdx    Позиция в списке найденных конвертеров.
 * @param[out] pInfo   Информация о конвертере.
 *
 * @warning В @p pInfo ссылки `pszPortName` и `pszConnect` действительны до следующего вызова @ref
 * ilg_search_get_converter_count и пока жив дескриптор @p hSearch.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_search_get_converter_info(ilg_handle hSearch, size_t nIdx,
                                                          ilg_converter_info* pInfo);

/**
 * @brief Включает/выключает авто поиск конвертеров.
 *
 * @param[in] hSearch  Дескриптор поиска конвертеров.
 * @param[in] fEnable  `ILG_TRUE`, включает авто поиск, иначе - выключает.
 * @param[in] fWait    `ILG_TRUE`, ждёт полного завершения команды, иначе только устанавливает
 * флаг.
 *
 * @remark Если @p fWait =`ILG_TRUE`, то функция не возвращает управление пока ждёт выполнение
 * команды в потоке поиска конвертеров.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_search_set_auto_scan(ilg_handle hSearch,
                                                     ilg_bool fEnable = ILG_TRUE,
                                                     ilg_bool fWait = ILG_TRUE);

/**
 * @brief Запускает асинхронную команду вкл/выкл режим авто поиска конвертеров.
 *
 * @param[in]  hSearch  Дескриптор поиска конвертеров.
 * @param[in]  fEnable  `ILG_TRUE`, включает поиск в реальном времени, иначе - выключает.
 * @param[out] pCommand Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_search_begin_set_auto_scan(ilg_handle hSearch, ilg_bool fEnable,
                                                           ilg_handle* pCommand);

/**
 * @brief Возвращает флаг авто поиска конвертеров.
 *
 * @param[in]  hSearch   Дескриптор поиска конвертеров.
 * @param[out] pEnabled  `ILG_TRUE`, авто поиск включен, иначе - выключен.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup search
 */
ILG_API ilg_status ILG_CALL ilg_search_get_auto_scan(ilg_handle hSearch, ilg_bool* pEnabled);

/**
 * @brief Открывает порт и возвращает его дескриптор.
 *
 * @param[in]  hSearch     Дескриптор поиска конвертеров.
 * @param[in]  nPortType   Тип порта.
 * @param[in]  pszPortName Имя порта.
 * @param[out] pInfo       Информация о конвертере.
 * @param[out] pPort       Системный дескриптор порта.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
 * конвертеров.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILG_API ilg_status ILG_CALL ilg_search_open_port(ilg_handle hSearch, ilg_port_type nPortType,
                                                 const char* pszPortName, ilg_converter_info* pInfo,
                                                 int* pPort);

/**
 * @brief Запускает асинхронную команду открытия порта.
 *
 * @param[in]  hSearch     Дескриптор поиска конвертеров.
 * @param[in]  nPortType   Тип порта.
 * @param[in]  pszPortName Имя порта.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILG_API ilg_status ILG_CALL ilg_search_begin_open_port(ilg_handle hSearch, ilg_port_type nPortType,
                                                       const char* pszPortName,
                                                       ilg_handle* pCommand);

/**
 * @brief Возвращает результат открытия порта.
 *
 * @param[in]  hCommand Дескриптор команды.
 * @param[out] pPortFD  Системный дескриптор порта.
 * @param[out] pInfo    Информация о конвертере (если известно).
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILG_API ilg_status ILG_CALL ilg_search_end_open_port(ilg_handle hCommand, int* pPortFD,
                                                     ilg_converter_info* pInfo);

/**
 * @brief Закрывает порт.
 *
 * @param[in]  hSearch     Дескриптор поиска конвертеров.
 * @param[in]  nPortType   Тип порта.
 * @param[in]  pszPortName Имя порта.
 * @param[in]  hPort       Системный дескриптор порта.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
 * конвертеров.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILG_API ilg_status ILG_CALL ilg_search_close_port(ilg_handle hSearch, ilg_port_type nPortType,
                                                  const char* pszPortName, int hPort);

/**
 * @brief Запускает асинхронную команду закрытия порта.
 *
 * @param[in]  hSearch     Дескриптор поиска конвертеров.
 * @param[in]  nPortType   Тип порта.
 * @param[in]  pszPortName Имя порта.
 * @param[in]  hPortFD     Системный дескриптор порта.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILG_API ilg_status ILG_CALL ilg_search_begin_close_port(ilg_handle hSearch, ilg_port_type nPortType,
                                                        const char* pszPortName, int hPortFD,
                                                        ilg_handle* pCommand);

/**
 * @brief Устанавливает функцию обратного вызова для уведомлений конвертера.
 *
 * Устанавливает функцию для получения сообщений от дескриптора конвертера.
 *
 * @param[in] hConverter Дескриптор конвертера.
 * @param[in] pCallback  Указатель на функцию, которую библиотека будет вызывать при возникновении
 * события конвертера.
 * @param[in] pUserData  Указатель на данные пользователя, который будет передаваться в функцию.
 *
 * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в `try catch`.
 * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
 * из которого вызвана эта callback-функция, иначе вернёт ошибку `ILG_E_BLOCKING_CALL_NOT_ALLOWED`.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_set_message_callback(
    ilg_handle hConverter, ilg_converter_message_callback pCallback, void* pUserData = nullptr);

/**
 * @brief Включает/выключает очередь сообщений.
 *
 * Эта функция устанавливает/снимает флаг в дескрипторе конвертера.
 * Очередь сообщений предназначена для синхронизации обработки сообщений.
 *
 * @remark Алгоритм синхронизации: при возникновении события в очередь добавляется сообщение и
 * вызывается функция обратного вызова, установленная функцией @ref
 * ilg_converter_set_message_callback, из которой посылается сигнал потоку, обрабатывающему
 * сообщения, этот поток при получении сигнала циклично вызывает @ref ilg_converter_get_message,
 * чтобы получить и обработать все сообщения.
 *
 * @param[in] hConverter  Дескриптор конвертера.
 * @param[in] fEnable     `ILG_TRUE`, включает очередь, иначе - выключает.
 *
 * @warning Если не извлекать сообщения из очереди функцией @ref ilg_converter_get_message, то
 * она будет расти пока не закончится память.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_enable_message_queue(ilg_handle hConverter,
                                                               ilg_bool fEnable = ILG_TRUE);

/**
 * @brief Извлекает следующее сообщение из очереди.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[out] pMsg        Тип сообщения.
 * @param[out] pMsgData    Указатель на данные пользователя, которые были установлены функцией
 * @ref ilg_converter_set_message_callback.
 * @param[out] pFound      `ILG_TRUE`, если сообщение успешно извлечено, иначе - очередь пуста.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_get_message(ilg_handle hConverter,
                                                      ilg_converter_msg* pMsg,
                                                      const void** pMsgData, ilg_bool* pFound);

/**
 * @brief Устанавливает параметры конвертера.
 *
 * @param[in]  hConverter Дескриптор конвертера.
 * @param[in]  pOptions   Параметры конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_set_options(ilg_handle hConverter,
                                                      const ilg_converter_options* pOptions);

/**
 * @brief Возвращает параметры конвертера.
 *
 * @param[in] hConverter  Дескриптор конвертера.
 * @param[in] pOptions    Параметры конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_get_options(ilg_handle hConverter,
                                                      ilg_converter_options* pOptions);

/**
 * @brief Подключается к конвертеру.
 *
 * @param[in] hConverter  Дескриптор конвертера.
 * @param[in] fReconnect  `ILG_TRUE`, Отключается перед подключением.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера. Если
 * подключиться не удалось и период авто подключения (nReconnectPeriod в @ref ilg_converter_options)
 * не равен -1, то продолжает периодически пытаться подключиться.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_connect(ilg_handle hConverter,
                                                  ilg_bool fReconnect = ILG_FALSE);

/**
 * @brief Запускает асинхронную команду подключения к конвертеру.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[in]  fReconnect  `ILG_TRUE`, переподключиться.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_begin_connect(ilg_handle hConverter, ilg_bool fReconnect,
                                                        ilg_handle* pCommand);

/**
 * @brief Отключается от конвертера.
 *
 * @param[in] hConverter  Дескриптор конвертера.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 * Запрещает авто подключение к конвертеру.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_disconnect(ilg_handle hConverter);

/**
 * @brief Запускает асинхронную команду отключения от конвертера.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_begin_disconnect(ilg_handle hConverter,
                                                           ilg_handle* pCommand);

/**
 * @brief Возвращает состояние подключения к конвертеру.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[out] pStatus     Состояние подключения к конвертеру.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_get_connection_status(ilg_handle hConverter,
                                                                ilg_connection_status* pStatus);

/**
 * @brief Возвращает информацию о конвертере.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[out] pInfo       Информация о конвертере.
 *
 * @warning В @p pInfo ссылки `pszPortName` и `pszConnect` действительны пока жив дескриптор @p
 * hConverter.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_get_info(ilg_handle hConverter,
                                                   ilg_converter_info* pInfo);

/**
 * @brief Возвращает информацию о лицензии конвертера.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[out] pInfo       Информация о лицензии.
 * @param[in]  nLicenseN   Номер лицензии, информацию о которой нужно получить. По умолчанию =5.
 * @param[in]  fForce      `ILG_TRUE`, прочитать инфо из конвертера, иначе взять из кэша (только
 * лицензия №5)
 *
 * @remark Если @p fForce равен ILG_TRUE или @p nLicenseN не равен `ILG_LICENSE_N`, то функция не
 * возвращает управление пока ждёт выполнение команды в потоке конвертера.
 * Работает только в режиме `Advanced` конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_read_license_info(ilg_handle hConverter,
                                                            ilg_license_info* pInfo,
                                                            uint32_t nLicenseN = ILG_LICENSE_N,
                                                            ilg_bool fForce = ILG_FALSE);

/**
 * @brief Запускает асинхронную команду запроса инфо о лицензии конвертера.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[in]  nLicenseN   Номер лицензии, информацию о которой нужно получить.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @remark Работает только в режиме `Advanced` конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_begin_read_license_info(ilg_handle hConverter,
                                                                  uint32_t nLicenseN,
                                                                  ilg_handle* pCommand);

/**
 * @brief Возвращает результат запроса инфо о лицензии конвертера.
 *
 * @param[in]  hCommand  Дескриптор команды, которую вернула функция @ref
 * ilg_converter_begin_read_license_info.
 * @param[out] pInfo     Информация о лицензии конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_end_read_license_info(ilg_handle hCommand,
                                                                ilg_license_info* pInfo);

/**
 * @brief Возвращает количество установленных лицензий.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[out] pCount      Количество установленных лицензий.
 *
 * @warning Эта функция читает из контроллера список лицензий и сохраняет его в памяти дескриптора,
 * поэтому её лучше не вызывать в цикле, а сохранить значение @p *pCount в переменной, и уже её
 * использовать в цикле.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 * Работает только в режиме `Advanced` конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_get_license_count(ilg_handle hConverter, size_t* pCount);

/**
 * @brief Запускает асинхронную команду получения количества установленных лицензий.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @remark Эта функция читает из контроллера список лицензий и сохраняет его в памяти дескриптора.
 * Работает только в режиме `Advanced` конвертера.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_begin_get_license_count(ilg_handle hConverter,
                                                                  ilg_handle* pCommand);

/**
 * @brief Возвращает количество установленных лицензий.
 *
 * @param[in]  hCommand  Дескриптор команды, которую вернула функция @ref
 * ilg_converter_begin_get_license_count.
 * @param[out] pCount    Количество установленных лицензий.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_end_get_license_count(ilg_handle hCommand,
                                                                size_t* pCount);

/**
 * @brief Возвращает инфо о лицензии из списка, полученного методом @ref
 * ilg_converter_get_license_count.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[in]  nIdx        Позиция в списке.
 * @param[out] pInfo       Информация о лицензии.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_get_license_info(ilg_handle hConverter, size_t nIdx,
                                                           ilg_license_info* pInfo);

/**
 * @brief Устанавливает лицензию в конвертер.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[in]  pData       Данные лицензии.
 * @param[in]  nSize       Количество байт данных лицензии.
 * @param[out] pInfo       Информация о лицензии. Может быть равен `nullptr`.
 * @param[in]  nLicenseN   Номер лицензии.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 * Работает только в режиме `Advanced` конвертера.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_write_license(ilg_handle hConverter, const void* pData,
                                                        size_t nSize,
                                                        ilg_license_info* pInfo = nullptr,
                                                        uint32_t nLicenseN = ILG_LICENSE_N);

/**
 * @brief Запускает асинхронную команду установки лицензии в конвертер.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[in]  pData       Данные лицензии.
 * @param[in]  nSize       Количество байт данных лицензии.
 * @param[in]  nLicenseN   Номер лицензии.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @remark Работает только в режиме `Advanced` конвертера.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_begin_write_license(ilg_handle hConverter,
                                                              const void* pData, size_t nSize,
                                                              uint32_t nLicenseN,
                                                              ilg_handle* pCommand);

/**
 * @brief Возвращает результат установки лицензии в конвертер.
 *
 * @param[in]  hCommand  Дескриптор команды, которую вернула функция @ref
 * ilg_converter_begin_write_license.
 * @param[out] pInfo       Информация о лицензии.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_end_write_license(ilg_handle hCommand,
                                                            ilg_license_info* pInfo);

/**
 * @brief Удаляет все лицензии.
 *
 * @param[in] hConverter   Дескриптор конвертера.
 * @param[in] fDisconnect  `ILG_TRUE`, после установки лицензии отключается от конвертера , иначе -
 * устанавливает лицензию #5 по умолчанию.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 * Работает только в режиме `Advanced` конвертера.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_clear_licenses(ilg_handle hConverter,
                                                         ilg_bool fDisconnect = ILG_FALSE);

/**
 * @brief Запускает асинхронную команду удаления всех лицензий.
 *
 * @param[in]  hConverter   Дескриптор конвертера.
 * @param[in]  fDisconnect  `ILG_TRUE`, после установки лицензии отключается от конвертера , иначе -
 * устанавливает лицензию #5 по умолчанию.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @remark Работает только в режиме `Advanced` конвертера.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_begin_clear_licenses(ilg_handle hConverter,
                                                               ilg_bool fDisconnect,
                                                               ilg_handle* pCommand);

/**
 * @brief Ищет контроллеры.
 *
 * @param[in] hConverter  Дескриптор конвертера.
 * @param[in] fReset      `ILG_TRUE`, сбросить старые результаты поиска.
 * @param[in] fForce      `ILG_TRUE`, повторить поиск, даже если уже выполнен авто поиск.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup ctr_search
 */
ILG_API ilg_status ILG_CALL ilg_converter_scan(ilg_handle hConverter, ilg_bool fReset = ILG_FALSE,
                                               ilg_bool fForce = ILG_FALSE);

/**
 * @brief Запускает асинхронную команду поиска конвертеров.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[in]  fReset      `ILG_TRUE`, очистить список найденных перед поиском.
 * @param[in]  fForce      `ILG_TRUE`, повторить поиск, даже если уже выполнен авто поиск.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @ingroup ctr_search
 */
ILG_API ilg_status ILG_CALL ilg_converter_begin_scan(ilg_handle hConverter, ilg_bool fReset,
                                                     ilg_bool fForce, ilg_handle* pCommand);

/**
 * @brief Возвращает количество найденных контроллеров.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[out] pCount      Количество найденных контроллеров.
 *
 * @ingroup ctr_search
 */
ILG_API ilg_status ILG_CALL ilg_converter_get_controller_count(ilg_handle hConverter,
                                                               size_t* pCount);

/**
 * @brief Возвращает инфо о найденном контроллере.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[in]  nIdx        Позиция в списке найденных контроллеров.
 * @param[out] pInfo       Информация о найденном контроллере.
 *
 * @ingroup ctr_search
 */
ILG_API ilg_status ILG_CALL ilg_converter_get_controller_info(ilg_handle hConverter, size_t nIdx,
                                                              ilg_controller_info* pInfo);

/**
 * @brief Вкл/выкл режим авто поиска контроллеров.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[in]  fEnable     `ILG_TRUE`, включить поиск в реальном времени, иначе - выключить.
 * @param[in]  fWait       `ILG_TRUE`, ждать завершение операции.
 *
 * @remark Если @p fWait =`ILR_TRUE`, то функция не возвращает управление пока ждёт выполнение
 * команды в потоке конвертера.
 *
 * @ingroup ctr_search
 */
ILG_API ilg_status ILG_CALL ilg_converter_set_auto_scan(ilg_handle hConverter,
                                                        ilg_bool fEnable = ILG_TRUE,
                                                        ilg_bool fWait = ILG_FALSE);

/**
 * @brief Запускает асинхронную команду вкл/выкл автоматического сканирования контроллеров.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[in]  fEnable     `ILG_TRUE`, включает авто сканирование, иначе - выключает.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup ctr_search
 */
ILG_API ilg_status ILG_CALL ilg_converter_begin_set_auto_scan(ilg_handle hConverter,
                                                              ilg_bool fEnable,
                                                              ilg_handle* pCommand);

/**
 * @brief Возвращает True если авто поиск контроллеров включен.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[out] pEnabled    `ILG_TRUE`, поиск контроллеров включен, иначе - выключен.
 *
 * @ingroup ctr_search
 */
ILG_API ilg_status ILG_CALL ilg_converter_get_auto_scan(ilg_handle hConverter, ilg_bool* pEnabled);

/**
 * @brief Устанавливает прошивку в конвертер.
 *
 * @param[in] hConverter  Дескриптор конвертера.
 * @param[in] pData       Данные прошивки.
 * @param[in] nSize       Размер данных прошивки в байтах.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_set_firmware(ilg_handle hConverter, const void* pData,
                                                       size_t nSize);

/**
 * @brief Запускает асинхронную команду установки прошивки конвертера.
 *
 * @param[in]  hConverter  Дескриптор конвертера.
 * @param[in]  pData       Данные прошивки.
 * @param[in]  nSize       Размер данных прошивки в байтах.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @ingroup converter
 */
ILG_API ilg_status ILG_CALL ilg_converter_begin_set_firmware(ilg_handle hConverter,
                                                             const void* pData, size_t nSize,
                                                             ilg_handle* pCommand);

/**
 * @brief Возвращает дескриптор подключения к контроллеру.
 *
 * @param[in]  hConverter   Дескриптор конвертера.
 * @param[in]  nModel       Модель контроллера.
 * @param[in]  nSn          Серийный номер контроллера.
 * @param[out] pController  Дескриптор контроллера.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_converter_get_controller(ilg_handle hConverter,
                                                         ilg_controller_model nModel, int nSn,
                                                         ilg_handle* pController);

/**
 * @brief Устанавливает функцию обратного вызова для уведомлений контроллера.
 *
 * Устанавливает функцию для получения сообщений от дескриптора контроллера.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] pCallback    Указатель на функцию, которую библиотека будет вызывать при возникновении
 * события контроллера.
 * @param[in] pUserData    Указатель на данные пользователя, который будет передаваться в функцию.
 *
 * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в `try catch`.
 * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
 * из которого вызвана эта callback-функция, иначе вернёт ошибку `ILG_E_BLOCKING_CALL_NOT_ALLOWED`.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_set_message_callback(
    ilg_handle hController, ilg_controller_message_callback pCallback, void* pUserData = nullptr);

/**
 * @brief Включает/выключает очередь сообщений.
 *
 * Эта функция устанавливает/снимает флаг в дескрипторе контроллера.
 * Очередь сообщений предназначена для синхронизации обработки сообщений.
 *
 * @remark Алгоритм синхронизации: при возникновении события в очередь добавляется сообщение и
 * вызывается функция обратного вызова, установленная функцией @ref
 * ilg_controller_set_message_callback, из которой посылается сигнал потоку, обрабатывающему
 * сообщения, этот поток при получении сигнала циклично вызывает @ref ilg_controller_get_message,
 * чтобы получить и обработать все сообщения.
 *
 * @param[in]  hController  Дескриптор конвертера.
 * @param[in]  fEnable      `ILG_TRUE`, включает очередь, иначе - выключает.
 *
 * @warning Если не извлекать сообщения из очереди функцией @ref ilg_controller_get_message, то она
 * будет расти пока не закончится память.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_enable_message_queue(ilg_handle hController,
                                                                ilg_bool fEnable = ILG_TRUE);

/**
 * @brief Извлекает следующее сообщение из очереди.
 *
 * @param[in]  hController  Дескриптор конвертера.
 * @param[out] pMsg         Тип сообщения.
 * @param[out] pMsgData     Указатель на данные пользователя, которые были установлены функцией
 * @ref ilg_controller_set_message_callback.
 * @param[out] pFound       `ILG_TRUE`, если сообщение успешно извлечено, иначе - очередь пуста.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_get_message(ilg_handle hController,
                                                       ilg_controller_msg* pMsg,
                                                       const void** pMsgData, ilg_bool* pFound);

/**
 * @brief Устанавливает параметры контроллера.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] pOptions     Параметры контроллера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера.
 * @retval ILG_E_INVALIDARG Неправильные параметры: @p pOptions равен `nullptr`.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_set_options(ilg_handle hController,
                                                       const ilg_controller_options* pOptions);

/**
 * @brief Возвращает параметры контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pOptions     Параметры контроллера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера.
 * @retval ILG_E_POINTER Неправильный выходной параметр: @p pOptions равен `nullptr`.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_get_options(ilg_handle hController,
                                                       ilg_controller_options* pOptions);

/**
 * @brief Подключается к контроллеру.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] fReconnect   `ILG_TRUE`, отключается перед подключением.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_INVALIDARG Неправильное имя порта конвертера.
 * @retval ILG_E_OUTOFMEMORY Недостаточно памяти.
 * @retval ILG_E_OUT_OF_RESOURCES Недостаточно ресурсов для завершения операции.
 * @retval ILG_E_BLOCKING_CALL_NOT_ALLOWED Вызов из callback-функции не разрешен.
 * @retval ILG_E_LICENSE_CONTROLLERS Ограничение лицензии на количество контроллеров.
 * @retval ILG_E_PORT_NOT_EXIST Порт конвертера не существует.
 * @retval ILG_E_PORT_ACCESS_DENIED Нет доступа к порту.
 * @retval ILG_E_PORT_OPEN_FAILED Неизвестная ошибка открытия порта конвертера.
 * @retval ILG_E_CONNECTION_ERROR Ошибка подключения.
 * @retval ILG_E_CONNECTION_DOWN Связь с конвертером была потеряна.
 * @retval ILG_E_REQUEST_TIMEOUT Тайм-аут запроса к конвертеру.
 * @retval ILG_E_BAD_RESPONSE Не распознан ответ конвертера.
 * @retval ILG_E_CONVERTER_ERROR Неизвестная ошибка конвертера.
 * @retval ILG_E_CTR_NOT_EXIST Контроллер не существует по указанному адресу.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_connect(ilg_handle hController,
                                                   ilg_bool fReconnect = ILG_FALSE);

/**
 * @brief Запускает асинхронную команду подключения к контроллеру.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  fReconnect   `ILG_TRUE`, переподключиться.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_connect(ilg_handle hController,
                                                         ilg_bool fReconnect, ilg_handle* pCommand);

/**
 * @brief Отключается от контроллера.
 *
 * @param[in] hController  Дескриптор контроллера.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_disconnect(ilg_handle hController);

/**
 * @brief Запускает асинхронную команду отключения от контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pCommand     Дескриптор контроллера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_disconnect(ilg_handle hController,
                                                            ilg_handle* pCommand);

/**
 * @brief Возвращает состояние подключения к контроллеру.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pStatus      Состояние подключения к контроллеру.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_get_connection_status(ilg_handle hController,
                                                                 ilg_connection_status* pStatus);

/**
 * @brief Возвращает информацию о контроллере.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pInfo        Информация о контроллере.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_get_info(ilg_handle hController,
                                                    ilg_controller_info* pInfo);

/**
 * @brief Возвращает информацию о конвертере.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pInfo        Информация о конвертере.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_get_converter_info(ilg_handle hController,
                                                              ilg_converter_info* pInfo);

/**
 * @brief Создаёт дескриптор конвертера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pConverter   Новый дескриптор конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_get_converter(ilg_handle hController,
                                                         ilg_handle* pConverter);

/**
 * @brief Читает информационные строки контроллера.
 *
 * @param[in]     hController  Дескриптор контроллера.
 * @param[in out] pBuf         Буфер для строк контроллера.
 * @param[in out] pSize        Размер буфера в символах.
 *
 * @remark Если @p pSize равно @ref ILG_AUTOALLOCATE, то @p pBuf должен ссылаться на указатель,
 * библиотека выделит память и установит её адрес в указатель, после использования указателя нужно
 * освободить память с помощью @ref ilg_free_memory. При завершении эта функция устанавливает в @p
 * pSize требуемый размер для буфера.
 * Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_POINTER Необходимый выходной параметр равен `nullptr`.
 * @retval ILG_E_BUFFER_TOO_SMALL Размер буфера не достаточный.
 * @retval ILG_E_OUTOFMEMORY Не удалось выделить память.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_lines(ilg_handle hController, char* pBuf,
                                                      size_t* pSize);

/**
 * @brief Запускает асинхронную команду чтения инфо строк контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_lines(ilg_handle hController,
                                                            ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения инфо строк контроллера.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref
 * ilg_controller_begin_read_lines.
 * @param[out] ppLines   Ссылка на строки контроллера. Строки разделены символом '\n' (std::endl),
 * завершается нулевым символом. Ссылка действительна до закрытия дескриптора команды @p hCommand.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор команды @p hCommand.
 * @retval ILG_E_POINTER Необходимый выходной параметр равен `nullptr`.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_lines(ilg_handle hCommand,
                                                          const char** ppLines);

/**
 * @brief Читает времена замка из памяти контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[out] pOpen        Время открывания замка в 1/10 секунды.
 * @param[out] pLet         Время ожидания открытия двери в 1/10 секунды.
 * @param[out] pMax         Время ожидания закрытия двери в 1/10 секунды.
 * @param[out] pBigTime     `ILG_TRUE`, используется формат большого времени 0..65535, иначе 0..255.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_lock_times(ilg_handle hController, uint8_t nBankN,
                                                           uint16_t* pOpen, uint16_t* pLet,
                                                           uint16_t* pMax, ilg_bool* pBigTime);

/**
 * @brief Запускает асинхронную команду чтения времён замка.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_lock_times(ilg_handle hController,
                                                                 uint8_t nBankN,
                                                                 ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения времён замка.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref
 * ilg_controller_begin_read_lock_times.
 * @param[out] pOpen     Время открывания замка в 1/10 секунды.
 * @param[out] pLet      Время ожидания открытия двери в 1/10 секунды.
 * @param[out] pMax      Время ожидания закрытия двери в 1/10 секунды.
 * @param[out] pBigTime  `ILG_TRUE`, используется формат большого времени 0..65535, иначе 0..255.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор команды @p hCommand.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_lock_times(ilg_handle hCommand, uint16_t* pOpen,
                                                               uint16_t* pLet, uint16_t* pMax,
                                                               ilg_bool* pBigTime);

/**
 * @brief Пишет времена замка в память контроллера.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] nBankN       Номер банка: =0 вход, =1 выход.
 * @param[in] nOpen        Время открывания замка в 1/10 секунды.
 * @param[in] nLet         Время ожидания открытия двери в 1/10 секунды.
 * @param[in] nMax         Время ожидания закрытия двери в 1/10 секунды.
 * @param[in] fBigTime     `ILG_TRUE`, используется формат большого времени 0..65535, иначе 0..255.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 * Узнать поддерживает ли контроллер формат большого времени можно с помощью функции @ref
 * ilg_controller_get_info, если в структуре @ref ilg_controller_info в параметре @c nCtrFlags
 * установлен флаг @c ILG_CTR_F_BIGTIME, то поддерживает.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_write_lock_times(ilg_handle hController, uint8_t nBankN,
                                                            uint16_t nOpen, uint16_t nLet,
                                                            uint16_t nMax, ilg_bool fBigTime);

/**
 * @brief Запускает асинхронную команду записи времён замка.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[in]  nOpen        Время открывания замка в 1/10 секунды.
 * @param[in]  nLet         Время ожидания открытия двери в 1/10 секунды.
 * @param[in]  nMax         Время ожидания закрытия двери в 1/10 секунды.
 * @param[in]  fBigTime     `ILG_TRUE`, используется формат большого времени 0..65535, иначе 0..255.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_write_lock_times(ilg_handle hController,
                                                                  uint8_t nBankN, uint16_t nOpen,
                                                                  uint16_t nLet, uint16_t nMax,
                                                                  ilg_bool fBigTime,
                                                                  ilg_handle* pCommand);

/**
 * @brief Читает время антипассбэк из контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pEnabled     `ILG_TRUE`, время задействовано.
 * @param[out] pMinutes     Время антипассбэк в минутах.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_apb_time(ilg_handle hController, ilg_bool* pEnabled,
                                                         uint16_t* pMinutes);

/**
 * @brief Запускает асинхронную команду чтения времени антипассбэк.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_apb_time(ilg_handle hController,
                                                               ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения времени антипассбэк.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref
 * ilg_controller_begin_read_apb_time.
 * @param[out] pEnabled  `ILG_TRUE`, время задействовано.
 * @param[out] pMinutes  Время антипассбэк в минутах.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор команды @p hCommand.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_apb_time(ilg_handle hCommand,
                                                             ilg_bool* pEnabled,
                                                             uint16_t* pMinutes);

/**
 * @brief Пишет время антипассбэк в контроллер.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] fEnabled     `ILG_TRUE`, время задействовано.
 * @param[in] nMinutes     Время антипассбэк в минутах.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_write_apb_time(ilg_handle hController, ilg_bool fEnabled,
                                                          uint16_t nMinutes);

/**
 * @brief Запускает асинхронную команду записи времён замка.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  fEnabled     `ILG_TRUE`, время задействовано.
 * @param[in]  nMinutes     Время антипассбэк в минутах.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_write_apb_time(ilg_handle hController,
                                                                ilg_bool fEnabled,
                                                                uint16_t nMinutes,
                                                                ilg_handle* pCommand);

/**
 * @brief Читает временные зоны из контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[in]  nIdx         Позиция временной зоны в банке.
 * @param[out] pBuf         Буфер для временных зон.
 * @param[in]  nCount       Количество зон, которые нужно прочитать.
 * @param[out] pRead        Количество прочитанных временных зон. Если функция завершилась успешно,
 * то всегда равно @p nCount.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_time_zones(ilg_handle hController, uint8_t nBankN,
                                                           size_t nIdx, ilg_time_zone* pBuf,
                                                           size_t nCount, size_t* pRead);

/**
 * @brief Запускает асинхронную команду чтения временных зон.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[in]  nIdx         Позиция временной зоны в банке.
 * @param[in]  nCount       Количество зон, которые нужно прочитать.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_time_zones(ilg_handle hController,
                                                                 uint8_t nBankN, size_t nIdx,
                                                                 size_t nCount,
                                                                 ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения временных зон.
 *
 * @param[in]  hCommand  Дескриптор команды.
 * @param[out] ppList    Ссылка на список прочитанных временных зон. Ссылка действительна до
 * закрытия дескриптора команды @p hCommand.
 * @param[out] pRead     Количество прочитанных временных зон. Если команда выполнена успешно, то
 * равно запрошенному количеству временных зон.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор команды @p hCommand.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_time_zones(ilg_handle hCommand,
                                                               const ilg_time_zone** ppList,
                                                               size_t* pRead);

/**
 * @brief Пишет временные зоны в контроллер.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[in]  nIdx         Позиция временной зоны в банке.
 * @param[in]  pTZs         Список временных зон.
 * @param[in]  nCount       Количество зон, которые нужно записать.
 * @param[out] pWritten     Количество записанных зон.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_write_time_zones(ilg_handle hController, uint8_t nBankN,
                                                            size_t nIdx, const ilg_time_zone* pTZs,
                                                            size_t nCount, size_t* pWritten);

/**
 * @brief Запускает асинхронную команду записи временных зон.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[in]  nIdx         Позиция временной зоны в банке.
 * @param[in]  pTZs         Список временных зон.
 * @param[in]  nCount       Количество зон, которые нужно записать.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_write_time_zones(ilg_handle hController,
                                                                  uint8_t nBankN, size_t nIdx,
                                                                  const ilg_time_zone* pTZs,
                                                                  size_t nCount,
                                                                  ilg_handle* pCommand);

/**
 * @brief Возвращает результат записи временных зон.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref
 * ilg_controller_begin_write_time_zones.
 * @param[out] pWritten  Количество записанных зон. Если функция выполнена успешно, то равно
 * количеству записываемых временных зон.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор команды @p hCommand.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_write_time_zones(ilg_handle hCommand,
                                                                size_t* pWritten);

/**
 * @brief Читает временные зоны для переключения режима контроллера.
 *
 * Читает из памяти контроллера временные зоны для переключения режима контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nIdx         Позиция временной зоны 0..1.
 * @param[out] pBuf         Буфер для временных зон.
 * @param[in]  nCount       Количество считываемых временных зон 1..2.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_mode_time_zones(ilg_handle hController, size_t nIdx,
                                                                ilg_mode_time_zone* pBuf,
                                                                size_t nCount);

/**
 * @brief Запускает асинхронную команду чтения временных зон для переключения режима контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nIdx         Позиция временной зоны 0..1.
 * @param[in]  nCount       Количество считываемых временных зон 1..2.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_mode_time_zones(ilg_handle hController,
                                                                      size_t nIdx, size_t nCount,
                                                                      ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения временных зон для переключения режима контроллера.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref
 * ilg_controller_begin_read_mode_time_zones.
 * @param[out] ppList    Ссылка на список прочитанных временных зон. Ссылка действительна до
 * закрытия дескриптора команды @p hCommand.
 * @param[out] pRead     Количество прочитанных временных зон. Равно количеству запрошенных зон.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_mode_time_zones(
    ilg_handle hCommand, const ilg_mode_time_zone** ppList, size_t* pRead);

/**
 * @brief Пишет временные зоны для переключения режима контроллера.
 *
 * Пишет в память контроллера временные зоны для переключения режима контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nIdx         Позиция временной зоны 0..1.
 * @param[in]  pTZs         Список временных зон.
 * @param[in]  nCount       Количество записываемых временных зон 1..2.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_write_mode_time_zones(ilg_handle hController,
                                                                 size_t nIdx,
                                                                 const ilg_mode_time_zone* pTZs,
                                                                 size_t nCount);

/**
 * @brief Запускает асинхронную команду записи временных зон для переключения режима контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nIdx         Позиция временной зоны 0..1.
 * @param[in]  pTZs         Список временных зон.
 * @param[in]  nCount       Количество записываемых временных зон 1..2.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_write_mode_time_zones(
    ilg_handle hController, size_t nIdx, const ilg_mode_time_zone* pTZs, size_t nCount,
    ilg_handle* pCommand);

/**
 * @brief Читает ключи из памяти контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[in]  nIdx         Позиция ключа в банке ключей.
 * @param[out] pBuf         Буфер для ключей.
 * @param[in]  nCount       Количество ключей, которые нужно прочитать.
 * @param[out] pRead        Количество прочитанных ключей. Если функция завершилась успешно, то
 * всегда равно @p nCount.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_keys(ilg_handle hController, uint8_t nBankN,
                                                     size_t nIdx, ilg_key* pBuf, size_t nCount,
                                                     size_t* pRead = nullptr);

/**
 * @brief Запускает асинхронную команду чтения ключей.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[in]  nIdx         Позиция ключа в банке ключей.
 * @param[in]  nCount       Количество ключей, которые нужно прочитать.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_keys(ilg_handle hController, uint8_t nBankN,
                                                           size_t nIdx, size_t nCount,
                                                           ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения ключей.
 *
 * @param[in]  hCommand     Дескриптор команды, который вернула @ref ilg_controller_begin_read_keys.
 * @param[out] ppList       Ссылка на массив прочитанных ключей. Ссылка действительна до закрытия
 * дескриптора команды.
 * @param[out] pRead        Количество прочитанных ключей. Если команда выполнена успешно, то равно
 * количеству запрошенных ключей.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_keys(ilg_handle hCommand,
                                                         const ilg_key** ppList, size_t* pRead);

/**
 * @brief Пишет ключи в память контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[in]  nIdx         Позиция ключа в банке ключей.
 * @param[in]  pKeys        Список ключей.
 * @param[in]  nCount       Количество ключей, которые нужно записать.
 * @param[out] pWritten     Количество записанных ключей. Если функция завершилась успешно, то
 * всегда равно @p nCount.
 * @param[in]  fSelected    `ILG_TRUE`, записывать только выделенные ключи с установленным флагом
 * fSelected в @ref ilg_key.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 * @warning В режиме `Advanced` конвертера действует лицензия (@see @ref ilg_license_info), которая
 * может ограничивать до какой позиции можно записывать ключи.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_write_keys(ilg_handle hController, uint8_t nBankN,
                                                      size_t nIdx, const ilg_key* pKeys,
                                                      size_t nCount, size_t* pWritten,
                                                      ilg_bool fSelected);

/**
 * @brief Запускает асинхронную команду записи ключей.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[in]  nIdx         Позиция ключа в банке ключей.
 * @param[in]  pKeys        Список ключей.
 * @param[in]  nCount       Количество ключей, которые нужно записать.
 * @param[in]  fSelected    `ILG_TRUE`, записывать только выделенные ключи с установленным флагом
 * ilg_key.fSelected.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_write_keys(ilg_handle hController, uint8_t nBankN,
                                                            size_t nIdx, const ilg_key* pKeys,
                                                            size_t nCount, ilg_bool fSelected,
                                                            ilg_handle* pCommand);

/**
 * @brief Возвращает результат записи ключей.
 *
 * @param[in]  hCommand     Дескриптор команды, который вернула @ref
 * ilg_controller_begin_write_keys.
 * @param[out] pWritten     Количество записанных ключей. Если команда выполнена успешно, то равно
 * количеству записываемых ключей.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_write_keys(ilg_handle hCommand, size_t* pWritten);

/**
 * @brief Стирает ключи в памяти контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[in]  nIdx         Позиция ключа в банке ключей.
 * @param[in]  nCount       Количество ключей, которые нужно стереть.
 * @param[out] pErased      Количество стёртых ключей. Если функция завершилась успешно, то всегда
 * равно @p nCount.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_erase_keys(ilg_handle hController, uint8_t nBankN,
                                                      size_t nIdx, size_t nCount, size_t* pErased);

/**
 * @brief Запускает асинхронную команду стирания ключей.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[in]  nIdx         Позиция ключа в банке ключей.
 * @param[in]  nCount       Количество ключей, которые нужно стереть.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_erase_keys(ilg_handle hController, uint8_t nBankN,
                                                            size_t nIdx, size_t nCount,
                                                            ilg_handle* pCommand);

/**
 * @brief Возвращает результат стирания ключей.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref ilg_controller_begin_erase_keys.
 * @param[out] pErased   Количество стёртых ключей. Если функция завершилась успешно, то равно
 * количеству стираемых ключей.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_erase_keys(ilg_handle hCommand, size_t* pErased);

/**
 * @brief Стирает ключи в указанных ячейках.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[in]  pIdxs        Список позиций ключей, сортированный по возрастанию.
 * @param[in]  nCount       Количество ключей, которые нужно стереть.
 * @param[out] pErased      Количество стёртых ключей. Если функция завершилась успешно, то всегда
 * равно @p nCount.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_erase_keys2(ilg_handle hController, uint8_t nBankN,
                                                       const size_t* pIdxs, size_t nCount,
                                                       size_t* pErased);

/**
 * @brief Запускает асинхронную команду стирания ключей в указанных ячейках.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[in]  pIdxs        Список позиций ключей, сортированный по возрастанию.
 * @param[in]  nCount       Количество ключей, которые нужно стереть.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_erase_keys2(ilg_handle hController, uint8_t nBankN,
                                                             const size_t* pIdxs, size_t nCount,
                                                             ilg_handle* pCommand);

/**
 * @brief Возвращает результат стирания ключей в указанных ячейках.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref ilg_controller_begin_erase_keys2.
 * @param[out] pErased   Количество стёртых ключей. Если функция завершилась успешно, то равно
 * количеству стираемых ключей.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_erase_keys2(ilg_handle hCommand, size_t* pErased);

/**
 * @brief Читает верхнюю границу ключей.
 *
 * Читает из контроллера позицию верхней границы ключей, начиная с которой все ячейки стёрты.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[out] pIdx         Позиция верхней границы ключей. Если равно -1, то в контроллере адрес
 * верхней границы не корректный.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_key_top_idx(ilg_handle hController, uint8_t nBankN,
                                                            ssize_t* pIdx);

/**
 * @brief Запускает асинхронную команду чтения верхней границы ключей.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nBankN       Номер банка: =0 вход, =1 выход.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_key_top_idx(ilg_handle hController,
                                                                  uint8_t nBankN,
                                                                  ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения верхней границы ключей.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref
 * ilg_controller_begin_read_key_top_idx.
 * @param[out] pIdx      Позиция верхней границы ключей. Если равно -1, то в контроллере адрес
 * верхней границы не корректный.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_key_top_idx(ilg_handle hCommand, ssize_t* pIdx);

/**
 * @brief Читает события из памяти контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nIdx         Позиция события в банке событий.
 * @param[out] pBuf         Буфер для событий.
 * @param[in]  nCount       Количество событий, которые нужно прочитать.
 * @param[out] pRead        Количество прочитанных событий. Если функция завершилась успешно, то
 * всегда равно @p nCount.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 * Получить позиции указателя чтения и указателя записи можно с помощью функции @ref
 * ilg_controller_read_rtc_params.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_events(ilg_handle hController, size_t nIdx,
                                                       uint64_t* pBuf, size_t nCount,
                                                       size_t* pRead = nullptr);

/**
 * @brief Запускает асинхронную команду чтения событий.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nIdx         Позиция события в банке событий.
 * @param[in]  nCount       Количество событий, которые нужно прочитать.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_events(ilg_handle hController, size_t nIdx,
                                                             size_t nCount, ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения событий.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref ilg_controller_begin_read_events.
 * @param[out] ppList    Ссылка на список прочитанных событий. Ссылка действительна до закрытия
 * дескриптора команды.
 * @param[out] pRead     Количество прочитанных событий. Если функция завершилась успешно, то
 * равно количеству запрошенных событий.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_events(ilg_handle hCommand,
                                                           const uint64_t** ppList, size_t* pRead);

/**
 * @brief Пишет позицию указателя чтения событий.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nIdx         Позиция указателя чтения событий.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_write_event_read_idx(ilg_handle hController,
                                                                size_t nIdx);

/**
 * @brief Запускает асинхронную команду записи указателя чтения событий.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nIdx         Позиция указателя чтения событий.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_write_event_read_idx(ilg_handle hController,
                                                                      size_t nIdx,
                                                                      ilg_handle* pCommand);

/**
 * @brief Читает параметры RTC из контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pParams      Параметры RTC.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_rtc_params(ilg_handle hController,
                                                           ilg_rtc_params* pParams);

/**
 * @brief Запускает асинхронную команду чтения параметров RTC.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_rtc_params(ilg_handle hController,
                                                                 ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения параметров RTC.
 *
 * @param[in]  hCommand  Дескриптор команды.
 * @param[out] pParams   Параметры RTC.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_rtc_params(ilg_handle hCommand,
                                                               ilg_rtc_params* pParams);

/**
 * @brief Возвращает результат авто опроса параметров RTC.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pParams      Параметры RTC.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_get_rtc_params(ilg_handle hController,
                                                          ilg_rtc_params* pParams);

/**
 * @brief Читает параметры ExtAsk из контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pParams      Параметры ExtAsk.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_extask_params(ilg_handle hController,
                                                              ilg_extask_params* pParams);

/**
 * @brief Запускает асинхронную команду чтения параметров ExtAsk.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_extask_params(ilg_handle hController,
                                                                    ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения параметров ExtAsk.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref
 * ilg_controller_begin_read_extask_params.
 * @param[out] pParams   Параметры ExtAsk.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_extask_params(ilg_handle hCommand,
                                                                  ilg_extask_params* pParams);

/**
 * @brief Возвращает результат авто опроса ExtAsk.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pParams      Параметры ExtAsk.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_get_extask_params(ilg_handle hController,
                                                             ilg_extask_params* pParams);

/**
 * @brief Синхронизирует часы контроллера с часами ПК.
 *
 * @param[in]  hController  Дескриптор контроллера.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 * Получить текущее время контроллера можно с помощью функции @ref ilg_controller_read_rtc_params.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_sync_clock(ilg_handle hController);

/**
 * @brief Запускает асинхронную команду синхронизации часов контроллера с часами ПК.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @remark Получить текущее время контроллера можно с помощью функции @ref
 * ilg_controller_read_rtc_params.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_sync_clock(ilg_handle hController,
                                                            ilg_handle* pCommand);

/**
 * @brief Вкл/выкл старый режим аварийного открывания дверей.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  fEnable      `ILG_TRUE`, включить аварийный режим.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_set_emergency_mode(ilg_handle hController,
                                                              ilg_bool fEnable);

/**
 * @brief Запускает асинхронную команду вкл/выкл старого режима аварийного открывания дверей.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  fEnable      `ILG_TRUE`, включить аварийный режим.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_set_emergency_mode(ilg_handle hController,
                                                                    ilg_bool fEnable,
                                                                    ilg_handle* pCommand);

/**
 * @brief Читает параметры AntiCovid.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pMax8        Максимумы, массив из 8 элементов.
 * @param[out] pCount8      Счётчики, массив из 8 элементов.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_anticovid(ilg_handle hController, uint16_t* pMax8,
                                                          uint16_t* pCount8);

/**
 * @brief Запускает асинхронную команду чтения параметров AntiCovid.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_anticovid(ilg_handle hController,
                                                                ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения параметров AntiCovid.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref
 * ilg_controller_begin_read_anticovid.
 * @param[out] pMax8     Максимумы, массив из 8 элементов.
 * @param[out] pCount8   Счётчики, массив из 8 элементов.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_anticovid(ilg_handle hCommand, uint16_t* pMax8,
                                                              uint16_t* pCount8);

/**
 * @brief Пишет максимумы AntiCovid.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] pMax8        Максимумы, массив из 8 элементов.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_write_anticovid(ilg_handle hController,
                                                           const uint16_t* pMax8);

/**
 * @brief Запускает асинхронную команду записи максимумов AntiCovid.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  pMax8        Максимумы, массив из 8 элементов.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_write_anticovid(ilg_handle hController,
                                                                 const uint16_t* pMax8,
                                                                 ilg_handle* pCommand);

/**
 * @brief Открывает дверь.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] fOut         `ILG_TRUE`, открывает для выхода, иначе - для входа.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_open_door(ilg_handle hController, ilg_bool fOut);

/**
 * @brief Запускает асинхронную команду открытия двери.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] fOut         `ILG_TRUE`, открывает для выхода, иначе - для входа.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_open_door(ilg_handle hController, ilg_bool fOut,
                                                           ilg_handle* pCommand);

/**
 * @brief Вкл/выкл противопожарный режим.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] fOn          `ILG_TRUE`, включить режим.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_set_fire_mode(ilg_handle hController, ilg_bool fOn);

/**
 * @brief Запускает асинхронную команду вкл/выкл противопожарного режима.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  fOn          `ILG_TRUE`, включить режим.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_set_fire_mode(ilg_handle hController, ilg_bool fOn,
                                                               ilg_handle* pCommand);

/**
 * @brief Вкл/выкл режим охраны.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] fOn          `ILG_TRUE`, включить режим.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_set_security_mode(ilg_handle hController, ilg_bool fOn);

/**
 * @brief Запускает асинхронную команду вкл/выкл режима охраны.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  fOn          `ILG_TRUE`, включить режим.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_set_security_mode(ilg_handle hController,
                                                                   ilg_bool fOn,
                                                                   ilg_handle* pCommand);

/**
 * @brief Вкл/выкл режим тревоги.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] fOn          `ILG_TRUE`, включить режим.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_set_alarm_mode(ilg_handle hController, ilg_bool fOn);

/**
 * @brief Запускает асинхронную команду вкл/выкл режима тревоги.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] fOn          `ILG_TRUE`, включить режим.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_set_alarm_mode(ilg_handle hController,
                                                                ilg_bool fOn, ilg_handle* pCommand);

/**
 * @brief Вкл/выкл питание в режиме Электроконтроль.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] fOn          `ILG_TRUE`, включить питание.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_set_ec_power(ilg_handle hController, ilg_bool fOn);

/**
 * @brief Запускает асинхронную команду вкл/выкл питания в режиме Электроконтроль.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] fOn          `ILG_TRUE`, включить питание.
 * @param[out] pCommand    Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_set_ec_power(ilg_handle hController, ilg_bool fOn,
                                                              ilg_handle* pCommand);

/**
 * @brief Читает параметры противопожарного режима.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pSrcMask     Маска разрешения/запрещения источников.
 *  Бит 0 - разрешен пожарный режим по входу FIRE.
 *  Бит 1 - разрешен пожарный режим по превышению температуры.
 * @param[out] pLimitT      Пороговая температура.
 * @param[out] pState       Флаги состояния.
 *  Бит 0 - пожарный режим включён.
 *  Бит 1 - активен пожарный режим по входу FIRE.
 *  Бит 2 - активен пожарный режим по превышению температуры.
 *  Бит 3 - активен пожарный режим по внешней команде.
 * @param[out] pT           Текущая температура.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_fire_params(ilg_handle hController,
                                                            uint8_t* pSrcMask, uint8_t* pLimitT,
                                                            uint8_t* pState, uint8_t* pT);

/**
 * @brief Запускает асинхронную команду чтения параметров противопожарного режима.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_fire_params(ilg_handle hController,
                                                                  ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения параметров противопожарного режима.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref
 * ilg_controller_begin_read_fire_params.
 * @param[out] pSrcMask  Маска разрешения/запрещения источников.
 * @param[out] pLimitT   Пороговая температура.
 * @param[out] pState    Флаги состояния.
 * @param[out] pT        Текущая температура.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_fire_params(ilg_handle hCommand,
                                                                uint8_t* pSrcMask, uint8_t* pLimitT,
                                                                uint8_t* pState, uint8_t* pT);

/**
 * @brief Пишет параметры противопожарного режима.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nSrcMask     Маска разрешения/запрещения источников.
 * @param[in]  nLimitT      Пороговая температура.
 * @param[out] pState       Флаги состояния.
 * @param[out] pT           Текущая температура.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_write_fire_params(ilg_handle hController,
                                                             uint8_t nSrcMask, uint8_t nLimitT,
                                                             uint8_t* pState, uint8_t* pT);

/**
 * @brief Запускает асинхронную команду записи параметров противопожарного режима.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nSrcMask     Маска разрешения/запрещения источников.
 * @param[in]  nLimitT      Пороговая температура.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_write_fire_params(ilg_handle hController,
                                                                   uint8_t nSrcMask,
                                                                   uint8_t nLimitT,
                                                                   ilg_handle* pCommand);

/**
 * @brief Возвращает результат записи параметров противопожарного режима.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref
 * ilg_controller_begin_write_fire_params.
 * @param[out] pState    Флаги состояния.
 * @param[out] pT        Текущая температура.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_write_fire_params(ilg_handle hCommand,
                                                                 uint8_t* pState, uint8_t* pT);

/**
 * @brief Читает параметры режима охраны.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pSrcMask     Маска разрешения/запрещения источников.
 *  Бит 0 - разрешена тревога по входу ALARM.
 *  Бит 1 - разрешена тревога по тамперу.
 *  Бит 2 - разрешена тревога по датчику двери.
 * @param[out] pTime        Время звучания сирены после исчезновения источника тревоги.
 * @param[out] pState       Флаги состояния.
 *  Бит 0 - охранный режим включён.
 *  Бит 1 - тревога включена.
 *  Бит 2 - тревога по входу ALARM.
 *  Бит 3 - тревога по тамперу.
 *  Бит 4 - тревога по датчику двери.
 *  Бит 5 - тревога включена по сети.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_security_params(ilg_handle hController,
                                                                uint8_t* pSrcMask, uint8_t* pTime,
                                                                uint8_t* pState);

/**
 * @brief Запускает асинхронную команду чтения параметров режима охраны.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_security_params(ilg_handle hController,
                                                                      ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения параметров режима охраны.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref
 * ilg_controller_begin_read_security_params.
 * @param[out] pSrcMask  Маска разрешения/запрещения источников.
 * @param[out] pTime     Время звучания сирены после исчезновения источника тревоги.
 * @param[out] pState    Флаги состояния.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_security_params(ilg_handle hCommand,
                                                                    uint8_t* pSrcMask,
                                                                    uint8_t* pTime,
                                                                    uint8_t* pState);

/**
 * @brief Пишет параметры режима охраны.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nSrcMask     Маска разрешения/запрещения источников.
 * @param[in]  nTime        Время звучания сирены после исчезновения источника тревоги.
 * @param[out] pState       Флаги состояния.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_write_security_params(ilg_handle hController,
                                                                 uint8_t nSrcMask, uint8_t nTime,
                                                                 uint8_t* pState);

/**
 * @brief Запускает асинхронную команду записи параметров режима охраны.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nSrcMask     Маска разрешения/запрещения источников.
 * @param[in]  nTime        Время звучания сирены после исчезновения источника тревоги.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_write_security_params(ilg_handle hController,
                                                                       uint8_t nSrcMask,
                                                                       uint8_t nTime,
                                                                       ilg_handle* pCommand);

/**
 * @brief Возвращает результат записи параметров режима охраны.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref
 * ilg_controller_begin_write_security_params.
 * @param[out] pState    Флаги состояния.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_write_security_params(ilg_handle hCommand,
                                                                     uint8_t* pState);

/**
 * @brief Читает параметры режима ЭлектроКонтроль.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pPowerConfig Флаги конфигурации режима.
 *  Бит 0 (ILG_ECPC_F_ENABLED) Управление питанием включено.
 *  Бит 1 (ILG_ECPC_F_SCHEDULE) Использовать временную зону 6 для включения питания.
 *  Бит 2 (ILG_ECPC_F_EXT_READER) Контрольный считыватель: «0» Matrix-II Net, «1» внешний
 * считыватель.
 *  Бит 3 (ILG_ECPC_F_INVERT) Инвертировать управляющий выход.
 *  Бит 4 (ILG_ECPC_F_EXIT_OFF) Задействовать датчик двери.
 *  Бит 5 (ILG_ECPC_F_CARD_OPEN) Не блокировать функцию открывания для контрольного считывателя.
 * @param[out] pDelay       Время задержки в секундах.
 * @param[out] pState       Флаги состояния.
 *  Бит 0 - питание включено.
 *  Бит 1 - активно включение по временной зоне.
 *  Бит 2 - включено командой по сети
 *  Бит 3 - идет отработка задержки.
 *  Бит 4 - карта в поле контрольного считывателя.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 * Для установки параметров режима ЭлектроКонтроль используйте функцию @ref
 * ilg_controller_write_time_zones (временная зона 6).
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_ec_params(ilg_handle hController,
                                                          uint8_t* pPowerConfig, uint8_t* pDelay,
                                                          uint8_t* pState);

/**
 * @brief Запускает асинхронную команду чтения параметров режима Электроконтроль.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_ec_params(ilg_handle hController,
                                                                ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения параметров режима ЭлектроКонтроль.
 *
 * @param[in]  hCommand     Дескриптор команды, который вернула @ref
 * ilg_controller_begin_read_ec_params.
 * @param[out] pPowerConfig Флаги конфигурации режима.
 * @param[out] pDelay       Время задержки в секундах.
 * @param[out] pState       Флаги состояния.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_ec_params(ilg_handle hCommand,
                                                              uint8_t* pPowerConfig,
                                                              uint8_t* pDelay, uint8_t* pState);

/**
 * @brief Пишет режим контроллера.
 *
 * Устанавливает в контроллер текущий режим контроллера.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] nMode        Режим контроллера.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_write_mode(ilg_handle hController,
                                                      ilg_controller_mode nMode);

/**
 * @brief Запускает асинхронную команду записи режима контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nMode        Режим контроллера.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_write_mode(ilg_handle hController,
                                                            ilg_controller_mode nMode,
                                                            ilg_handle* pCommand);

/**
 * @brief Читает режим контроллера.
 *
 * Возвращает текущий режим контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pMode        Режим контроллера.
 * @param[out] pFlags       Флаги активизации.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_mode(ilg_handle hController,
                                                     ilg_controller_mode* pMode, uint8_t* pFlags);

/**
 * @brief Запускает асинхронную команду чтения режима контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_mode(ilg_handle hController,
                                                           ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения режима контроллера.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref ilg_controller_begin_read_mode.
 * @param[out] pMode     Режим контроллера.
 * @param[out] pFlags    Флаги активизации.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_mode(ilg_handle hCommand,
                                                         ilg_controller_mode* pMode,
                                                         uint8_t* pFlags);

/**
 * @brief Пишет конфигурацию контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  pData        Данные конфигурации.
 * @param[in]  nSize        Размер данных в байтах.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера. Данные
 * конфигурации можно получить с помощью ПО GuardLight.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 * @retval ILG_E_INVALIDARG Данные конфигурации пустые, т.е. @p pData равен `nullptr` или @p nSize
 * равен 0.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_write_configuration(ilg_handle hController,
                                                               const void* pData, size_t nSize);

/**
 * @brief Запускает асинхронную команду записи конфигурации контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  pData        Данные конфигурации.
 * @param[in]  nSize        Размер данных в байтах.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 * @retval ILG_E_INVALIDARG Данные конфигурации пустые, т.е. @p pData равен `nullptr` или @p nSize
 * равен 0.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_write_configuration(ilg_handle hController,
                                                                     const void* pData,
                                                                     size_t nSize,
                                                                     ilg_handle* pCommand);

/**
 * @brief Читает конфигурацию контроллера.
 *
 * @param[in]     hController  Дескриптор контроллера.
 * @param[out]    pBuf         Буфер для данных конфигурации.
 * @param[in out] pSize        Размер буфера в байтах.
 *
 * @remark Если @p pSize равно @ref ILG_AUTOALLOCATE, то @p pBuf должен ссылаться на указатель,
 * библиотека выделит память и установит её адрес в указатель, после использования указателя нужно
 * освободить память с помощью @ref ilg_free_memory. При завершении эта функция устанавливает в @p
 * pSize требуемый размер для буфера.
 * Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_POINTER Необходимый выходной параметр равен `nullptr`.
 * @retval ILG_E_BUFFER_TOO_SMALL Размер буфера не достаточный.
 * @retval ILG_E_OUTOFMEMORY Не удалось выделить память.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_read_configuration(ilg_handle hController, uint8_t* pBuf,
                                                              size_t* pSize);

/**
 * @brief Запускает асинхронную команду чтения конфигурации контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается контроллером, @see @ref
 * ilg_controller_flags.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_read_configuration(ilg_handle hController,
                                                                    ilg_handle* pCommand);

/**
 * @brief Возвращает результат чтения конфигурации контроллера.
 *
 * @param[in]  hCommand  Дескриптор команды, который вернула @ref
 * ilg_controller_begin_read_configuration.
 * @param[out] ppData    Ссылка на данные конфигурации. Ссылка действительна до закрытия дескриптора
 * команды @p hCommand.
 * @param[out] pSize     Размер данных в байтах.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор команды @p hCommand.
 * @retval ILG_E_POINTER Необходимый выходной параметр равен `nullptr`.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_end_read_configuration(ilg_handle hCommand,
                                                                  const uint8_t** ppData,
                                                                  size_t* pSize);

/**
 * @brief Устанавливает сетевой адрес контроллеру.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] nAddress     Адрес контроллера 0..254.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 * Работает только в режиме Normal конвертера или с Z-397 Guard.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается конвертером или текущим режимом конвертера.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_set_network_address(ilg_handle hController,
                                                               uint8_t nAddress);

/**
 * @brief Запускает асинхронную команду установки сетевого адреса.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nAddress     Адрес контроллера 0..254.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_NOT_SUPPORTED Команда не поддерживается конвертером или текущим режимом конвертера.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_set_network_address(ilg_handle hController,
                                                                     uint8_t nAddress,
                                                                     ilg_handle* pCommand);

/**
 * @brief Устанавливает прошивку контроллера.
 *
 * @param[in] hController  Дескриптор контроллера.
 * @param[in] pData        Данные прошивки.
 * @param[in] nSize        Размер данных прошивки в байтах.
 *
 * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке конвертера.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_set_firmware(ilg_handle hController, const void* pData,
                                                        size_t nSize);

/**
 * @brief Запускает асинхронную команду установки прошивки контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  pData        Данные прошивки.
 * @param[in]  nSize        Размер данных прошивки в байтах.
 * @param[out] pCommand     Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_INVALIDARG Данные прошивки пустые, т.е. @p pData равен nullptr или @p nSize равен
 * 0.
 * @retval ILG_E_POINTER Выходной параметр @p pCommand равен `nullptr`.
 * @retval ILG_E_OUTOFMEMORY Недостаточно памяти.
 * @retval ILG_E_FIRMWARE_IMAGE_INVALID Некорректный размер данных прошивки.
 * @retval ILG_E_BOOTLOADER_NOSTART Не удалось активировать режим прошивки.
 * @retval ILG_E_NO_COMPATIBLE Не подходит для данной модели устройства.
 * @retval ILG_E_INVALID_DEV_NUM Не подходит для данного серийного номера устройства.
 * @retval ILG_E_FIRMWARE_OTHER Неизвестная ошибка прошивки.
 * @retval ILG_E_RUN_FIRMWARE_FAIL Не удалось запустить программу контроллера.
 *
 * @ingroup controller
 */
ILG_API ilg_status ILG_CALL ilg_controller_begin_set_firmware(ilg_handle hController,
                                                              const void* pData, size_t nSize,
                                                              ilg_handle* pCommand);

/**
 * @brief Определяет тип события контроллера и формат записи события.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nEvent       Данные события.
 * @param[out] pFormat      Формат записи события.
 * @param[out] pType        Тип события.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_POINTER Выходной параметр @p pType равен `nullptr`.
 */
ILG_API ilg_status ILG_CALL ilg_controller_decode_event_type(ilg_handle hController,
                                                             const uint64_t& nEvent,
                                                             ilg_event_format* pFormat,
                                                             ilg_event_type* pType);

/**
 * @brief Декодирует событие прохода.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nEvent       Данные события.
 * @param[out] pTime        Дата и время события.
 * @param[out] pDirection   Направление прохода.
 * @param[out] pKeyBankN    Номер банка ключей.
 * @param[out] pKeyIdx      Позиция ключа в банке ключей.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_POINTER Выходной параметр равен `nullptr`.
 */
ILG_API ilg_status ILG_CALL ilg_controller_decode_passage_event(
    ilg_handle hController, const uint64_t& nEvent, ilg_controller_time* pTime,
    ilg_direction* pDirection, uint8_t* pKeyBankN, ssize_t* pKeyIdx);

/**
 * @brief Декодирует событие с датой и временем.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nEvent       Данные события.
 * @param[out] pTime        Дата и время события.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_POINTER Выходной параметр равен `nullptr`.
 */
ILG_API ilg_status ILG_CALL ilg_controller_decode_time_event(ilg_handle hController,
                                                             const uint64_t& nEvent,
                                                             ilg_controller_time* pTime);

/**
 * @brief Декодирует событие переключения режима контроллера.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nEvent       Данные события.
 * @param[out] pTime        Дата и время события.
 * @param[out] pMode        Режим контроллера.
 * @param[out] pFlags       Флаги состояния.
 * @param[out] pTrigger     Код условия вызвавшего срабатывание.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_POINTER Выходной параметр равен `nullptr`.
 */
ILG_API ilg_status ILG_CALL ilg_controller_decode_controller_mode_event(
    ilg_handle hController, const uint64_t& nEvent, ilg_controller_time* pTime,
    ilg_controller_mode* pMode, uint8_t* pFlags, uint8_t* pTrigger);

/**
 * @brief Декодирует событие изменения состояния.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nEvent       Данные события.
 * @param[out] pTime        Дата и время события.
 * @param[out] pFlags       Флаги состояния.
 * @param[out] pTrigger     Код условия вызвавшего срабатывание.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_POINTER Выходной параметр равен `nullptr`.
 */
ILG_API ilg_status ILG_CALL ilg_controller_decode_state_event(ilg_handle hController,
                                                              const uint64_t& nEvent,
                                                              ilg_controller_time* pTime,
                                                              uint8_t* pFlags, uint8_t* pTrigger);

/**
 * @brief Декодирует событие с номером ключа.
 *
 * @param[in]  hController  Дескриптор контроллера.
 * @param[in]  nEvent       Данные события.
 * @param[out] pKeyNumber   Номер ключа.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор контроллера @p hController.
 * @retval ILG_E_POINTER Выходной параметр равен `nullptr`.
 */
ILG_API ilg_status ILG_CALL ilg_controller_decode_key_number(ilg_handle hController,
                                                             const uint64_t& nEvent,
                                                             ilg_key_number* pKeyNumber);

/**
 * @brief Отменяет команду. Устанавливает статус `ILG_E_ABORT`.
 *
 * @param[in] hCommand Дескриптор команды.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор команды @p hCommand.
 */
ILG_API ilg_status ILG_CALL ilg_command_cancel(ilg_handle hCommand);

/**
 * @brief Возвращает состояние команды.
 *
 * @param[in]  hCommand  Дескриптор команды.
 * @param[out] pStatus   Состояние команды: `ILG_E_PENDING`, команда ещё выполняется, иначе -
 * завершена.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор команды @p hCommand.
 * @retval ILG_E_POINTER Выходной параметр равен `nullptr`.
 */
ILG_API ilg_status ILG_CALL ilg_command_get_status(ilg_handle hCommand, ilg_status* pStatus);

/**
 * @brief Возвращает состояние прогресса выполнения команды.
 *
 * @param[in]  hCommand  Дескриптор команды.
 * @param[out] pCurrent  Текущий шаг.
 * @param[out] pTotal    Всего шагов.
 *
 * @return `ILG_OK` в случае успеха или код ошибки в случае неудачи.
 * @retval ILG_E_HANDLE Неправильный дескриптор команды @p hCommand.
 * @retval ILG_E_POINTER Выходной параметр равен `nullptr`.
 */
ILG_API ilg_status ILG_CALL ilg_command_get_progress(ilg_handle hCommand, size_t* pCurrent,
                                                     size_t* pTotal);

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif  // ILGUARD_H_INCLUDED
