#include <exception>
#include <fstream>   // для std::fstream
#include <iomanip>   // для std::put_time
#include <iostream>  // для std::cout и std::cin

#include "ilg_cpp_helpers.h"
#include "ilguard/ilguard.h"

// #define ILG_LOG  // Раскомментируйте, чтобы включить показ отладочных сообщений
#define ILG_LOG_FILE  // Писать лог в файл

using namespace ilg;

#ifdef ILG_LOG
const char kLogLevelChars[] = {'-', 'E', 'W', 'I', 'D'};
const char kLogFileName[] = "ilguard.log";  // Путь к лог файлу

void ILG_CALL LogCallback(ilg_log_level level, const char* pContext, const char* pMessage, void*) {
#if 1  // Запись в файл
    std::fstream file(kLogFileName, std::ios_base::out | std::ios_base::app);
    auto& out = file;
#else  // иначе в консоль
    auto& out = std::cout;
#endif
    auto t = std::time(nullptr);
    auto tmb = std::localtime(&t);
    out << std::put_time(tmb, "%d-%m-%Y %H:%M:%S") << " [" << kLogLevelChars[level] << ' '
        << pContext << "] " << pMessage << std::endl;
}
#endif

bool DoConnectToConverter(CILG& oILR, CConverter& oConverter) {
    // Ищем конвертеры
    CConverterSearch oSearch(oILR.GetSearch());
    std::cout << "Поиск конвертеров..." << std::endl;
    oSearch.Scan();
    auto nCount = oSearch.GetConverterCount();
    if (0 == nCount) {
        std::cout << "Конвертер не найден" << std::endl;
        return false;
    }
    std::cout << "Найдено конвертеров: " << nCount << std::endl;

    std::cout << std::endl << "Выберите конвертер:" << std::endl;
    for (size_t i = 0; i < nCount; i++) {
        ilg_converter_info rInfo;
        oSearch.GetConverterInfo(i, rInfo);

        std::stringstream ss;
        if (rInfo.nModel != ILG_CONVERTER_MODEL_UNKNOWN)
            ss << kConverterModelNames[rInfo.nModel];
        if (rInfo.nSn != -1)
            ss << " с/н:" << rInfo.nSn;
        if (rInfo.nFwVersion != 0)
            ss << " прошивка:" << VersionToStr(rInfo.nFwVersion);
        if (rInfo.nFwBuildDate != 0)
            ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
        if (rInfo.nMode != ILG_CONVERTER_MODE_UNKNOWN)
            ss << " режим: " << kConverterModeNames[rInfo.nMode];
        std::cout << 1 + i << ". " << rInfo.pszPortName << " [" << rInfo.pszConnect
                  << "]: " << ss.str() << std::endl;
    }
    std::cout << "0 - Выйти из программы" << std::endl;

    int nCommand;
    std::cin >> nCommand;
    std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
    if (std::cin.fail()) {
        std::cin.clear();
        nCommand = -1;
    }

    if ((nCommand <= 0) || (static_cast<size_t>(nCommand) > nCount))
        return false;

    ilg_converter_info rInfo;
    oSearch.GetConverterInfo(static_cast<size_t>(nCommand - 1), rInfo);

    // Подключаемся к конвертеру
    // Дескриптор поиска oSearch не удаляем пока не подключимся, т.к. в нём удерживаются
    // подключения к конвертерам, подключённым к IP конвертеру в режиме "Клиент"
    std::cout << "Подключение к конвертеру [" << kPortTypeNames[rInfo.nPortType] << ": "
              << rInfo.pszPortName << "]... " << std::endl;
    oConverter = oILR.GetConverter(rInfo.nPortType, rInfo.pszPortName);
    ilg_converter_options rOptions;
    oConverter.GetOptions(rOptions);
    rOptions.nConnectModel = rInfo.nModel;
    oConverter.SetOptions(rOptions);
    oConverter.SetAutoScan(false);
    oConverter.Connect();
    // Получаем информацию о конвертере
    oConverter.GetConverterInfo(rInfo);
    std::stringstream ss;
    if (rInfo.nModel != ILG_CONVERTER_MODEL_UNKNOWN)
        ss << kConverterModelNames[rInfo.nModel];
    if (rInfo.nSn != -1)
        ss << " с/н:" << rInfo.nSn;
    if (rInfo.nFwVersion != 0)
        ss << " прошивка:" << VersionToStr(rInfo.nFwVersion);
    if (rInfo.nFwBuildDate != 0)
        ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
    if (rInfo.nMode != ILG_CONVERTER_MODE_UNKNOWN)
        ss << " режим: " << kConverterModeNames[rInfo.nMode];
    std::cout << "Конвертер успешно подключён [" << ss.str() << ']' << std::endl;
    return true;
}

int main() {
    try {
#ifdef ILG_LOG
#ifdef ILG_LOG_FILE
        // Очищаем лог файл
        std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::trunc);
        file.close();
#endif
        // Включаем лог отладки
        CILG::SetLogCallback(LogCallback);
        CILG::SetLogLevel(ILG_LOG_LEVEL_DEBUG);
#endif

        CILG oILG;
        CConverter oConverter;
        if (!DoConnectToConverter(oILG, oConverter))
            return 0;

        while (true) {
            // Ищем контроллеры
            std::cout << "Поиск контроллеров..." << std::endl;
            oConverter.Scan();
            auto nCount = oConverter.GetControllerCount();
            if (0 == nCount)
                std::cout << "Контроллеры не найдены" << std::endl;
            else {
                std::cout << "Найдено контроллеров: " << nCount << std::endl;
                ilg_controller_info rCI;
                for (size_t i = 0; i < nCount; i++) {
                    oConverter.GetControllerInfo(i, rCI);
                    std::stringstream ss;
                    if (rCI.nModel != ILG_CONTROLLER_MODEL_UNKNOWN)
                        ss << kControllerModelNames[rCI.nModel];
                    if (rCI.nSn != -1)
                        ss << " с/н:" << rCI.nSn;
                    if (rCI.nFwVersion != 0)
                        ss << " прошивка:" << VersionToStr(rCI.nFwVersion);
                    std::cout << i + 1 << ". #" << (uint)rCI.nAddress << ": " << ss.str()
                              << std::endl;
                }
            }
            std::cout << "-----" << std::endl;
            std::cout << "Введите номер команды:" << std::endl;
            std::cout << "1 - Искать снова" << std::endl;
            std::cout << "0 - Выйти из программы" << std::endl;

            int nCommand;
            std::cin >> nCommand;
            std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
            if (std::cin.fail()) {
                std::cin.clear();
                nCommand = -1;
            }

            switch (nCommand) {
            case 0:
                return 0;

            case 1:
                break;

            default:
                std::cout << "Неправильный ввод" << std::endl;
                break;
            }
        }
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
    return 0;
}
