#include <exception>
#include <fstream>   // для std::fstream
#include <iomanip>   // для std::put_time
#include <iostream>  // для std::cout и std::cin

#include "ilg_cpp_helpers.h"
#include "ilguard/ilguard.h"

// #define ILG_LOG  // Раскомментируйте, чтобы включить показ отладочных сообщений
#define ILG_LOG_FILE  // Писать лог в файл

using namespace ilg;

// Глобальные переменные:
uint8_t g_nBankCount = 0;
bool g_fEmergencyMode = false;

#ifdef ILG_LOG
const char kLogLevelChars[] = {'-', 'E', 'W', 'I', 'D'};
const char kLogFileName[] = "ilguard.log";  // Путь к лог файлу

void ILG_CALL LogCallback(ilg_log_level level, const char* pContext, const char* pMessage, void*) {
#if 1  // Запись в файл
    std::fstream file(kLogFileName, std::ios_base::out | std::ios_base::app);
    auto& out = file;
#else  // иначе в консоль
    auto& out = std::cout;
#endif
    auto t = std::time(nullptr);
    auto tmb = std::localtime(&t);
    out << std::put_time(tmb, "%d-%m-%Y %H:%M:%S") << " [" << kLogLevelChars[level] << ' '
        << pContext << "] " << pMessage << std::endl;
}
#endif

// Показывает времена замка контроллера
void ShowLockTimes(CController& oController) {
    uint16_t nOpen, nLet, nMax;
    bool fBigTime;
    for (uint8_t i = 0; i < g_nBankCount; i++) {
        oController.ReadLockTimes(i, nOpen, nLet, nMax, fBigTime);
        std::cout << "Банк " << i << ':' << std::endl;
        std::cout << "Время открытия двери (0.1 с): " << nOpen << std::endl;
        std::cout << "Время контроля закрытой двери (0.1 с): " << nLet << std::endl;
        std::cout << "Время контроля открытой двери (0.1 с): " << nMax << std::endl;
        std::cout << "Большое время: " << fBigTime << std::endl;
        std::cout << "------------" << std::endl;
    }
}

// Устанавливает времена замка контроллера
void SetLockTimes(CController& oController) {
    std::cout << "Введите номер банка, время открытия (0.1 с), время контроля закрытой двери (0.1 "
                 "с), время контроля открытой двери (0.1 с):"
              << std::endl;
    uint nBankN, nOpen, nLet, nMax;
    std::cin >> nBankN >> nOpen >> nLet >> nMax;
    std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
    if (std::cin.fail()) {
        std::cout << "Неправильный ввод" << std::endl;
        return;
    }
    auto fBigTime = (nOpen > 255) || (nLet > 255) || (nMax > 255);
    std::cout << "Запись..." << std::endl;
    auto tStartTime = now();
    oController.WriteLockTimes(static_cast<uint8_t>(nBankN), static_cast<uint16_t>(nOpen),
                               static_cast<uint16_t>(nLet), static_cast<uint16_t>(nMax), fBigTime);
    std::cout << "Записано за " << since(tStartTime).count() << " мс" << std::endl;
}

// Открывает дверь
void OpenDoor(CController& oController, bool fOut) {
    std::cout << "Открытие двери..." << std::endl;
    auto tStartTime = now();
    oController.OpenDoor(fOut);
    std::cout << "Выполнено за " << since(tStartTime).count() << " мс" << std::endl;
}

// Переключает режим аварийного открывания дверей
void ToggleEmergencyMode(CController& oController) {
    std::cout << "Переключение режима..." << std::endl;
    auto fEnable = !g_fEmergencyMode;
    auto tStartTime = now();
    oController.SetEmergencyMode(fEnable);
    g_fEmergencyMode = fEnable;
    std::cout << "Выполнено за " << since(tStartTime).count() << " мс" << std::endl;
}

// Подключается к конвертеру
bool DoConnectToConverter(CILG& oILR, CConverter& oConverter) {
    // Ищем конвертеры
    CConverterSearch oSearch(oILR.GetSearch());
    std::cout << "Поиск конвертеров..." << std::endl;
    oSearch.Scan();
    auto nCount = oSearch.GetConverterCount();
    if (0 == nCount) {
        std::cout << "Конвертер не найден" << std::endl;
        return false;
    }
    std::cout << "Найдено конвертеров: " << nCount << std::endl;

    std::cout << std::endl << "Выберите конвертер:" << std::endl;
    for (size_t i = 0; i < nCount; i++) {
        ilg_converter_info rInfo;
        oSearch.GetConverterInfo(i, rInfo);

        std::stringstream ss;
        if (rInfo.nModel != ILG_CONVERTER_MODEL_UNKNOWN)
            ss << kConverterModelNames[rInfo.nModel];
        if (rInfo.nSn != -1)
            ss << " с/н:" << rInfo.nSn;
        if (rInfo.nFwVersion != 0)
            ss << " прошивка:" << VersionToStr(rInfo.nFwVersion);
        if (rInfo.nFwBuildDate != 0)
            ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
        if (rInfo.nMode != ILG_CONVERTER_MODE_UNKNOWN)
            ss << " режим: " << kConverterModeNames[rInfo.nMode];
        std::cout << 1 + i << ". " << rInfo.pszPortName << " [" << rInfo.pszConnect
                  << "]: " << ss.str() << std::endl;
    }
    std::cout << "0 - Выйти из программы" << std::endl;

    int nCommand;
    std::cin >> nCommand;
    std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
    if (std::cin.fail()) {
        std::cin.clear();
        nCommand = -1;
    }

    if ((nCommand <= 0) || (static_cast<size_t>(nCommand) > nCount))
        return false;

    ilg_converter_info rInfo;
    oSearch.GetConverterInfo(static_cast<size_t>(nCommand - 1), rInfo);

    std::cout << "Подключение к конвертеру [" << kPortTypeNames[rInfo.nPortType] << ": "
              << rInfo.pszPortName << "]..." << std::endl;
    oConverter = oILR.GetConverter(rInfo.nPortType, rInfo.pszPortName);
    ilg_converter_options rOptions;
    oConverter.GetOptions(rOptions);
    rOptions.nConnectModel = rInfo.nModel;
    oConverter.SetOptions(rOptions);
    // Подключаемся к конвертеру
    oConverter.Connect();
    // Получаем информацию о конвертере
    oConverter.GetConverterInfo(rInfo);
    std::stringstream ss;
    if (rInfo.nModel != ILG_CONVERTER_MODEL_UNKNOWN)
        ss << kConverterModelNames[rInfo.nModel];
    if (rInfo.nSn != -1)
        ss << " с/н:" << rInfo.nSn;
    if (rInfo.nFwVersion != 0)
        ss << " прошивка:" << VersionToStr(rInfo.nFwVersion);
    if (rInfo.nFwBuildDate != 0)
        ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
    if (rInfo.nMode != ILG_CONVERTER_MODE_UNKNOWN)
        ss << " режим: " << kConverterModeNames[rInfo.nMode];
    std::cout << "Конвертер успешно подключён [" << ss.str() << ']' << std::endl;
    return true;
}

// Подключается к контроллеру
bool DoConnectToController(CConverter& oConverter, CController& oController) {
    // Поиск контроллеров
    int nCommand;
    while (true) {
        std::cout << "Выберите контроллер:" << std::endl;
        oConverter.Scan();
        auto nCount = oConverter.GetControllerCount();
        for (size_t i = 0; i < nCount; i++) {
            ilg_controller_info rInfo;
            oConverter.GetControllerInfo(i, rInfo);
            std::stringstream ss;
            if (rInfo.nModel != ILG_CONTROLLER_MODEL_UNKNOWN)
                ss << " " << kControllerModelNames[rInfo.nModel];
            if (rInfo.nSn != -1)
                ss << " с/н:" << rInfo.nSn;
            if (rInfo.nFwVersion != 0)
                ss << " прошивка:" << VersionToStr(rInfo.nFwVersion);
            std::cout << i + 1 << ". #" << (uint)rInfo.nAddress << ": " << ss.str() << std::endl;
        }
        if (0 == nCount)
            std::cout << "1 - Искать снова" << std::endl;
        std::cout << "0 - Выйти из программы" << std::endl;

        std::cin >> nCommand;
        std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
        if (std::cin.fail()) {
            std::cin.clear();
            nCommand = -1;
        }

        if (0 == nCommand)
            return false;
        if ((nCommand >= 1) && (static_cast<size_t>(nCommand) <= nCount))
            break;
    }
    ilg_controller_info rInfo;
    oConverter.GetControllerInfo(static_cast<size_t>(nCommand - 1), rInfo);
    oController = oConverter.GetController(rInfo.nModel, rInfo.nSn);
    // Подключаемся к контроллеру
    std::cout << "Подключение к контроллеру [" << kControllerModelNames[rInfo.nModel] << ": "
              << rInfo.nSn << "]..." << std::endl;
    oController.Connect();
    // Получаем информацию о контроллере
    oController.GetControllerInfo(rInfo);
    std::cout << "Контроллер успешно подключён [#" << (uint)rInfo.nAddress << ' '
              << kControllerModelNames[rInfo.nModel] << " с/н:" << rInfo.nSn
              << " прошивка:" << VersionToStr(rInfo.nFwVersion) << ']' << std::endl;
    // Выключаем авто поиск контроллеров (не обязательно)
    oConverter.SetAutoScan(false);
    return true;
}

int main() {
    try {
#ifdef ILG_LOG
#ifdef ILG_LOG_FILE
        // Очищаем лог файл
        std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::trunc);
        file.close();
#endif
        // Включаем лог отладки
        CILG::SetLogCallback(LogCallback);
        CILG::SetLogLevel(ILG_LOG_LEVEL_DEBUG);
#endif

        CILG oILG;

        // Подключаемся к конвертеру
        CConverter oConverter;
        if (!DoConnectToConverter(oILG, oConverter))
            return 0;

        // Подключаемся к контроллеру
        CController oController;
        if (!DoConnectToController(oConverter, oController))
            return 0;

        // Инициализируем глобальные справочные переменные
        ilg_controller_info rInfo;
        oController.GetControllerInfo(rInfo);
        g_nBankCount = (rInfo.nCtrFlags & ILG_CTR_F_TWOBANKS) ? 2 : 1;
        if (rInfo.nCtrFlags & ILG_CTR_F_OLDEMERGOPEN) {
            ilg_rtc_params rParams;
            oController.ReadRtcParams(rParams);
            g_fEmergencyMode = rParams.fEmergencyOpen;
        }

        while (true) {
            std::cout << "-----" << std::endl;
            std::cout << "Введите номер команды:" << std::endl;
            std::cout << "1 - Показать времена замка" << std::endl;
            std::cout << "2 - Установить времена замка..." << std::endl;
            std::cout << "3 - Открыть дверь (вход)" << std::endl;
            std::cout << "4 - Открыть дверь (выход)" << std::endl;
            if (rInfo.nCtrFlags & ILG_CTR_F_OLDEMERGOPEN)
                std::cout << "5 - Переключить режим аварийного открывания дверей: "
                          << std::boolalpha << g_fEmergencyMode << std::endl;
            std::cout << "0 - Выйти из программы" << std::endl;

            int nCommand;
            std::cin >> nCommand;
            std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
            if (std::cin.fail()) {
                std::cin.clear();
                nCommand = -1;
            }

            switch (nCommand) {
            case 0:
                return 0;

            case 1:
                ShowLockTimes(oController);
                break;

            case 2:
                SetLockTimes(oController);
                break;

            case 3:
                OpenDoor(oController, false);
                break;

            case 4:
                OpenDoor(oController, true);
                break;

            case 5:
                ToggleEmergencyMode(oController);
                break;

            default:
                std::cout << "Неправильный ввод" << std::endl;
                break;
            }
        }
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
    return 0;
}
