#include <exception>
#include <fstream>   // для std::fstream
#include <iomanip>   // для std::put_time
#include <iostream>  // для std::cout и std::cin

#include "ilg_cpp_helpers.h"
#include "ilguard/ilguard.h"

// #define ILG_LOG  // Раскомментируйте, чтобы включить показ отладочных сообщений
#define ILG_LOG_FILE  // Писать лог в файл

using namespace ilg;

#ifdef ILG_LOG
const char kLogLevelChars[] = {'-', 'E', 'W', 'I', 'D'};
const char kLogFileName[] = "ilguard.log";  // Путь к лог файлу

void ILG_CALL LogCallback(ilg_log_level level, const char* pContext, const char* pMessage, void*) {
#if 1  // Запись в файл
    std::fstream file(kLogFileName, std::ios_base::out | std::ios_base::app);
    auto& out = file;
#else  // иначе в консоль
    auto& out = std::cout;
#endif
    auto t = std::time(nullptr);
    auto tmb = std::localtime(&t);
    out << std::put_time(tmb, "%d-%m-%Y %H:%M:%S") << " [" << kLogLevelChars[level] << ' '
        << pContext << "] " << pMessage << std::endl;
}
#endif

void ILG_CALL MessageCallback(ilg_converter_msg nMsg, const void* pMsgData, void* pUserData) {
    try {
        auto pConverter = static_cast<ilg::CConverter*>(pUserData);
        switch (nMsg) {
        case ILG_CONVERTER_MSG_CONNECTION_CHANGED:
            {
                auto nStatus = pConverter->GetConnectionStatus();
                switch (nStatus) {
                case ILG_CONNECTION_DISCONNECTED:
                    std::cout << "{!} Конвертер отключён" << std::endl;
                    break;

                case ILG_CONNECTION_CONNECTED:
                    std::cout << "{!} Конвертер подключён" << std::endl;
                    break;

                case ILG_CONNECTION_CONNECTING:
                    std::cout << "{!} Идёт подключение к конвертеру" << std::endl;
                    break;

                default:
                    break;
                }
            }
            break;

        case ILG_CONVERTER_MSG_CONTROLLER_FOUND:
            {
                auto pInfo = static_cast<const ilg_controller_info*>(pMsgData);
                std::cout << "{!} Контроллер найден #" << (uint)pInfo->nAddress << ": "
                          << kControllerModelNames[pInfo->nModel] << " с/н:" << pInfo->nSn
                          << " прошивка:" << VersionToStr(pInfo->nFwVersion) << std::endl;
            }
            break;

        case ILG_CONVERTER_MSG_CONTROLLER_LOST:
            {
                auto pInfo = static_cast<const ilg_controller_info*>(pMsgData);

                std::cout << "{!} Контроллер потерян #" << (uint)pInfo->nAddress << ": "
                          << kControllerModelNames[pInfo->nModel] << " с/н:" << pInfo->nSn
                          << " прошивка:" << VersionToStr(pInfo->nFwVersion) << std::endl;
            }
            break;

        default:
            break;
        }
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
}

bool DoConnectToConverter(CILG& oILR, CConverter& oConverter) {
    // Ищем конвертеры
    CConverterSearch oSearch(oILR.GetSearch());
    std::cout << "Поиск конвертеров..." << std::endl;
    oSearch.Scan();
    auto nCount = oSearch.GetConverterCount();
    if (0 == nCount) {
        std::cout << "Конвертер не найден" << std::endl;
        return false;
    }
    std::cout << "Найдено конвертеров: " << nCount << std::endl;

    std::cout << std::endl << "Выберите конвертер:" << std::endl;
    for (size_t i = 0; i < nCount; i++) {
        ilg_converter_info rInfo;
        oSearch.GetConverterInfo(i, rInfo);

        std::stringstream ss;
        if (rInfo.nModel != ILG_CONVERTER_MODEL_UNKNOWN)
            ss << kConverterModelNames[rInfo.nModel];
        if (rInfo.nSn != -1)
            ss << " с/н:" << rInfo.nSn;
        if (rInfo.nFwVersion != 0)
            ss << " прошивка:" << VersionToStr(rInfo.nFwVersion);
        if (rInfo.nFwBuildDate != 0)
            ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
        if (rInfo.nMode != ILG_CONVERTER_MODE_UNKNOWN)
            ss << " режим: " << kConverterModeNames[rInfo.nMode];
        std::cout << 1 + i << ". " << rInfo.pszPortName << " [" << rInfo.pszConnect
                  << "]: " << ss.str() << std::endl;
    }
    std::cout << "0 - Выйти из программы" << std::endl;

    int nCommand;
    std::cin >> nCommand;
    if (std::cin.fail()) {
        std::cin.clear();
        nCommand = -1;
    }
    std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');

    if ((nCommand <= 0) || (static_cast<size_t>(nCommand) > nCount))
        return false;

    ilg_converter_info rInfo;
    oSearch.GetConverterInfo(static_cast<size_t>(nCommand - 1), rInfo);

    std::cout << "Подключение к конвертеру [" << kPortTypeNames[rInfo.nPortType] << ": "
              << rInfo.pszPortName << "]..." << std::endl;
    oConverter = oILR.GetConverter(rInfo.nPortType, rInfo.pszPortName);
    ilg_converter_options rOptions;
    oConverter.GetOptions(rOptions);
    rOptions.nConnectModel = rInfo.nModel;
    oConverter.SetOptions(rOptions);
    // Выключаем авто поиск контроллеров (не обязательно)
    oConverter.SetAutoScan(false);
    // Подключаемся к конвертеру
    oConverter.Connect();
    // Получаем информацию о конвертере
    oConverter.GetConverterInfo(rInfo);
    std::stringstream ss;
    if (rInfo.nModel != ILG_CONVERTER_MODEL_UNKNOWN)
        ss << kConverterModelNames[rInfo.nModel];
    if (rInfo.nSn != -1)
        ss << " с/н:" << rInfo.nSn;
    if (rInfo.nFwVersion != 0)
        ss << " прошивка:" << VersionToStr(rInfo.nFwVersion);
    if (rInfo.nFwBuildDate != 0)
        ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
    if (rInfo.nMode != ILG_CONVERTER_MODE_UNKNOWN)
        ss << " режим: " << kConverterModeNames[rInfo.nMode];
    std::cout << "Конвертер успешно подключён [" << ss.str() << ']' << std::endl;
    return true;
}

int main() {
    try {
#ifdef ILG_LOG
#ifdef ILG_LOG_FILE
        // Очищаем лог файл
        std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::trunc);
        file.close();
#endif
        // Включаем лог отладки
        CILG::SetLogCallback(LogCallback);
        CILG::SetLogLevel(ILG_LOG_LEVEL_DEBUG);
#endif
        CILG oILG;
        CConverter oConverter;
        if (!DoConnectToConverter(oILG, oConverter))
            return 0;

        // Подписываемся на уведомления от конвертера
        oConverter.SetMessageCallback(MessageCallback, &oConverter);

        // Ищем контроллеры
        std::cout << "Поиск контроллеров..." << std::endl;
        oConverter.Scan();
        auto nCount = oConverter.GetControllerCount();
        if (0 == nCount)
            std::cout << "Контроллеры не найдены" << std::endl;
        else {
            std::cout << "Найдено контроллеров: " << nCount << std::endl;
            ilg_controller_info rInfo;
            for (size_t i = 0; i < nCount; i++) {
                oConverter.GetControllerInfo(i, rInfo);
                std::cout << i + 1 << ". #" << (uint)rInfo.nAddress << ": "
                          << kControllerModelNames[rInfo.nModel] << " с/н:" << rInfo.nSn
                          << " прошивка:" << VersionToStr(rInfo.nFwVersion) << std::endl;
            }
        }
        std::cout << "-----" << std::endl;

        // Включает авто поиск контроллеров (нет необходимости, т.к. включён по умолчанию)
        oConverter.SetAutoScan(true);
        // Ожидаем нахождение/потерю контроллера
        std::cout << "Ожидание нахождения/потери контроллера..." << std::endl;

        std::cout << "Нажмите <Enter> для выхода..." << std::endl;
        std::cin.get();
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
    return 0;
}
