#include "CKeyDialog.h"

#include <iostream>

#include "CAppSettings.h"

#define HAS_STYLE_PROVIDER_ADD_PROVIDER_FOR_DISPLAY GTKMM_CHECK_VERSION(4, 9, 1)

CKeyDialog::CKeyDialog() :
    m_fOut(false),
    m_nIdx(0),
    m_fAccept(false),
    m_nErrorEditIdx(-1),
    m_nSaveAccess(0x7f),
    m_NumberFrame("Номер ключа"),
    m_FlagsBox(Gtk::Orientation::VERTICAL),
    m_ShortCheckButton("Короткий номер"),
    m_FuncCheckButton("Функциональный"),
    m_DualCheckButton("Двойной"),
    m_TypeFrame("Тип"),
    m_TypeBox(Gtk::Orientation::VERTICAL),
    m_TypeRadioButtons(Gtk::CheckButton("Простой"), Gtk::CheckButton("Блокирующий"),
                       Gtk::CheckButton("Мастер")),
    m_AccessFrame("Доступ"),
    m_AccessBox(Gtk::Orientation::VERTICAL),
    m_EverCheckButton("Всегда"),
    m_NeverCheckButton("Никогда"),
    m_ScheduleCheckButton("По расписанию"),
    m_CancelButton("Отмена"),
    m_OkButton("OK") {
    memset(&m_rKey, 0, sizeof(m_rKey));
    memset(m_aTzs, 0, sizeof(m_aTzs));
    set_destroy_with_parent(true);
    set_title("Параметры ключа");
    set_default_size(620, 350);

    set_child(m_Grid);

    m_Grid.set_margin(5);
    m_Grid.set_row_spacing(10);
    m_Grid.set_column_spacing(10);
    m_Grid.attach(m_NumberFrame, 0, 0);
    m_Grid.attach(m_FlagsBox, 1, 0);
    m_Grid.attach(m_TypeFrame, 2, 0);
    m_Grid.attach(m_AccessFrame, 0, 1, 3);
    m_Grid.attach_next_to(m_BottomBox, Gtk::PositionType::BOTTOM, 3);

    m_NumberFrame.set_child(m_NumberGrid);
    m_NumberGrid.set_margin(5);
    m_NumberGrid.attach(m_NumberEntry, 0, 0);
    m_NumberGrid.attach(m_FormatDropDown, 1, 0);
    m_NumberGrid.attach(m_Number2Entry, 0, 1);
    m_NumberGrid.attach(m_Format2DropDown, 1, 1);

    m_refFormatStringList = Gtk::StringList::create({});
    m_refFormatStringList->append("Dallas");
    m_refFormatStringList->append("Em-Marine");
    m_refFormatStringList->append("Код клавиатуры");
    m_FormatDropDown.set_model(m_refFormatStringList);
    m_FormatDropDown.property_selected().signal_changed().connect(
        sigc::mem_fun(*this, &CKeyDialog::on_dropdown_format_selected_changed));
    m_Format2DropDown.set_model(m_refFormatStringList);
    m_Format2DropDown.property_selected().signal_changed().connect(
        sigc::mem_fun(*this, &CKeyDialog::on_dropdown_format2_selected_changed));

    m_FlagsBox.append(m_ShortCheckButton);
    m_FlagsBox.append(m_FuncCheckButton);
    m_FlagsBox.append(m_DualCheckButton);

    m_DualCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CKeyDialog::on_button_dual_toggled));

    m_TypeFrame.set_child(m_TypeBox);
    m_TypeBox.set_margin(5);
    for (size_t i = 0; i < std::size(m_TypeRadioButtons); i++) {
        m_TypeBox.append(m_TypeRadioButtons[i]);
        m_TypeRadioButtons[i].set_expand();
        m_TypeRadioButtons[i].signal_toggled().connect(
            sigc::bind(sigc::mem_fun(*this, &CKeyDialog::on_button_type_clicked), i));
    }
    m_TypeRadioButtons[1].set_group(m_TypeRadioButtons[0]);
    m_TypeRadioButtons[2].set_group(m_TypeRadioButtons[0]);

    m_AccessFrame.set_child(m_AccessBox);
    m_AccessBox.set_margin(5);
    m_AccessBox.append(m_EverCheckButton);
    m_AccessBox.append(m_NeverCheckButton);
    m_AccessBox.append(m_ScheduleCheckButton);
    m_AccessBox.append(m_TzListBox);

    m_NeverCheckButton.set_group(m_EverCheckButton);
    m_ScheduleCheckButton.set_group(m_EverCheckButton);
    m_EverCheckButton.signal_toggled().connect(
        sigc::bind(sigc::mem_fun(*this, &CKeyDialog::on_button_access_clicked), 0xff));
    m_NeverCheckButton.signal_toggled().connect(
        sigc::bind(sigc::mem_fun(*this, &CKeyDialog::on_button_access_clicked), 0));
    m_ScheduleCheckButton.signal_toggled().connect(
        sigc::bind(sigc::mem_fun(*this, &CKeyDialog::on_button_access_clicked), 0x7f));
    m_TzListBox.set_margin(5);
    m_TzListBox.set_margin_start(17);
    for (size_t i = 0; i < std::size(m_aTzRows); i++)
        m_TzListBox.append(m_aTzRows[i]);

    // Load extra CSS file.
    m_refCssProvider = Gtk::CssProvider::create();
#if HAS_STYLE_PROVIDER_ADD_PROVIDER_FOR_DISPLAY
    Gtk::StyleProvider::add_provider_for_display(get_display(), m_refCssProvider,
                                                 GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
#else
    Gtk::StyleContext::add_provider_for_display(get_display(), m_refCssProvider,
                                                GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
#endif
    std::string sCSS =
        ".tz {"
        "  font: 100% Monospace, Courier New, Courier; "
        "}";

    m_refCssProvider->load_from_data(sCSS);
    m_TzListBox.add_css_class("tz");

    m_BottomBox.set_margin(5);
    m_BottomBox.set_spacing(5);
    m_BottomBox.append(m_CancelButton);
    m_BottomBox.append(m_OkButton);
    m_BottomBox.set_valign(Gtk::Align::END);
    m_BottomBox.set_vexpand(true);
    m_BottomBox.set_halign(Gtk::Align::END);

    m_CancelButton.signal_clicked().connect(sigc::mem_fun(*this, &CKeyDialog::on_button_cancel));
    m_OkButton.signal_clicked().connect(sigc::mem_fun(*this, &CKeyDialog::on_button_ok));

    m_nEditFormat = g_AppSet.m_nEditFormat;
    m_nEditFormat2 = g_AppSet.m_nEditFormat2;
    signal_destroy().connect(sigc::mem_fun(*this, &CKeyDialog::on_destroy));
}

CKeyDialog::~CKeyDialog() {
}

void CKeyDialog::Init(bool fOut, size_t nIdx, uint32_t nCtrFlags) {
    m_fOut = fOut;
    m_nIdx = nIdx;
    m_nCtrFlags = nCtrFlags;
    m_fAccept = false;

    std::stringstream ss;
    ss << "Ключ " << (nIdx + 1);
    set_title(ss.str());

    InitScheduleListBox();
    UpdateCtrlData(false);
}

void CKeyDialog::on_button_cancel() {
    set_visible(false);
}

void CKeyDialog::on_button_ok() {
    UpdateCtrlData(true);
    if (m_nErrorEditIdx != -1) {
        switch (m_nErrorEditIdx) {
        case 0:
            m_NumberEntry.grab_focus();
            ShowMessage("Некорректный номер ключа.", Gtk::MessageType::ERROR);
            break;

        case 1:
            m_Number2Entry.grab_focus();
            ShowMessage("Некорректный номер ключа 2.", Gtk::MessageType::ERROR);
            break;
        }
    }

    m_fAccept = true;
    set_visible(false);
}

void CKeyDialog::on_button_type_clicked(int nType) {
    m_rKey.nType = (ilg_key_type)nType;
}

void CKeyDialog::on_button_access_clicked(int n) {
    if ((m_rKey.nAccess != 0) && (m_rKey.nAccess != 0xff))
        m_nSaveAccess = m_rKey.nAccess;
    if (n != 0x7f)
        m_rKey.nAccess = static_cast<uint8_t>(n);
    else
        m_rKey.nAccess = m_nSaveAccess;
    UpdateAccessCtrlState();
}

void CKeyDialog::on_destroy() {
    g_AppSet.SetEditFormat(m_nEditFormat);
    g_AppSet.SetEditFormat2(m_nEditFormat2);
}

void CKeyDialog::on_dialog_response(int) {
    m_refDialog = nullptr;
}

void CKeyDialog::on_dropdown_format_selected_changed() {
    auto nIdx = m_FormatDropDown.get_selected();
    if (GTK_INVALID_LIST_POSITION == nIdx)
        return;
    auto nNewFormat = static_cast<edit_key_format>(EDIT_KEY_FORMAT_DALLAS + nIdx);
    if (m_nEditFormat == nNewFormat)
        return;
    UpdateKeyNumData(true);
    m_nEditFormat = nNewFormat;
    UpdateKeyNumData(false);
}

void CKeyDialog::on_dropdown_format2_selected_changed() {
    auto nIdx = m_Format2DropDown.get_selected();
    if (GTK_INVALID_LIST_POSITION == nIdx)
        return;
    auto nNewFormat = static_cast<edit_key_format>(EDIT_KEY_FORMAT_DALLAS + nIdx);
    if (m_nEditFormat2 == nNewFormat)
        return;
    UpdateKeyNumData(true);
    m_nEditFormat2 = nNewFormat;
    UpdateKeyNumData(false);
}

void CKeyDialog::on_button_dual_toggled() {
    auto fDual = m_DualCheckButton.get_active();
    if (fDual != !(m_rKey.nFlags & ILG_KEY_F_DUAL))
        return;
    UpdateKeyNumData(true);
    if (fDual)
        m_rKey.nFlags |= ILG_KEY_F_DUAL;
    else
        m_rKey.nFlags &= ~ILG_KEY_F_DUAL;
    UpdateKeyNumData(false);
}

void CKeyDialog::InitScheduleListBox() {
    const char32_t kDowTitles[] = U"ПВСЧПСВ";
    char32_t szDows[8] = {0};
    ilg_time_zone* pTz;
    std::wstring_convert<std::codecvt_utf8<char32_t>, char32_t> cv;
    if (m_nCtrFlags & ILG_CTR_F_TWOBANKS) {
        for (size_t i = 0; i < std::size(m_aTzRows); i++) {
            pTz = &m_aTzs[0][i];
            std::stringstream ss;
            ss << std::dec << (i + 1) << " - Вход: ";
            for (size_t j = 0; j < 7; j++)
                szDows[j] = GET_BIT(pTz->nDaysOfWeek, j) ? kDowTitles[j] : U'-';
            ss << cv.to_bytes(szDows) << "  " << std::setfill('0') << std::setw(2)
               << (uint)pTz->nBeginHour << ':' << std::setw(2) << (uint)pTz->nBeginMinute << '-'
               << std::setw(2) << (uint)pTz->nEndHour << ':' << std::setw(2)
               << (uint)pTz->nEndMinute;

            pTz = &m_aTzs[1][i];
            ss << " Выход: ";
            for (size_t k = 0; k < 7; k++)
                szDows[k] = GET_BIT(pTz->nDaysOfWeek, k) ? kDowTitles[k] : U'-';
            ss << cv.to_bytes(szDows) << "  " << std::setfill('0') << std::setw(2)
               << (uint)pTz->nBeginHour << ':' << std::setw(2) << (uint)pTz->nBeginMinute << '-'
               << std::setw(2) << (uint)pTz->nEndHour << ':' << std::setw(2)
               << (uint)pTz->nEndMinute;

            m_aTzRows[i].set_text(ss.str());
        }
    }
    else {
        for (size_t i = 0; i < std::size(m_aTzRows); i++) {
            pTz = &m_aTzs[0][i];
            std::stringstream ss;
            ss << std::dec << (i + 1) << " - ";
            for (size_t j = 0; j < 7; j++)
                szDows[j] = GET_BIT(pTz->nDaysOfWeek, j) ? kDowTitles[j] : U'-';
            ss << cv.to_bytes(szDows) << "  " << std::setfill('0') << std::setw(2)
               << (uint)pTz->nBeginHour << ':' << std::setw(2) << (uint)pTz->nBeginMinute << '-'
               << std::setw(2) << (uint)pTz->nEndHour << ':' << std::setw(2)
               << (uint)pTz->nEndMinute;
            m_aTzRows[i].set_text(ss.str());
        }
    }
}

void CKeyDialog::UpdateCtrlData(bool fSave) {
    UpdateKeyNumData(fSave);
    if (fSave) {
        auto nIdx = m_FormatDropDown.get_selected();
        if (nIdx != GTK_INVALID_LIST_POSITION)
            m_nEditFormat = (edit_key_format)nIdx;
        nIdx = m_Format2DropDown.get_selected();
        if (nIdx != GTK_INVALID_LIST_POSITION)
            m_nEditFormat2 = (edit_key_format)nIdx;
        if (m_ShortCheckButton.get_active())
            m_rKey.nFlags |= ILG_KEY_F_SHORT;
        else
            m_rKey.nFlags &= ~ILG_KEY_F_SHORT;
        if (m_FuncCheckButton.get_active())
            m_rKey.nFlags |= ILG_KEY_F_FUNC;
        else
            m_rKey.nFlags &= ~ILG_KEY_F_FUNC;
        if (m_DualCheckButton.get_active())
            m_rKey.nFlags |= ILG_KEY_F_DUAL;
        else
            m_rKey.nFlags &= ~ILG_KEY_F_DUAL;

        for (size_t i = 0; i < std::size(m_TypeRadioButtons); i++)
            if (m_TypeRadioButtons[i].get_active()) {
                m_rKey.nType = static_cast<ilg_key_type>(ILG_KEY_NORMAL + i);
                break;
            }
        if (m_EverCheckButton.get_active())
            m_rKey.nAccess = 0xff;
        else if (m_NeverCheckButton.get_active())
            m_rKey.nAccess = 0;
        else {
            uint nTzs = 0;
            for (size_t i = 0; i < std::size(m_aTzRows); i++)
                if (m_aTzRows[i].get_active())
                    SET_BIT(nTzs, i, true);
            m_rKey.nAccess = nTzs;
        }
    }
    else {
        m_FormatDropDown.set_selected((guint)m_nEditFormat);
        m_Format2DropDown.set_selected((guint)m_nEditFormat2);
        m_TypeRadioButtons[m_rKey.nType].set_active();
        auto f = !(m_nCtrFlags & ILG_CTR_F_X2);
        m_TypeFrame.set_sensitive(f);
        m_ShortCheckButton.set_active((m_rKey.nFlags & ILG_KEY_F_SHORT) != 0);
        // m_ShortCheckButton.set_sensitive(f);
        m_FuncCheckButton.set_active((m_rKey.nFlags & ILG_KEY_F_FUNC) != 0);
        // m_FuncCheckButton.set_sensitive(f);
        m_DualCheckButton.set_active((m_rKey.nFlags & ILG_KEY_F_DUAL) != 0);
        // m_DualCheckButton.set_sensitive(f);
        m_FlagsBox.set_sensitive(f);
        switch (m_rKey.nAccess) {
        case 0xff:
            m_EverCheckButton.set_active();
            break;

        case 0:
            m_NeverCheckButton.set_active();
            break;

        default:
            m_ScheduleCheckButton.set_active();
            for (size_t i = 0; i < std::size(m_aTzRows); i++)
                m_aTzRows[i].set_active(GET_BIT(m_rKey.nAccess, i));
            break;
        }
        UpdateAccessCtrlState();
    }
}

void CKeyDialog::UpdateKeyNumData(bool fSave) {
    if (fSave) {
        m_nErrorEditIdx = -1;
        if (m_rKey.nFlags & ILG_KEY_F_DUAL) {
            if (!TryStrToKeyNum(m_Number2Entry.get_text(), m_rKey.rNumber, 1))
                m_nErrorEditIdx = 1;
            if (!TryStrToKeyNum(m_NumberEntry.get_text(), m_rKey.rNumber, 0))
                m_nErrorEditIdx = 0;
        }
        else if (!TryStrToKeyNum(m_NumberEntry.get_text(), m_rKey.rNumber))
            m_nErrorEditIdx = 0;
    }
    else {
        if (m_rKey.nFlags & ILG_KEY_F_DUAL) {
            switch (m_nEditFormat) {
            case EDIT_KEY_FORMAT_DALLAS:
                m_NumberEntry.set_text(
                    Glib::ustring::sprintf("%.2X %.2X %.2X", m_rKey.rNumber.dual_dallas.aDallas1[2],
                                           m_rKey.rNumber.dual_dallas.aDallas1[1],
                                           m_rKey.rNumber.dual_dallas.aDallas1[0]));
                break;

            case EDIT_KEY_FORMAT_EM_MARINE:
                m_NumberEntry.set_text(
                    Glib::ustring::sprintf("%.3u,%.5u", m_rKey.rNumber.dual_em_marine.nSeries1,
                                           m_rKey.rNumber.dual_em_marine.nNumber1));
                break;

            case EDIT_KEY_FORMAT_KEYPAD:
                m_NumberEntry.set_text(ilg::KeybCodeToStr(m_rKey.rNumber, 0));
                break;
            }
            switch (m_nEditFormat2) {
            case EDIT_KEY_FORMAT_DALLAS:
                m_Number2Entry.set_text(
                    Glib::ustring::sprintf("%.2X %.2X %.2X", m_rKey.rNumber.dual_dallas.aDallas2[2],
                                           m_rKey.rNumber.dual_dallas.aDallas2[1],
                                           m_rKey.rNumber.dual_dallas.aDallas2[0]));
                break;

            case EDIT_KEY_FORMAT_EM_MARINE:
                m_Number2Entry.set_text(
                    Glib::ustring::sprintf("%.3u,%.5u", m_rKey.rNumber.dual_em_marine.nSeries2,
                                           m_rKey.rNumber.dual_em_marine.nNumber2));
                break;

            case EDIT_KEY_FORMAT_KEYPAD:
                m_Number2Entry.set_text(ilg::KeybCodeToStr(m_rKey.rNumber, 1));
                break;
            }
            m_Number2Entry.set_visible();
            m_Format2DropDown.set_visible();
        }
        else {
            switch (m_nEditFormat) {
            case EDIT_KEY_FORMAT_DALLAS:
                m_NumberEntry.set_text(Glib::ustring::sprintf(
                    "%.2X %.2X %.2X %.2X %.2X %.2X", m_rKey.rNumber.aDallas[5],
                    m_rKey.rNumber.aDallas[4], m_rKey.rNumber.aDallas[3], m_rKey.rNumber.aDallas[2],
                    m_rKey.rNumber.aDallas[1], m_rKey.rNumber.aDallas[0]));
                break;

            case EDIT_KEY_FORMAT_EM_MARINE:
                m_NumberEntry.set_text(Glib::ustring::sprintf(
                    "[%.2X %.2X %.2X] %.3u,%.5u", m_rKey.rNumber.em_marine.aFacility[2],
                    m_rKey.rNumber.em_marine.aFacility[1], m_rKey.rNumber.em_marine.aFacility[0],
                    m_rKey.rNumber.em_marine.nSeries, m_rKey.rNumber.em_marine.nNumber));
                break;

            case EDIT_KEY_FORMAT_KEYPAD:
                m_NumberEntry.set_text(ilg::KeybCodeToStr(m_rKey.rNumber));
                break;
            }
            m_Number2Entry.set_visible(false);
            m_Format2DropDown.set_visible(false);
        }
    }
}

void CKeyDialog::UpdateAccessCtrlState() {
    auto f = (m_rKey.nAccess != 0) && (m_rKey.nAccess != 0xff);
    m_TzListBox.set_sensitive(f);
}

bool CKeyDialog::TryStrToKeyNum(const std::string& s, ilg_key_number& rNumber, int nDualIdx) {
    bool fRes = false;

    switch (nDualIdx) {
    case -1:
        {
            switch (m_nEditFormat) {
            case EDIT_KEY_FORMAT_DALLAS:
                {
                    uint a[6];
                    fRes = (sscanf(s.c_str(), "%X %X %X %X %X %X", &a[5], &a[4], &a[3], &a[2],
                                   &a[1], &a[0]) == 6);
                    if (fRes)
                        for (size_t i = 0; i < std::size(rNumber.aDallas); i++)
                            rNumber.aDallas[i] = static_cast<uint8_t>(a[i]);
                    break;
                }

            case EDIT_KEY_FORMAT_EM_MARINE:
                {
                    uint nSeries, nNumber;
                    uint aFacility[3];
                    fRes = (sscanf(s.c_str(), "[%x %x %x] %u,%u", &aFacility[2], &aFacility[1],
                                   &aFacility[0], &nSeries, &nNumber) == 5);
                    if (fRes) {
                        rNumber.em_marine.nSeries = static_cast<uint8_t>(nSeries);
                        rNumber.em_marine.nNumber = static_cast<uint16_t>(nNumber);
                        rNumber.em_marine.aFacility[0] = static_cast<uint8_t>(aFacility[0]);
                        rNumber.em_marine.aFacility[1] = static_cast<uint8_t>(aFacility[1]);
                        rNumber.em_marine.aFacility[2] = static_cast<uint8_t>(aFacility[2]);
                    }
                    break;
                }

            case EDIT_KEY_FORMAT_KEYPAD:
                fRes = ilg::TryParseKeybCodeStr(s.c_str(), &rNumber);
                break;
            }
            break;
        }

    case 0:
        {
            switch (m_nEditFormat) {
            case EDIT_KEY_FORMAT_DALLAS:
                {
                    uint a[3];
                    fRes = (sscanf(s.c_str(), "%X %X %X", &a[2], &a[1], &a[0]) == 3);
                    if (fRes)
                        for (size_t i = 0; i < 2; i++)
                            rNumber.dual_dallas.aDallas1[i] = a[i];
                    break;
                }

            case EDIT_KEY_FORMAT_EM_MARINE:
                {
                    uint nSeries, nNumber;
                    fRes = (sscanf(s.c_str(), "%u,%u", &nSeries, &nNumber) == 2);
                    if (fRes) {
                        rNumber.em_marine.nSeries = static_cast<uint8_t>(nSeries);
                        rNumber.em_marine.nNumber = static_cast<uint16_t>(nNumber);
                    }
                    break;
                }

            case EDIT_KEY_FORMAT_KEYPAD:
                {
                    ilg::CKeyNumber num;
                    fRes = ilg::TryParseKeybCodeStr(s.c_str(), &num);
                    if (fRes)
                        memcpy(rNumber.dual_dallas.aDallas1, num.aDallas,
                               sizeof(rNumber.dual_dallas.aDallas1));
                    break;
                }
            }
            break;
        }

    case 1:
        {
            switch (m_nEditFormat2) {
            case EDIT_KEY_FORMAT_DALLAS:
                {
                    uint a[3];
                    fRes = (sscanf(s.c_str(), "%X %X %X", &a[2], &a[1], &a[0]) == 3);
                    if (fRes)
                        for (size_t i = 0; i < 3; i++)
                            rNumber.dual_dallas.aDallas2[i] = a[i];
                    break;
                }

            case EDIT_KEY_FORMAT_EM_MARINE:
                {
                    uint nSeries, nNumber;
                    fRes = (sscanf(s.c_str(), "%u,%u", &nSeries, &nNumber) == 2);
                    if (fRes) {
                        rNumber.dual_em_marine.nSeries2 = static_cast<uint8_t>(nSeries);
                        rNumber.dual_em_marine.nNumber2 = static_cast<uint16_t>(nNumber);
                    }
                    break;
                }

            case EDIT_KEY_FORMAT_KEYPAD:
                {
                    ilg_key_number rNum;
                    fRes = ilg::TryParseKeybCodeStr(s.c_str(), &rNum);
                    if (fRes)
                        memcpy(rNumber.dual_dallas.aDallas2, rNum.aDallas,
                               sizeof(rNumber.dual_dallas.aDallas2));
                    break;
                }
            }
            break;
        }
    }
    return fRes;
}

void CKeyDialog::ShowMessage(const std::string& sMessage, Gtk::MessageType nType) {
    if (nullptr == m_refDialog)
        m_refDialog.reset(new Gtk::MessageDialog("Демо", false, nType));
    m_refDialog->set_transient_for(*this);
    m_refDialog->set_message(sMessage);
    m_refDialog->show();
    m_refDialog->signal_response().connect(
        sigc::bind(sigc::mem_fun(*this, &CKeyDialog::on_dialog_response)));
}
