#include "CControllerDialog.h"

#include <filesystem>
#include <fstream>
#include <iostream>

#include "utils.h"

CControllerDialog::CControllerDialog() :
    m_VBox(Gtk::Orientation::VERTICAL),
    m_StatusFrame("Состояние подключения"),
    m_ConnectButton("Подключиться"),
    m_DisconnectButton("Отключиться"),
    m_SetFwButton("Установить прошивку..."),
    m_LinesFrame("Строки"),
    m_Box1(Gtk::Orientation::VERTICAL),
    m_AddressFrame("Сетевой адрес"),
    m_refAddressAdjustment(Gtk::Adjustment::create(0, 0, 254.0)),
    m_AddressSpinButton(m_refAddressAdjustment),
    m_SetAddressButton("Установить"),
    m_ParamsFrame("Параметры"),
    m_ParamsBox(Gtk::Orientation::VERTICAL),
    m_LockButton("Замок..."),
    m_ScheduleButton("Расписание..."),
    m_KeysButton("Ключи..."),
    m_FireButton("Пожар..."),
    m_SecurityButton("Охрана..."),
    m_EcButton("Электро контроль..."),
    m_ReadCfgButton("Читать конфигурацию..."),
    m_WriteCfgButton("Записать конфигурацию..."),
    m_ModeFrame("Режим контроллера"),
    m_ReadModeButton("Читать"),
    m_WriteModeButton("Записать"),
    m_Box2(Gtk::Orientation::VERTICAL),
    m_ControlFrame("Управление"),
    m_ControlBox(Gtk::Orientation::VERTICAL),
    m_OpenButton("Открыть дверь"),
    m_OpenExitButton("Открыть дверь (выход)"),
    m_FireOnButton("Вкл. режим Пожар"),
    m_FireOffButton("Выкл. режим Пожар"),
    m_SecurityOnButton("Вкл. режим Охрана"),
    m_SecurityOffButton("Выкл. режим Охрана"),
    m_AlarmOnButton("Вкл. Тревогу"),
    m_AlarmOffButton("Выкл. Тревогу"),
    m_PowerOnButton("Вкл. Питание"),
    m_PowerOffButton("Выкл. Питание"),
    m_EventsFrame("События"),
    m_EventsBox(Gtk::Orientation::VERTICAL),
    m_EventsButton("События..."),
    m_MonitorButton("Монитор..."),
    m_ExtAskFrame("Ext Ask"),
    m_ExtAskBox(Gtk::Orientation::VERTICAL) {
    set_destroy_with_parent(true);

    set_title("Контроллер");
    set_default_size(900, 577);

    m_VBox.set_margin(5);
    set_child(m_VBox);

    m_TopBox.set_margin(5);
    m_TopBox.set_expand(false);
    m_BottomBox.set_margin(5);
    m_BottomBox.set_expand();

    m_VBox.append(m_TopBox);
    m_VBox.append(m_BottomBox);

    m_StatusFrame.set_margin(10);
    m_StatusFrame.set_expand();
    m_TopBox.append(m_StatusFrame);
    m_StatusBox.set_margin(5);
    m_StatusBox.set_spacing(5);
    m_StatusBox.set_expand();
    m_StatusFrame.set_child(m_StatusBox);

    m_StatusBox.append(m_StatusLabel);
    m_StatusLabel.set_hexpand(true);
    m_StatusLabel.set_halign(Gtk::Align::START);
    m_ConnectButton.set_halign(Gtk::Align::END);
    m_ConnectButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_connect));
    m_StatusBox.append(m_ConnectButton);
    m_DisconnectButton.set_halign(Gtk::Align::END);
    m_DisconnectButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_disconnect));
    m_StatusBox.append(m_DisconnectButton);
    m_SetFwButton.set_hexpand(true);
    m_SetFwButton.set_halign(Gtk::Align::END);
    m_SetFwButton.set_valign(Gtk::Align::START);
    m_SetFwButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_setfw));
    m_TopBox.append(m_SetFwButton);

    m_BottomBox.append(m_LinesFrame);
    // Add the TreeView, inside a ScrolledWindow, with the button underneath:
    m_LinesScrolledWindow.set_child(m_LinesTextView);

    // Only show the scrollbars when they are necessary:
    m_LinesScrolledWindow.set_policy(Gtk::PolicyType::AUTOMATIC, Gtk::PolicyType::AUTOMATIC);
    m_LinesScrolledWindow.set_expand();
    m_LinesFrame.set_child(m_LinesScrolledWindow);
    m_refLinesBuffer = Gtk::TextBuffer::create();
    m_LinesTextView.set_buffer(m_refLinesBuffer);
    m_LinesTextView.set_editable(false);
    m_LinesFrame.set_size_request(200);

    m_BottomBox.append(m_Box1);
    m_Box1.set_margin(5);
    m_Box1.set_expand();
    m_Box1.append(m_AddressFrame);
    m_Box1.append(m_ParamsFrame);
    m_Box1.append(m_ModeFrame);

    m_AddressFrame.set_child(m_AddressBox);
    m_AddressBox.set_margin(5);
    m_AddressBox.append(m_AddressSpinButton);
    m_AddressBox.set_expand(false);
    m_AddressSpinButton.set_wrap();
    m_AddressSpinButton.set_expand();
    m_AddressSpinButton.set_size_request(55, -1);
    m_AddressBox.append(m_SetAddressButton);
    m_SetAddressButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_setaddress));

    m_ParamsFrame.set_child(m_ParamsBox);
    m_ParamsBox.set_margin(5);
    m_ParamsBox.set_spacing(5);
    m_ParamsBox.append(m_LockButton);
    m_ParamsBox.append(m_ScheduleButton);
    m_ParamsBox.append(m_KeysButton);
    m_ParamsBox.append(m_FireButton);
    m_ParamsBox.append(m_SecurityButton);
    m_ParamsBox.append(m_EcButton);
    m_ParamsBox.append(m_ReadCfgButton);
    m_ParamsBox.append(m_WriteCfgButton);

    m_LockButton.signal_clicked().connect(sigc::mem_fun(*this, &CControllerDialog::on_button_lock));
    m_ScheduleButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_schedule));
    m_KeysButton.signal_clicked().connect(sigc::mem_fun(*this, &CControllerDialog::on_button_keys));
    m_FireButton.signal_clicked().connect(sigc::mem_fun(*this, &CControllerDialog::on_button_fire));
    m_SecurityButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_security));
    m_EcButton.signal_clicked().connect(sigc::mem_fun(*this, &CControllerDialog::on_button_ec));
    m_ReadCfgButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_read_cfg));
    m_WriteCfgButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_write_cfg));

    m_ModeFrame.set_child(m_ModeGrid);
    /* set the spacing to 10 on x and 10 on y */
    m_ModeGrid.set_row_spacing(5);
    m_ModeGrid.set_column_spacing(5);
    m_ModeGrid.set_margin(5);
    m_ModeGrid.set_expand();

    m_ModeGrid.attach(m_oModeDropDown, 0, 0, 2);
    m_ModeGrid.attach_next_to(m_ReadModeButton, Gtk::PositionType::BOTTOM);
    m_ModeGrid.attach(m_WriteModeButton, 1, 1);
    m_refModeStringList = Gtk::StringList::create({});
    for (size_t i = ILG_CONTROLLER_MODE_NORMAL; i < std::size(kControllerModeNames); i++)
        m_refModeStringList->append(kControllerModeNames[i]);
    m_oModeDropDown.set_model(m_refModeStringList);
    m_oModeDropDown.set_hexpand();

    m_ReadModeButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_read_mode));
    m_WriteModeButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_write_mode));

    m_BottomBox.append(m_Box2);
    m_Box2.set_margin(5);
    m_Box2.set_spacing(5);
    m_Box2.set_expand();

    m_Box2.append(m_ControlFrame);
    m_Box2.append(m_EventsFrame);

    m_ControlFrame.set_child(m_ControlBox);
    m_ControlBox.set_margin(5);
    m_ControlBox.set_spacing(5);
    m_ControlBox.append(m_OpenButton);
    m_ControlBox.append(m_OpenExitButton);
    m_ControlBox.append(m_FireOnButton);
    m_ControlBox.append(m_FireOffButton);
    m_ControlBox.append(m_SecurityOnButton);
    m_ControlBox.append(m_SecurityOffButton);
    m_ControlBox.append(m_AlarmOnButton);
    m_ControlBox.append(m_AlarmOffButton);
    m_ControlBox.append(m_PowerOnButton);
    m_ControlBox.append(m_PowerOffButton);

    m_OpenButton.signal_clicked().connect(sigc::mem_fun(*this, &CControllerDialog::on_button_open));
    m_OpenExitButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_open_exit));
    m_FireOnButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_fire_on));
    m_FireOffButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_fire_off));
    m_SecurityOnButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_security_on));
    m_SecurityOffButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_security_off));
    m_AlarmOnButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_alarm_on));
    m_AlarmOffButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_alarm_off));
    m_PowerOnButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_power_on));
    m_PowerOffButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_power_off));

    m_EventsFrame.set_child(m_EventsBox);
    m_EventsBox.set_margin(5);
    m_EventsBox.set_spacing(5);
    m_EventsBox.append(m_EventsButton);
    m_EventsBox.append(m_MonitorButton);

    m_EventsButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_events));
    m_MonitorButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_button_monitor));

    m_BottomBox.append(m_ExtAskFrame);
    m_ExtAskFrame.set_child(m_ExtAskBox);
    m_ExtAskBox.set_margin(5);
    m_ExtAskBox.set_expand();
    for (size_t i = 0; i < std::size(m_aExtAskLabels); i++) {
        m_ExtAskBox.append(m_aExtAskLabels[i]);
        m_ExtAskBox.append(m_aExtAskEntrys[i]);
        m_aExtAskEntrys[i].set_editable(false);
        m_aExtAskEntrys[i].set_alignment(Gtk::Align::CENTER);
    }
    m_aExtAskLabels[0].set_label("Дверь:");
    m_aExtAskLabels[1].set_label("Кнопка:");
    m_aExtAskLabels[2].set_label("Блокировка двери:");
    m_aExtAskLabels[3].set_label("Охрана:");
    m_aExtAskLabels[4].set_label("Режим контроллера:");
    m_aExtAskLabels[5].set_label("Пожар:");
    m_aExtAskLabels[6].set_label("Аварийное открывание:");
    m_aExtAskLabels[7].set_label("Питание 12 В");
    m_aExtAskLabels[8].set_label("Питание CR2032:");
    m_aExtAskLabels[9].set_label("Температура:");

    m_oDisp.connect(sigc::mem_fun(*this, &CControllerDialog::on_ilg));
}

CControllerDialog::~CControllerDialog() {
}

void CControllerDialog::SetController(ilg::CController& oController) {
    if (oController) {
        m_oController = std::move(oController);
        UpdateConnectionStatus();
        UpdateControllerInfo();
        m_oController.EnableMessageQueue();
        m_oController.SetMessageCallback(&on_ilg_message, this);

        ilg_controller_info rInfo;
        m_oController.GetControllerInfo(rInfo);
        if (rInfo.nCtrFlags & ILG_CTR_F_MODES) {
            auto nMode = m_oController.ReadMode();
            m_oModeDropDown.set_selected((guint)(nMode - ILG_CONTROLLER_MODE_NORMAL));
        }

        m_oCommand = m_oController.Begin_ReadLines();
        m_refLinesBuffer->set_text("Загрузка...");
        if (rInfo.nCtrFlags & ILG_CTR_F_EXTASK)
            EnableExtAskPolling();
    }
    else {
        m_oCommand = nullptr;
        m_oController.SetMessageCallback(nullptr);
        m_oController = std::move(oController);
    }
}

void CControllerDialog::on_button_connect() {
    m_oController.Connect();
}

void CControllerDialog::on_button_disconnect() {
    m_oController.Disconnect();
}

void CControllerDialog::on_button_setfw() {
    auto dialog = new Gtk::FileChooserDialog("Выбор файла прошивки контроллера");
    dialog->set_transient_for(*this);
    dialog->set_modal();
    dialog->add_button("Отмена", Gtk::ResponseType::CANCEL);
    dialog->add_button("OK", Gtk::ResponseType::ACCEPT);

    auto fw_filter = Gtk::FileFilter::create();
    fw_filter->add_pattern("*.rom");
    fw_filter->set_name("Прошивки");
    dialog->add_filter(fw_filter);
    dialog->signal_response().connect(
        sigc::bind(sigc::mem_fun(*this, &CControllerDialog::on_fw_dialog_response), dialog));
    dialog->show();
}

void CControllerDialog::on_fw_dialog_response(int response_id, Gtk::FileChooserDialog* dialog) {
    std::unique_ptr<Gtk::FileChooserDialog> oDialogPtr(dialog);
    if (response_id != Gtk::ResponseType::ACCEPT)
        return;

    auto file = dialog->get_file();
    std::filesystem::path sPath = file->get_path();

    if (std::filesystem::exists(sPath)) {
        std::ifstream file(sPath, std::ios::binary | std::ios::ate);
        auto size = file.tellg();
        file.seekg(0, std::ios::beg);
        std::vector<uint8_t> oData(size);
        file.read((char*)oData.data(), size);

        uint32_t nProgSize = (oData.size() < 4) ? 0 : *(uint32_t*)&oData[oData.size() - 4];
        if ((nProgSize + 4) > oData.size()) {
            MessageDialog("Файл не содержит прошивки!");
            return;
        }
        uint32_t N = *(uint32_t*)&oData[oData.size() - nProgSize - 4];
        if (N > (oData.size() - nProgSize)) {
            MessageDialog("Файл не содержит прошивки!");
            return;
        }
        auto p = &oData[oData.size() - nProgSize - N];
        N = *(uint32_t*)p;
        std::stringstream ss;
        ss << "Данный файл содержит следующее описание:" << std::endl
           << std::endl
           << "Тип контроллера " << std::hex << std::showbase << std::setw(2) << (N & 0xFF)
           << ". Версия прошивки " << std::dec << ((N >> 8) & 0xFF) << '.' << ((N >> 16) & 0xFF)
           << std::endl;
        auto nLines = (N >> 24);
        ss << "Серийные номера контроллеров:";
        p += 8;
        uint16_t w, h;
        for (size_t i = 0; i < 4; i++) {
            w = *(uint16_t*)p;
            p += 2;
            h = *(uint16_t*)p;
            p += 2;
            if ((0 == h) || (0xffff == w))
                continue;
            if (i != 0)
                ss << ',';
            else
                ss << ' ';
            if ((1 == w) && (0xffff == h))
                ss << "Все";
            else if (h != w)
                ss << '(' << w << '-' << h << ')';
            else
                ss << w;
        }
        ss << std::endl << std::endl;
        std::string s;
        for (size_t i = 0; i < nLines; i++) {
            w = *(uint16_t*)p;
            p += 2;
            if (w > 130)
                break;
            s.assign((const char*)p, (const char*)p + w);
            p += w;
            ss << s << std::endl;
        }
        ss << std::endl << "Вы уверены что хотите установить эту прошивку?";

        m_oFwData = std::move(oData);
        MessageDialog(ss.str(), Gtk::MessageType::QUESTION, Gtk::ButtonsType::OK_CANCEL,
                      sigc::mem_fun(*this, &CControllerDialog::on_setfw_dialog_response));
    }
}

void CControllerDialog::on_setfw_dialog_response(int response_id) {
    m_refDialog = nullptr;
    if (response_id != GTK_RESPONSE_OK) {
        m_oFwData.clear();
        return;
    }
    m_oCommand = m_oController.Begin_SetFirmware(m_oFwData.data(), m_oFwData.size());
    m_oFwData.clear();
    m_refProgressDialog.reset(new CProgressDialog());
    m_refProgressDialog->set_transient_for(*this);
    m_refProgressDialog->set_modal();
    m_refProgressDialog->set_hide_on_close();
    m_refProgressDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_progress_dialog_hide));
    m_refProgressDialog->signal_cancel().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_progress_dialog_cancel));
    m_refProgressDialog->signal_progress().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_progress_dialog_progress));
    m_refProgressDialog->set_visible(true);
}

void CControllerDialog::on_progress_dialog_cancel() {
    m_oCommand.Cancel();
    m_refProgressDialog = nullptr;
    m_oCommand = nullptr;
}

bool CControllerDialog::on_progress_dialog_progress(size_t& nCurrent, size_t& nTotal) {
    m_oCommand.GetProgress(nCurrent, nTotal);
    auto nStatus = m_oCommand.GetStatus();
    if (nStatus != ILG_E_PENDING) {
        m_refProgressDialog = nullptr;
        m_oCommand = nullptr;
        if (ILG_FAILED(nStatus))
            MessageDialog(ilg_get_error_text(nStatus), Gtk::MessageType::ERROR);
        return false;
    }
    return true;
}

void CControllerDialog::on_progress_dialog_hide() {
    m_refProgressDialog = nullptr;
    m_oCommand = nullptr;
}

void CControllerDialog::on_button_setaddress() {
    auto nNewAddress = m_AddressSpinButton.get_value_as_int();
    if ((nNewAddress < 0) || (nNewAddress > 254)) {
        MessageDialog("Некорректный адрес (нужно в диапазоне 0..254).");
        return;
    }
    m_oController.SetNetworkAddress(static_cast<uint8_t>(nNewAddress));
}

void CControllerDialog::on_button_lock() {
    if (nullptr == m_refLockDialog)
        m_refLockDialog.reset(new CLockDialog());
    m_refLockDialog->set_transient_for(*this);
    m_refLockDialog->set_modal();
    m_refLockDialog->set_hide_on_close();
    m_refLockDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_lock_dialog_hide));

    m_refLockDialog->Init(m_oController);
    m_refLockDialog->set_visible(true);
}

void CControllerDialog::on_lock_dialog_hide() {
    m_refLockDialog = nullptr;
}

void CControllerDialog::on_button_schedule() {
    if (nullptr == m_refScheduleDialog)
        m_refScheduleDialog.reset(new CScheduleDialog());
    m_refScheduleDialog->set_transient_for(*this);
    m_refScheduleDialog->set_modal();
    m_refScheduleDialog->set_hide_on_close();
    m_refScheduleDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_schedule_dialog_hide));

    EnableExtAskPolling(false);

    m_refScheduleDialog->Init(m_oController);
    m_refScheduleDialog->set_visible(true);
}

void CControllerDialog::on_schedule_dialog_hide() {
    m_refScheduleDialog = nullptr;
    if (m_ExtAskFrame.get_visible())
        EnableExtAskPolling();
}

void CControllerDialog::on_button_keys() {
    if (nullptr == m_refKeysDialog)
        m_refKeysDialog.reset(new CKeysDialog());
    m_refKeysDialog->set_transient_for(*this);
    m_refKeysDialog->set_modal();
    m_refKeysDialog->set_hide_on_close();
    m_refKeysDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_keys_dialog_hide));

    EnableExtAskPolling(false);

    m_refKeysDialog->Init(m_oController);
    m_refKeysDialog->set_visible(true);
}

void CControllerDialog::on_keys_dialog_hide() {
    m_refKeysDialog = nullptr;
    if (m_ExtAskFrame.get_visible())
        EnableExtAskPolling();
}

void CControllerDialog::on_button_fire() {
    if (nullptr == m_refFireDialog)
        m_refFireDialog.reset(new CFireDialog());
    m_refFireDialog->set_transient_for(*this);
    m_refFireDialog->set_modal();
    m_refFireDialog->set_hide_on_close();
    m_refFireDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_fire_dialog_hide));

    EnableExtAskPolling(false);

    m_refFireDialog->Init(m_oController);
    m_refFireDialog->set_visible(true);
}

void CControllerDialog::on_fire_dialog_hide() {
    m_refFireDialog = nullptr;
    if (m_ExtAskFrame.get_visible())
        EnableExtAskPolling();
}

void CControllerDialog::on_button_security() {
    if (nullptr == m_refSecurityDialog)
        m_refSecurityDialog.reset(new CSecurityDialog());
    m_refSecurityDialog->set_transient_for(*this);
    m_refSecurityDialog->set_modal();
    m_refSecurityDialog->set_hide_on_close();
    m_refSecurityDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_security_dialog_hide));

    EnableExtAskPolling(false);

    m_refSecurityDialog->Init(m_oController);
    m_refSecurityDialog->set_visible(true);
}

void CControllerDialog::on_security_dialog_hide() {
    m_refSecurityDialog = nullptr;
    if (m_ExtAskFrame.get_visible())
        EnableExtAskPolling();
}

void CControllerDialog::on_button_ec() {
    if (nullptr == m_refEcDialog)
        m_refEcDialog.reset(new CEcDialog());
    m_refEcDialog->set_transient_for(*this);
    m_refEcDialog->set_modal();
    m_refEcDialog->set_hide_on_close();
    m_refEcDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_ec_dialog_hide));

    EnableExtAskPolling(false);

    m_refEcDialog->Init(m_oController);
    m_refEcDialog->set_visible(true);
}

void CControllerDialog::on_ec_dialog_hide() {
    m_refEcDialog = nullptr;
    if (m_ExtAskFrame.get_visible())
        EnableExtAskPolling();
}

void CControllerDialog::on_button_read_cfg() {
    std::vector<uint8_t> oData;
    m_oController.ReadConfiguration(oData);
    m_oCfgData = std::move(oData);

    auto dialog = new Gtk::FileChooserDialog("Укажите имя файла для сохранения",
                                             Gtk::FileChooser::Action::SAVE);
    dialog->set_transient_for(*this);
    dialog->set_modal();
    dialog->add_button("Отмена", Gtk::ResponseType::CANCEL);
    dialog->add_button("OK", Gtk::ResponseType::ACCEPT);

    ilg_controller_info rInfo;
    m_oController.GetControllerInfo(rInfo);
    auto sExt = GetCtrConfigExtFromModel(rInfo.nModel);
    auto cfg_filter = Gtk::FileFilter::create();
    cfg_filter->add_pattern('*' + sExt);
    cfg_filter->set_name("Конфигурации");
    dialog->add_filter(cfg_filter);

    dialog->set_current_name(Glib::ustring::format(rInfo.nSn, sExt));
    dialog->signal_response().connect(
        sigc::bind(sigc::mem_fun(*this, &CControllerDialog::on_save_cfg_dialog_response), dialog));
    dialog->show();
}

void CControllerDialog::on_save_cfg_dialog_response(int response_id,
                                                    Gtk::FileChooserDialog* dialog) {
    std::unique_ptr<Gtk::FileChooserDialog> oDialogPtr(dialog);
    if (response_id != Gtk::ResponseType::ACCEPT) {
        m_oCfgData.clear();
        return;
    }

    auto file = dialog->get_file();
    std::filesystem::path sPath = file->get_path();

    ilg_controller_info rInfo;
    m_oController.GetControllerInfo(rInfo);
    if (!sPath.has_extension())
        sPath.replace_extension(GetCtrConfigExtFromModel(rInfo.nModel));

    std::stringstream ss;
    ss.imbue(std::locale("C"));
    ss << std::hex << std::setfill('0');
    for (auto b : m_oCfgData)
        ss << std::setw(2) << (uint)b;
    auto sHex = ss.str();
    for (size_t i = 21; i >= 2; i--)
        replaceAll(sHex, std::string(i, '0'), std::string(1, (char)(65 + 4 + i)));

    std::ofstream outFile;
    outFile.open(sPath);
    outFile.imbue(std::locale("C"));
    outFile << "[CONFIG]" << std::endl
            << "Model=" << ilg::kControllerModelNames[rInfo.nModel] << std::endl
            << "[IMAGE]" << std::endl
            << "BIN=" << sHex << std::endl;
}

void CControllerDialog::on_button_write_cfg() {
    auto dialog = new Gtk::FileChooserDialog("Выберите файл конфигурации");
    dialog->set_transient_for(*this);
    dialog->set_modal();
    dialog->add_button("Отмена", Gtk::ResponseType::CANCEL);
    dialog->add_button("OK", Gtk::ResponseType::ACCEPT);

    ilg_controller_info rInfo;
    m_oController.GetControllerInfo(rInfo);
    auto sExt = GetCtrConfigExtFromModel(rInfo.nModel);
    auto cfg_filter = Gtk::FileFilter::create();
    cfg_filter->add_pattern('*' + sExt);
    cfg_filter->set_name("Конфигурации");
    dialog->add_filter(cfg_filter);

    dialog->set_current_name(Glib::ustring::format("%1%2", rInfo.nSn, sExt.c_str()));
    dialog->signal_response().connect(
        sigc::bind(sigc::mem_fun(*this, &CControllerDialog::on_load_cfg_dialog_response), dialog));
    dialog->show();
}

void CControllerDialog::on_button_read_mode() {
    auto nMode = m_oController.ReadMode();
    m_oModeDropDown.set_selected((guint)(nMode - ILG_CONTROLLER_MODE_NORMAL));
}

void CControllerDialog::on_load_cfg_dialog_response(int response_id,
                                                    Gtk::FileChooserDialog* dialog) {
    std::unique_ptr<Gtk::FileChooserDialog> oDialogPtr(dialog);
    if (response_id != Gtk::ResponseType::ACCEPT)
        return;

    auto file = dialog->get_file();
    std::filesystem::path sPath = file->get_path();
    if (!std::filesystem::exists(sPath))
        return;
    std::vector<uint8_t> oData;
    std::ifstream inFile(sPath);
    std::string sLine;
    const char* pName;
    const char* pValue;
    while (std::getline(inFile, sLine)) {
        if (sLine.empty() || (sLine.front() == '['))
            continue;
        if (sLine.back() == '\r')
            sLine.pop_back();
        auto nPos = sLine.find('=');
        if (nPos != sLine.npos) {
            sLine[nPos] = '\0';
            pName = sLine.c_str();
            pValue = pName + nPos + 1;
            if (strcmp(pName, "BIN") == 0) {
                std::string sHex = pValue;
                for (size_t i = 21; i >= 2; i--)
                    replaceAll(sHex, std::string(1, (char)(65 + 4 + i)), std::string(i, '0'));

                oData = decodeHex(sHex);
                m_oController.WriteConfiguration(oData.data(), oData.size());
                break;
            }
        }
    }
}

void CControllerDialog::on_button_write_mode() {
    auto nIdx = m_oModeDropDown.get_selected();
    if (GTK_INVALID_LIST_POSITION == nIdx)
        return;
    auto nMode = (ilg_controller_mode)(ILG_CONTROLLER_MODE_NORMAL + nIdx);
    m_oController.WriteMode(nMode);
}

void CControllerDialog::on_button_open() {
    m_oController.OpenDoor(false);
}

void CControllerDialog::on_button_open_exit() {
    m_oController.OpenDoor(true);
}

void CControllerDialog::on_button_fire_on() {
    m_oController.SetFireMode(true);
}

void CControllerDialog::on_button_fire_off() {
    m_oController.SetFireMode(false);
}

void CControllerDialog::on_button_security_on() {
    m_oController.SetSecurityMode(true);
}

void CControllerDialog::on_button_security_off() {
    m_oController.SetSecurityMode(false);
}

void CControllerDialog::on_button_alarm_on() {
    m_oController.SetAlarmMode(true);
}

void CControllerDialog::on_button_alarm_off() {
    m_oController.SetAlarmMode(false);
}

void CControllerDialog::on_button_power_on() {
    m_oController.SetEcPower(true);
}

void CControllerDialog::on_button_power_off() {
    m_oController.SetEcPower(false);
}

void CControllerDialog::on_button_events() {
    if (nullptr == m_refEventsDialog)
        m_refEventsDialog.reset(new CEventsDialog());
    m_refEventsDialog->set_transient_for(*this);
    m_refEventsDialog->set_modal();
    m_refEventsDialog->set_hide_on_close();
    m_refEventsDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_events_dialog_hide));

    EnableExtAskPolling(false);

    m_refEventsDialog->Init(m_oController);
    m_refEventsDialog->set_visible(true);
}

void CControllerDialog::on_events_dialog_hide() {
    m_refEventsDialog = nullptr;
    if (m_ExtAskFrame.get_visible())
        EnableExtAskPolling();
}

void CControllerDialog::on_button_monitor() {
    if (nullptr == m_refMonitorDialog)
        m_refMonitorDialog.reset(new CMonitorDialog());
    m_refMonitorDialog->set_transient_for(*this);
    m_refMonitorDialog->set_modal();
    m_refMonitorDialog->set_hide_on_close();
    m_refMonitorDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CControllerDialog::on_monitor_dialog_hide));

    EnableExtAskPolling(false);

    m_refMonitorDialog->Init(m_oController);
    m_refMonitorDialog->set_visible(true);
}

void CControllerDialog::on_monitor_dialog_hide() {
    m_refMonitorDialog = nullptr;
    if (m_ExtAskFrame.get_visible())
        EnableExtAskPolling();
}

void ILG_CALL CControllerDialog::on_ilg_message(ilg_controller_msg nMsg, const void* pMsgData,
                                                void* pUserData) {
    auto p = (CControllerDialog*)pUserData;
    p->m_oDisp.emit();
}

void CControllerDialog::on_ilg() {
    ilg_controller_msg nMsg;
    const void* pMsgData;
    while (m_oController.GetMessage(nMsg, pMsgData)) {
        switch (nMsg) {
        case ILG_CONTROLLER_MSG_COMMAND_FINISH:
            {
                std::string sLines;
                m_oController.End_ReadLines(m_oCommand, sLines);
                m_oCommand = nullptr;
                m_refLinesBuffer->set_text(sLines);
                break;
            }

        case ILG_CONTROLLER_MSG_CONNECTION_CHANGED:
            UpdateConnectionStatus();
            if (m_oController.GetConnectionStatus() == ILG_CONNECTION_CONNECTED)
                UpdateControllerInfo();
            break;

        case ILG_CONTROLLER_MSG_ADDRESS_CHANGED:
            UpdateControllerInfo();
            break;

        case ILG_CONTROLLER_MSG_EXTASK_CHANGED:
            UpdateExtAskParams();
            break;
        }
    }
}

void CControllerDialog::UpdateConnectionStatus() {
    auto nStatus = m_oController.GetConnectionStatus();
    switch (nStatus) {
    case ILG_CONNECTION_CONNECTED:
        m_StatusLabel.set_text("Подключён");
        break;

    case ILG_CONNECTION_DISCONNECTED:
        m_StatusLabel.set_text("Отключён");
        break;

    case ILG_CONNECTION_CONNECTING:
        m_StatusLabel.set_text("Подключение...");
        break;
    }
}

void CControllerDialog::UpdateControllerInfo() {
    ilg_controller_info rInfo;
    m_oController.GetControllerInfo(rInfo);
    std::stringstream ss;
    ss << "Контроллер (#" << (uint)rInfo.nAddress
       << "): " << ilg::kControllerModelNames[rInfo.nModel];
    if (rInfo.nSn != -1)
        ss << " с/н:" << rInfo.nSn;
    if (rInfo.nFwVersion != 0)
        ss << " прошивка:" << ilg::VersionToStr(rInfo.nFwVersion);
    if (rInfo.nPassPoint != ILG_PASS_POINT_UNKNOWN)
        ss << ' ' << kPassPointTypeNames[rInfo.nPassPoint];
    set_title(ss.str());

    m_AddressSpinButton.set_value(rInfo.nAddress);
    ilg_converter_info rCvtInfo;
    m_oController.GetConverterInfo(rCvtInfo);
    m_SetAddressButton.set_sensitive(ILG_CONVERTER_MODE_NORMAL == rCvtInfo.nMode);

    auto f = (rInfo.nCtrFlags & ILG_CTR_F_EXTASK) != 0;
    m_FireButton.set_visible(f);
    m_FireOnButton.set_visible(f);
    m_FireOffButton.set_visible(f);
    m_SecurityButton.set_visible(f);
    m_SecurityOnButton.set_visible(f);
    m_SecurityOffButton.set_visible(f);
    m_AlarmOnButton.set_visible(f);
    m_AlarmOffButton.set_visible(f);
    m_ExtAskFrame.set_visible(f);

    f = (rInfo.nCtrFlags & ILG_CTR_F_EC) != 0;
    m_EcButton.set_visible(f);
    m_PowerOnButton.set_visible(f);
    m_PowerOffButton.set_visible(f);

    f = (rInfo.nCtrFlags & ILG_CTR_F_CONFIG) != 0;
    m_ReadCfgButton.set_visible(f);
    m_WriteCfgButton.set_visible(f);

    f = (rInfo.nCtrFlags & ILG_CTR_F_MODES) != 0;
    m_ModeFrame.set_visible(f);
}

void CControllerDialog::MessageDialog(const std::string& sMessage, Gtk::MessageType nType,
                                      Gtk::ButtonsType nButtons,
                                      Gio::ActionMap::ActivateWithIntParameterSlot&& slot) {
    if (nullptr == m_refDialog)
        m_refDialog.reset(new Gtk::MessageDialog("Демо", false, nType, nButtons, true));
    m_refDialog->set_transient_for(*this);
    m_refDialog->set_message(sMessage);
    m_refDialog->set_modal();
    m_refDialog->signal_response().connect(slot);
    m_refDialog->show();
}

std::string CControllerDialog::GetCtrConfigExtFromModel(ilg_controller_model nModel) {
    switch (nModel) {
    case ILG_CONTROLLER_MODEL_Z5R_NET:
        return ".z5n";

    case ILG_CONTROLLER_MODEL_Z5R_NET_16000:
        return ".z5n16";

    case ILG_CONTROLLER_MODEL_Z5R_WEB:
    case ILG_CONTROLLER_MODEL_Z5R_WEB_BT:
        return ".z5w";

    case ILG_CONTROLLER_MODEL_MATRIX2_WIFI:
    case ILG_CONTROLLER_MODEL_MATRIX6_WIFI:
    case ILG_CONTROLLER_MODEL_Z5R_WIFI:
    case ILG_CONTROLLER_MODEL_MATRIX6_EH_K_NET:
        return ".m2w";

    default:
        return ".ini";
    }
}

void CControllerDialog::EnableExtAskPolling(bool fEnable) {
    ilg_controller_options rOptions;
    m_oController.GetOptions(rOptions);
    rOptions.nExtAskPeriod = (fEnable ? 333 : -1);
    m_oController.SetOptions(rOptions);
}

void CControllerDialog::UpdateExtAskParams() {
    ilg_extask_params rParams;
    m_oController.GetExtAskParams(rParams);
    // Дверь
    m_aExtAskEntrys[0].set_text((rParams.nState & 1) ? "Open" : "Closed");
    // Кнопка
    m_aExtAskEntrys[1].set_text((rParams.nState & 2) ? "Нажата" : "Не нажата");
    // Блокировка двери
    m_aExtAskEntrys[2].set_text((rParams.nState & 4) ? "Активно" : "Неактивно");
    // Охрана
    m_aExtAskEntrys[3].set_text((rParams.nState & 8) ? "Активно" : "Неактивно");
    // Режим контроллера
    m_aExtAskEntrys[4].set_text(kControllerModeNames[rParams.nMode]);
    // Пожар
    m_aExtAskEntrys[5].set_text((rParams.nState & 0x40) ? "Активно" : "Неактивно");
    // Аварийное открывание (старая команда)
    m_aExtAskEntrys[6].set_text((rParams.nState & 0x80) ? "Активно" : "Неактивно");
    // Питание 12 В
    char szBuf[64] = {0};
    double d = rParams.n12V / 10;
    snprintf(szBuf, std::size(szBuf), "%.1f В", d);
    m_aExtAskEntrys[7].set_text(szBuf);
    // Питание CR2032
    const char* kNoneStr = "-----";
    if (rParams.nCR2032 != 0) {
        d = rParams.nCR2032 / 50;
        snprintf(szBuf, std::size(szBuf), "%.2f В", d);
        m_aExtAskEntrys[8].set_text(szBuf);
    }
    else
        m_aExtAskEntrys[8].set_text(kNoneStr);
    // Температура
    if (rParams.nT != 0) {
        snprintf(szBuf, std::size(szBuf), "%d° C", (int)rParams.nT - 64);
        m_aExtAskEntrys[9].set_text(szBuf);
    }
    else
        m_aExtAskEntrys[9].set_text(kNoneStr);
}
