#include <stdio.h>

#include <chrono>
#include <iomanip>  // для std::put_time
#include <iostream>

#include "ilr_cpp_helpers.h"
#include "ilreaders.h"

// #define ILR_LOG  // Раскомментируйте, чтобы включить отладочные сообщения
#define ILR_LOG_FILE  // Писать лог в файл

using namespace ilr;

// Пароль для доступа к Temic, =-1 нет пароля
const int64_t g_TemicPassword = -1;
// Ключ аутентификации Mifare
const CMifareClassicKey g_McKey;
// Ключ аутентификации Mifare Plus
const CMifarePlusKey g_MpKey;
// True, открытая передача, иначе - зашифрованная
const bool g_fOpenText = true;
// Текущая команда чтения карты
ilr::CAsyncCommand g_oReadCmd;
// Время старта чтения карты
std::chrono::steady_clock::time_point g_StartTime;
// Тип карты, который читаем
ilr_rwcard_type_flags g_nCardType = (ilr_rwcard_type_flags)0;
// Размер данных карты
uint32_t g_nCardMemSize;

#ifdef ILR_LOG
const char kLogLevelChars[] = {'-', 'E', 'W', 'I', 'D'};
const char kLogFileName[] = "ilreaders.log";  // Путь к лог файлу

void ILR_CALL LogCallback(ilr_log_level level, const char* pContext, const char* pMessage, void*) {
#ifdef ILR_LOG_FILE  // Запись в файл
    std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::app);
    auto& out = file;
#else  // иначе в консоль
    auto& out = std::cout;
#endif
    auto t = std::time(nullptr);
    auto tmb = std::localtime(&t);
    out << std::put_time(tmb, "%d-%m-%Y %H:%M:%S") << " [" << kLogLevelChars[level] << ' '
        << pContext << "] " << pMessage << std::endl;
}
#endif

void ILR_CALL MessageCallback(ilr_reader_msg nMsg, const void* pMsgData, void* pUserData) {
    try {
        switch (nMsg) {
        case ILR_READER_MSG_CARD_FOUND:  // Карта найдена
            {
                auto pInfo = static_cast<const ilr_card_info*>(pMsgData);
                std::stringstream ss;
                if (pInfo->nMpType != ILR_MF_PLUS_UNKNOWN)
                    ss << " " << kMpTypeNames[pInfo->nMpType];
                if (pInfo->nSL != ILR_MF_PLUS_SL_UNKNOWN)
                    ss << " SL" << static_cast<int>(pInfo->nSL);
                std::cout << "{!} Карта найдена " << kCardTypeNames[pInfo->nType] << ' '
                          << CardUIDToStr(pInfo->nType, pInfo->rUID) << ss.str() << std::endl;

                if (g_oReadCmd) {
                    g_oReadCmd.Cancel();
                    g_oReadCmd.Close();
                    std::cout << "Чтение данных отменено" << std::endl;
                }

                auto pReader = static_cast<CReader*>(pUserData);
                ilr_reader_info rRI;
                pReader->GetReaderInfo(rRI);
                auto nRwCards = pReader->GetSupportedRewrittenCardTypes(rRI.nModel, rRI.nFwVersion);
                g_nCardMemSize = pInfo->nMemSize;

                switch (pInfo->nType) {
                case ILR_CARD_TEMIC:
                    if (nRwCards & ILR_RWCT_F_TEMIC) {
                        g_nCardType = ILR_RWCT_F_TEMIC;
                        g_StartTime = ilr::now();
                        g_oReadCmd = pReader->Begin_ReadTemic(0, 10);
                        std::cout << "Чтение данных..." << std::endl;
                    }
                    break;

                case ILR_CARD_MF_ULTRALIGHT:
                    if (nRwCards & ILR_RWCT_F_MF_ULTRALIGHT) {
                        g_nCardType = ILR_RWCT_F_MF_ULTRALIGHT;
                        g_StartTime = ilr::now();
                        g_oReadCmd = pReader->Begin_ReadMfUltralight(0, 16);
                        std::cout << "Чтение данных..." << std::endl;
                    }
                    break;

                case ILR_CARD_MF_MINI:
                case ILR_CARD_MF_CLASSIC_1K:
                case ILR_CARD_MF_CLASSIC_2K:
                case ILR_CARD_MF_CLASSIC_4K:
                    if (nRwCards & ILR_RWCT_F_MF_CLASSIC) {
                        g_nCardType = ILR_RWCT_F_MF_CLASSIC;
                        g_StartTime = ilr::now();
                        auto nBlockCount = GetNumberOfMfBlocks(pInfo->nMemSize);
                        g_oReadCmd = pReader->Begin_ReadMfClassic(0, nBlockCount);
                        std::cout << "Чтение данных..." << std::endl;
                    }
                    break;

                case ILR_CARD_MF_PLUS_1K:
                case ILR_CARD_MF_PLUS_2K:
                case ILR_CARD_MF_PLUS_4K:
                case ILR_CARD_SMART_MX_MF1K:
                case ILR_CARD_SMART_MX_MF4K:
                    if (ILR_MF_PLUS_SL_SL3 == pInfo->nSL) {
                        if (nRwCards & ILR_RWCT_F_MF_PLUS_SL3) {
                            g_nCardType = ILR_RWCT_F_MF_PLUS_SL3;
                            g_StartTime = ilr::now();
                            auto nBlockCount = GetNumberOfMfBlocks(pInfo->nMemSize);
                            g_oReadCmd = pReader->Begin_ReadMfPlus(0, nBlockCount, g_fOpenText);
                            std::cout << "Чтение данных..." << std::endl;
                        }
                    }
                    else if (ILR_MF_PLUS_SL_SL1 == pInfo->nSL) {
                        if (nRwCards & ILR_RWCT_F_MF_CLASSIC) {
                            g_nCardType = ILR_RWCT_F_MF_CLASSIC;
                            g_StartTime = ilr::now();
                            auto nBlockCount = GetNumberOfMfBlocks(pInfo->nMemSize);
                            g_oReadCmd = pReader->Begin_ReadMfClassic(0, nBlockCount);
                            std::cout << "Чтение данных..." << std::endl;
                        }
                    }
                    break;

                default:
                    std::cout << "Считыватель не поддерживает чтение данных из карты" << std::endl;
                    break;
                }
                // Если команда чтения не стартовала, то возобновляем сканирование карт
                if (!g_oReadCmd)
                    pReader->SetAutoScan(true, false);
            }
            break;

        case ILR_READER_MSG_CARD_LOST:  // Карта потеряна
            {
                auto pInfo = static_cast<const ilr_card_info*>(pMsgData);
                std::stringstream ss;
                if (pInfo->nMpType != ILR_MF_PLUS_UNKNOWN)
                    ss << " " << kMpTypeNames[pInfo->nMpType];
                if (pInfo->nSL != ILR_MF_PLUS_SL_UNKNOWN)
                    ss << " SL" << static_cast<int>(pInfo->nSL);
                std::cout << "{!} Карта потеряна " << kCardTypeNames[pInfo->nType] << ' '
                          << CardUIDToStr(pInfo->nType, pInfo->rUID) << ss.str() << std::endl;

                if (g_oReadCmd) {
                    g_oReadCmd.Cancel();
                    g_oReadCmd.Close();
                    std::cout << "Чтение данных отменено" << std::endl;
                    // Возобновляем сканирование карт
                    auto pReader = (CReader*)pUserData;
                    pReader->SetAutoScan(true, false);
                }
            }
            break;

        case ILR_READER_MSG_COMMAND_FINISH:  // Завершена команда чтения
            if (g_oReadCmd) {
                auto nSpan = since(g_StartTime).count();
                auto pReader = static_cast<CReader*>(pUserData);
                auto nStatus = g_oReadCmd.GetStatus();
                // Возобновляем сканирование карт
                pReader->SetAutoScan(true, false);

                switch (g_nCardType) {
                case ILR_RWCT_F_TEMIC:
                    {
                        const uint* pBlocks = nullptr;  // 10 блоков по 4 байта
                        size_t nCount = 0;
                        pReader->End_ReadTemic(g_oReadCmd, pBlocks, nCount);
                        const uint8_t* p;
                        for (size_t i = 0; i < nCount; i++) {
                            p = (const uint8_t*)&pBlocks[i];
                            std::cout << std::dec << std::setw(2) << i << ". " << std::hex
                                      << std::setw(2) << (uint)p[0] << ' ' << std::setw(2)
                                      << (uint)p[1] << ' ' << std::setw(2) << (uint)p[2] << ' '
                                      << std::setw(2) << (uint)p[3] << std::endl;
                        }
                        std::cout << "Прочитано " << std::dec << nCount << " блоков за " << nSpan
                                  << " мс" << std::endl;
                    }
                    break;

                case ILR_RWCT_F_MF_ULTRALIGHT:
                    {
                        const uint* pPages = nullptr;  // 16 страниц по 4 байта
                        size_t nCount = 0;
                        pReader->End_ReadMfUltralight(g_oReadCmd, pPages, nCount);
                        const uint8_t* p;
                        for (size_t i = 0; i < nCount; i++) {
                            p = (const uint8_t*)&pPages[i];
                            std::cout << std::dec << std::setw(2) << i << ". " << std::hex
                                      << std::setw(2) << (uint)p[0] << ' ' << std::setw(2)
                                      << (uint)p[1] << ' ' << std::setw(2) << (uint)p[2] << ' '
                                      << std::setw(2) << (uint)p[3] << std::endl;
                        }
                        std::cout << "Прочитано " << nCount << " страниц за " << nSpan << " мс"
                                  << std::endl;
                    }
                    break;

                case ILR_RWCT_F_MF_CLASSIC:
                    {
                        const ilr_mf_block_data* pBlocks = nullptr;
                        size_t nCount;
                        pReader->End_ReadMfClassic(g_oReadCmd, pBlocks, nCount);
                        const ilr_mf_block_data* pBlock;
                        for (size_t i = 0; i < nCount; i++) {
                            pBlock = &pBlocks[i];
                            std::cout
                                << std::dec << std::setw(2) << i << ". " << std::hex << std::setw(2)
                                << (uint)pBlock->a[0] << ' ' << std::setw(2) << (uint)pBlock->a[1]
                                << ' ' << std::setw(2) << (uint)pBlock->a[2] << ' ' << std::setw(2)
                                << (uint)pBlock->a[3] << ' ' << std::setw(2) << (uint)pBlock->a[4]
                                << ' ' << std::setw(2) << (uint)pBlock->a[5] << ' ' << std::setw(2)
                                << (uint)pBlock->a[6] << ' ' << std::setw(2) << (uint)pBlock->a[7]
                                << ' ' << std::setw(2) << (uint)pBlock->a[8] << ' ' << std::setw(2)
                                << (uint)pBlock->a[9] << ' ' << std::setw(2) << (uint)pBlock->a[10]
                                << ' ' << std::setw(2) << (uint)pBlock->a[11] << ' ' << std::setw(2)
                                << (uint)pBlock->a[12] << ' ' << std::setw(2) << (uint)pBlock->a[13]
                                << ' ' << std::setw(2) << (uint)pBlock->a[14] << ' ' << std::setw(2)
                                << (uint)pBlock->a[15] << std::endl;
                        }
                        std::cout << "Прочитано " << nCount << " блоков за " << nSpan << " мс"
                                  << std::endl;
                    }
                    break;

                case ILR_RWCT_F_MF_PLUS_SL3:
                    {
                        const ilr_mf_block_data* pBlocks = nullptr;
                        size_t nCount;
                        pReader->End_ReadMfPlus(g_oReadCmd, pBlocks, nCount);
                        const ilr_mf_block_data* pBlock;
                        for (size_t i = 0; i < nCount; i++) {
                            pBlock = &pBlocks[i];
                            std::cout
                                << std::dec << std::setw(2) << i << ". " << std::hex << std::setw(2)
                                << (uint)pBlock->a[0] << ' ' << std::setw(2) << (uint)pBlock->a[1]
                                << ' ' << std::setw(2) << (uint)pBlock->a[2] << ' ' << std::setw(2)
                                << (uint)pBlock->a[3] << ' ' << std::setw(2) << (uint)pBlock->a[4]
                                << ' ' << std::setw(2) << (uint)pBlock->a[5] << ' ' << std::setw(2)
                                << (uint)pBlock->a[6] << ' ' << std::setw(2) << (uint)pBlock->a[7]
                                << ' ' << std::setw(2) << (uint)pBlock->a[8] << ' ' << std::setw(2)
                                << (uint)pBlock->a[9] << ' ' << std::setw(2) << (uint)pBlock->a[10]
                                << ' ' << std::setw(2) << (uint)pBlock->a[11] << ' ' << std::setw(2)
                                << (uint)pBlock->a[12] << ' ' << std::setw(2) << (uint)pBlock->a[13]
                                << ' ' << std::setw(2) << (uint)pBlock->a[14] << ' ' << std::setw(2)
                                << (uint)pBlock->a[15] << std::endl;
                        }
                        std::cout << "Прочитано " << nCount << " блоков за " << nSpan << " мс"
                                  << std::endl;
                    }
                    break;

                default:
                    std::cout << "Неожиданный тип карты!" << std::endl;
                    break;
                }
                if (ILR_SUCCEEDED(nStatus))
                    std::cout << "Успех" << std::endl;
                else
                    std::cout << "Ошибка: " << ilr_get_error_text(nStatus) << std::endl;
                g_oReadCmd.Close();
            }
            break;

        case ILR_READER_MSG_CONNECTION_CHANGED:  // Изменилось состояние подключения считывателя
            {
                auto pReader = static_cast<CReader*>(pUserData);
                auto nStatus = pReader->GetConnectionStatus();
                switch (nStatus) {
                case ILR_CONNECTION_CONNECTED:
                    std::cout << "{!} Считыватель подключён" << std::endl;
                    break;

                case ILR_CONNECTION_DISCONNECTED:
                    std::cout << "{!} Считыватель отключён" << std::endl;
                default:
                    break;
                }
            }
            break;
        }
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
}

int main() {
    try {
#ifdef ILR_LOG
#ifdef ILR_LOG_FILE
        // Очищаем лог файл
        std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::trunc);
        file.close();
#endif
        CILR::SetLogCallback(LogCallback);
        CILR::SetLogLevel(ILR_LOG_LEVEL_DEBUG);
#endif

        CILR oILR;

        // Ищем считыватель
        CReader oReader;
        {
            std::cout << "Поиск считывателя..." << std::endl;
            CReaderSearch oSearch(oILR.GetSearch());
#if 0
            // Выбираем типы считывателя для поиска
            ilr_search_options rOptions;
            oSearch.GetOptions(rOptions);
            rOptions.nReaderTypes |= ILR_READER_CCID;
            oSearch.SetOptions(rOptions);
            // Включаем поиск конвертеров в режиме "Клиент"
            const uint16_t aListenPorts[] = {25000};
            oSearch.SetListenPorts(aListenPorts, std::size(aListenPorts));
#endif
            // Ищем
            oSearch.Scan();
            auto nCount = oSearch.GetReaderCount();
            bool fFound = false;
            ilr_reader_info rInfo;
            for (size_t i = 0; i < nCount; i++) {
                oSearch.GetReaderInfo(i, rInfo);
                // Если порт занят, то пропускаем
                if (*rInfo.pszConnect != '\0')
                    continue;
                fFound = true;
                break;
            }

            if (!fFound) {
                std::cout << "Считыватель не найден" << std::endl;
                return 0;
            }

            oReader = oILR.GetReader(rInfo.nPortType, rInfo.pszPortName);
            // Устанавливает ключи аутентификации
            oReader.LoadTemicPassword(g_TemicPassword);
            oReader.LoadMfCKey(g_McKey);
            oReader.LoadMfPKey(g_MpKey);
            // Разрешаем сканировать Temic
            oReader.SetScanTemic(true);
            // Подключаемся к считывателю
            std::cout << "Подключение к считывателю [" << kPortTypeNames[rInfo.nPortType] << ": "
                      << rInfo.pszPortName << "]..." << std::endl;
            oReader.Connect();
        }

        // Получаем информацию о считывателе
        ilr_reader_info rInfo;
        oReader.GetReaderInfo(rInfo);
        std::stringstream ss;
        if (rInfo.nModel != ILR_READER_MODEL_UNKNOWN)
            ss << kReaderModelNames[rInfo.nModel];
        if (rInfo.nSn != -1)
            ss << " с/н:" << rInfo.nSn;
        if (rInfo.nFwVersion != 0)
            ss << " прошивка:" << ReaderVersionToStr(rInfo.nFwVersion);
        if (rInfo.nFwBuildDate != 0)
            ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
        std::cout << "Считыватель успешно подключён [" << ss.str() << ']' << std::endl;

        // Подписываемся на уведомления о поднесении/удалении карты
        oReader.SetMessageCallback(MessageCallback, &oReader);
        // Включаем задержку при поднесении перезаписываемой карты
        ilr_reader_options rOptions;
        oReader.GetOptions(rOptions);
        rOptions.nHoldCardTypes = (ILR_RWCT_F_TEMIC | ILR_RWCT_F_MF_ULTRALIGHT |
                                   ILR_RWCT_F_MF_CLASSIC | ILR_RWCT_F_MF_PLUS_SL3);
        oReader.SetOptions(rOptions);
        // Ожидаем поднесение/удаление карты
        std::cout << "Ожидание поднесения карты..." << std::endl;

        std::cout << "Нажмите <Enter> для выхода..." << std::endl;
        std::cin.get();

        if (g_oReadCmd) {
            g_oReadCmd.Cancel();
            g_oReadCmd.Close();
            std::cout << "Чтение данных отменено" << std::endl;
        }
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
    return 0;
}
