#include "CTemicDialog.h"

#include "CAppSettings.h"
#include "utils.h"

const char* CTemicDialog::kModulationNames[] = {
    "Direct",        // 0
    "PSK1",          // 1
    "PSK2",          // 2
    "PSK3",          // 3
    "FSK1",          // 4
    "FSK2",          // 5
    "FSK1a",         // 6
    "FSK2a",         // 7
    "Manchester",    // 8
    "",              // 9
    "",              // 10
    "",              // 11
    "",              // 12
    "",              // 13
    "",              // 14
    "",              // 15
    "Biphase('50)",  // 16
    "",              // 17
    "",              // 18
    "",              // 19
    "",              // 20
    "",              // 21
    "",              // 22
    "",              // 23
    "Biphase('57)",  // 24
    "",              // 25
    "",              // 26
    "",              // 27
    "",              // 28
    "",              // 29
    "",              // 30
    ""               // 31
};

CTemicDialog::CTemicDialog() :
    m_nActivePassword(-1),
    m_nLocks(0),
    m_fLockUpdate(false),
    m_nNewCardType(ILR_CARD_EM_MARINE),
    m_nNewHidWiegand(26),
    m_fEntryEmFacilityValid(true),
    m_fEntryHidFacilityValid(true),
    m_VBox(Gtk::Orientation::VERTICAL),
    m_RightBox(Gtk::Orientation::VERTICAL),
    m_SerialButton(Gdk::RGBA(0.65, 0.79, 0.94)),
    m_SerialLabel("Серийный номер"),
    m_PasswordButton(Gdk::RGBA(1.0, 0.75, 0.44)),
    m_PasswordLabel("Пароль"),
    m_UserDataButton(Gdk::RGBA(0.66, 0.94, 0.66)),
    m_UserDataLabel("Данные пользователя"),
    m_ConfigButton(Gdk::RGBA(1.0, 1.0, 0.6)),
    m_ConfigLabel("Конфигурация"),
    m_FormatFrame("Формат данных"),
    m_FormatBox(Gtk::Orientation::VERTICAL),
    m_HexCheckButton("Шестнадцатеричный"),
    m_DecCheckButton("Десятичный"),
    m_OctCheckButton("Восьмеричный"),
    m_BinCheckButton("Двоичный"),
    m_PasswordCheckButton("Пароль"),
    m_PasswordEditButton("..."),
    m_ConfigFrame("Конфигурация"),
    m_ModeLabel("Режим:"),
    m_BitRateLabel("Data Bit Rate:"),
    m_ModulationLabel("Modulation:"),
    m_PskCfLabel("PSK-CF:"),
    m_MaxBlockLabel("Max Block:"),
    m_TestModeCheckButton("Test mode"),
    m_AorModeCheckButton("Answer-On-Request (AOR) Mode"),
    m_StCheckButton("ST-sequence Terminator"),
    m_SstCheckButton("SST-Sequence StartMarker"),
    m_PorDelayCheckButton("POR delay"),
    m_OtpCheckButton("OTP"),
    m_FastWriteCheckButton("Fast write"),
    m_InverseDataCheckButton("Inverse data"),
    m_ConfigFlagsBox(Gtk::Orientation::VERTICAL),
    m_EmulationFrame("Эмуляция"),
    m_EmulationBox(Gtk::Orientation::VERTICAL),
    m_CurrentEmuFrame("Текущая"),
    m_CurrentEmuLabel("-----"),
    m_NewEmuFrame("Новая"),
    m_NewEmuBox(Gtk::Orientation::VERTICAL),
    m_EmMarineCheckButton("Em-Marine"),
    m_HidCheckButton("HID"),
    m_EmFacilityLabel("Код производителя"),
    m_EmSeriesLabel("Серия"),
    m_refEmSeriesAdjustment(Gtk::Adjustment::create(0, 0, 255.0)),
    m_EmSeriesSpinButton(m_refEmSeriesAdjustment),
    m_EmNumberLabel("Номер"),
    m_refEmNumberAdjustment(Gtk::Adjustment::create(0, 0, 65535.0)),
    m_EmNumberSpinButton(m_refEmNumberAdjustment),
    m_HidWiegandLabel("Виганд"),
    m_refHidWiegandAdjustment(Gtk::Adjustment::create(26.0, 10.0, 37.0)),
    m_HidWiegandSpinButton(m_refHidWiegandAdjustment),
    m_HidFacilityLabel("Код производителя"),
    m_HidNumberLabel("Номер"),
    m_refHidNumberAdjustment(Gtk::Adjustment::create(0, 0, 65535.0)),
    m_HidNumberSpinButton(m_refHidNumberAdjustment),
    m_EmuGetButton("Получить эмулируемый"),
    m_EmuSetButton("Применить"),
    m_ReadButton("Читать"),
    m_WriteButton("Записать") {
    m_rNewEmMarineUid.nLength = 5;
    set_destroy_with_parent(true);

    set_title("Temic");
    set_default_size(600, 765);

    m_VBox.set_expand();
    m_VBox.set_spacing(5);
    set_child(m_VBox);

    m_VBox.append(m_DataBox);
    m_VBox.append(m_ConfigFrame);
    m_VBox.append(m_EmulationFrame);
    m_VBox.append(m_BottomBox);

    m_DataBox.append(m_DataScrolledWindow);
    m_DataBox.append(m_RightBox);

    m_DataScrolledWindow.set_policy(Gtk::PolicyType::AUTOMATIC, Gtk::PolicyType::AUTOMATIC);
    m_DataScrolledWindow.set_expand();
    m_DataScrolledWindow.set_child(m_DataGrid);

    m_SerialButton.set_rgba(g_AppSet.m_TmcSerialColor);
    m_PasswordButton.set_rgba(g_AppSet.m_TmcPasswordColor);
    m_UserDataButton.set_rgba(g_AppSet.m_TmcUserDataColor);
    m_ConfigButton.set_rgba(g_AppSet.m_TmcConfigColor);

    m_DataGrid.set_expand();
    m_DataGrid.set_data_pointer(&m_aNewBlocks[0][0], sizeof(m_aNewBlocks));
    m_DataGrid.set_old_data_pointer(&m_aOldBlocks[0][0], sizeof(m_aOldBlocks));
    m_DataGrid.set_block_size(4);
    m_DataGrid.add_sector_size(0, 8);
    m_DataGrid.add_sector_size(8, 2);
    const char* kTitles[3] = {"Блок", "С", "б"};
    m_DataGrid.set_fix_col_titles(kTitles, std::size(kTitles));
    m_DataGrid.set_range_checkbox(0, 8);
    m_DataGrid.add_or_set_byte_attrs(0, 4, CCardDataGrid::BYTE_FORMAT_UNDEF, Gdk::RGBA(0, 0, 0),
                                     m_ConfigButton.get_rgba());
    m_DataGrid.add_or_set_byte_attrs(4, 24, CCardDataGrid::BYTE_FORMAT_UNDEF, Gdk::RGBA(0, 0, 0),
                                     m_UserDataButton.get_rgba());
    m_DataGrid.add_or_set_byte_attrs(28, 4, CCardDataGrid::BYTE_FORMAT_UNDEF, Gdk::RGBA(0, 0, 0),
                                     m_PasswordButton.get_rgba());
    m_DataGrid.add_or_set_byte_attrs(32, 8, CCardDataGrid::BYTE_FORMAT_UNDEF, Gdk::RGBA(0, 0, 0),
                                     m_SerialButton.get_rgba());
    m_DataGrid.update();
    m_DataGrid.signal_get_byte_access().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_data_get_byte_access));
    m_DataGrid.signal_validate_byte().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_data_validate_byte));
    m_DataGrid.signal_checkbox_toggle().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_data_checkbox_toggle));

    m_RightBox.set_margin(5);
    m_RightBox.set_spacing(5);
    m_RightBox.append(m_LegendFrame);
    m_RightBox.append(m_FormatFrame);
    m_RightBox.append(m_PasswordFrame);

    m_LegendFrame.set_child(m_LegendGrid);
    m_LegendGrid.set_margin(5);
    m_LegendGrid.set_column_spacing(5);
    m_LegendGrid.set_row_spacing(5);
    m_LegendGrid.attach(m_SerialButton, 0, 0);
    m_LegendGrid.attach(m_SerialLabel, 1, 0);
    m_LegendGrid.attach(m_PasswordButton, 0, 1);
    m_LegendGrid.attach(m_PasswordLabel, 1, 1);
    m_LegendGrid.attach(m_UserDataButton, 0, 2);
    m_LegendGrid.attach(m_UserDataLabel, 1, 2);
    m_LegendGrid.attach(m_ConfigButton, 0, 3);
    m_LegendGrid.attach(m_ConfigLabel, 1, 3);
    m_SerialLabel.set_halign(Gtk::Align::START);
    m_SerialLabel.set_mnemonic_widget(m_SerialButton);
    m_SerialButton.signal_color_set().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_serial_color_set));
    m_PasswordLabel.set_halign(Gtk::Align::START);
    m_PasswordLabel.set_mnemonic_widget(m_PasswordButton);
    m_PasswordButton.signal_color_set().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_password_color_set));
    m_UserDataLabel.set_halign(Gtk::Align::START);
    m_UserDataLabel.set_mnemonic_widget(m_UserDataButton);
    m_UserDataButton.signal_color_set().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_user_data_color_set));
    m_ConfigLabel.set_halign(Gtk::Align::START);
    m_ConfigLabel.set_mnemonic_widget(m_ConfigButton);
    m_ConfigButton.signal_color_set().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_config_color_set));

    m_FormatFrame.set_child(m_FormatBox);
    m_FormatBox.set_margin(5);
    m_FormatBox.set_spacing(5);
    m_FormatBox.append(m_HexCheckButton);
    m_FormatBox.append(m_DecCheckButton);
    m_FormatBox.append(m_OctCheckButton);
    m_FormatBox.append(m_BinCheckButton);

    m_DecCheckButton.set_group(m_HexCheckButton);
    m_OctCheckButton.set_group(m_DecCheckButton);
    m_BinCheckButton.set_group(m_OctCheckButton);
    m_HexCheckButton.signal_toggled().connect(sigc::mem_fun(*this, &CTemicDialog::on_hex_toggled));
    m_DecCheckButton.signal_toggled().connect(sigc::mem_fun(*this, &CTemicDialog::on_dec_toggled));
    m_OctCheckButton.signal_toggled().connect(sigc::mem_fun(*this, &CTemicDialog::on_oct_toggled));
    m_BinCheckButton.signal_toggled().connect(sigc::mem_fun(*this, &CTemicDialog::on_bin_toggled));

    m_PasswordFrame.set_child(m_PasswordBox);
    m_PasswordFrame.set_hexpand(false);
    m_PasswordFrame.set_label_widget(m_PasswordCheckButton);
    m_PasswordBox.set_margin(5);
    m_PasswordBox.set_spacing(5);
    m_PasswordBox.append(m_PasswordDropDown);
    m_PasswordBox.append(m_PasswordEditButton);
    m_PasswordDropDown.set_hexpand();
    m_refPasswordStringList = Gtk::StringList::create({});
    m_PasswordDropDown.set_model(m_refPasswordStringList);
    m_PasswordDropDown.property_selected().signal_changed().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_password_selected_changed));

    m_PasswordCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_button_toggle_password));
    m_PasswordEditButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_button_edit_passwords));

    m_ConfigFrame.set_margin(5);
    m_ConfigFrame.set_child(m_ConfigGrid);
    m_ConfigGrid.set_margin(5);
    m_ConfigGrid.set_column_spacing(5);
    m_ConfigGrid.set_row_spacing(5);
    m_ConfigGrid.set_hexpand();
    m_ConfigGrid.attach(m_ModeLabel, 0, 0);
    m_ConfigGrid.attach(m_ModeDropDown, 1, 0);
    m_ConfigGrid.attach(m_BitRateLabel, 0, 1);
    m_ConfigGrid.attach(m_BitRateDropDown, 1, 1);
    m_ConfigGrid.attach(m_ModulationLabel, 0, 2);
    m_ConfigGrid.attach(m_ModulationDropDown, 1, 2);
    m_ConfigGrid.attach(m_PskCfLabel, 0, 3);
    m_ConfigGrid.attach(m_PskCfDropDown, 1, 3);
    m_ConfigGrid.attach(m_MaxBlockLabel, 0, 4);
    m_ConfigGrid.attach(m_MaxBlockDropDown, 1, 4);
    m_ConfigGrid.attach(m_ConfigFlagsBox, 2, 0, 1, 5);

    m_ModeLabel.set_mnemonic_widget(m_ModeDropDown);
    m_ModeLabel.set_halign(Gtk::Align::END);
    m_refModeStringList = Gtk::StringList::create({"e5550 compatibility", "X-Mode"});
    m_ModeDropDown.set_model(m_refModeStringList);
    m_ModeDropDown.property_selected().signal_changed().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_mode_selected_changed));

    m_BitRateLabel.set_mnemonic_widget(m_BitRateDropDown);
    m_BitRateLabel.set_halign(Gtk::Align::END);
    m_refBitRateStringList = Gtk::StringList::create({});
    m_BitRateDropDown.set_model(m_refBitRateStringList);
    m_BitRateDropDown.property_selected().signal_changed().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_bitrate_selected_changed));

    m_ModulationLabel.set_mnemonic_widget(m_ModulationDropDown);
    m_ModulationLabel.set_halign(Gtk::Align::END);
    m_refModulationStringList = Gtk::StringList::create({});
    m_ModulationDropDown.set_model(m_refModulationStringList);
    m_ModulationDropDown.property_selected().signal_changed().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_modulation_selected_changed));

    m_PskCfLabel.set_mnemonic_widget(m_PskCfDropDown);
    m_PskCfLabel.set_halign(Gtk::Align::END);
    m_refPskCfStringList = Gtk::StringList::create({"RF/2", "RF/4", "RF/8", "Res"});
    m_PskCfDropDown.set_model(m_refPskCfStringList);
    m_PskCfDropDown.property_selected().signal_changed().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_pskcf_selected_changed));

    m_MaxBlockLabel.set_mnemonic_widget(m_MaxBlockDropDown);
    m_MaxBlockLabel.set_halign(Gtk::Align::END);
    m_refMaxBlockStringList = Gtk::StringList::create({});
    for (size_t i = 0; i < 8; i++)
        m_refMaxBlockStringList->append(Glib::ustring::format(i));
    m_MaxBlockDropDown.set_model(m_refMaxBlockStringList);
    m_MaxBlockDropDown.property_selected().signal_changed().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_maxblock_selected_changed));

    m_ConfigFlagsBox.set_margin(5);
    m_ConfigFlagsBox.append(m_TestModeCheckButton);
    m_ConfigFlagsBox.append(m_AorModeCheckButton);
    m_ConfigFlagsBox.append(m_StCheckButton);
    m_ConfigFlagsBox.append(m_SstCheckButton);
    m_ConfigFlagsBox.append(m_PorDelayCheckButton);
    m_ConfigFlagsBox.append(m_OtpCheckButton);
    m_ConfigFlagsBox.append(m_FastWriteCheckButton);
    m_ConfigFlagsBox.append(m_InverseDataCheckButton);

    m_TestModeCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_test_mode_toggle));
    m_AorModeCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_aor_mode_toggle));
    m_StCheckButton.signal_toggled().connect(sigc::mem_fun(*this, &CTemicDialog::on_st_toggle));
    m_SstCheckButton.signal_toggled().connect(sigc::mem_fun(*this, &CTemicDialog::on_sst_toggle));
    m_PorDelayCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_por_delay_toggle));
    m_OtpCheckButton.signal_toggled().connect(sigc::mem_fun(*this, &CTemicDialog::on_otp_toggle));
    m_FastWriteCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_fast_write_toggle));
    m_InverseDataCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_inverse_data_toggle));

    m_EmulationFrame.set_margin(5);
    m_EmulationFrame.set_child(m_EmulationBox);
    m_EmulationBox.set_margin(5);
    m_EmulationBox.set_spacing(5);
    m_EmulationBox.append(m_CurrentEmuFrame);
    m_EmulationBox.append(m_NewEmuFrame);
    m_CurrentEmuFrame.set_child(m_CurrentEmuLabel);
    m_CurrentEmuLabel.set_selectable();
    m_CurrentEmuLabel.set_halign(Gtk::Align::START);
    m_NewEmuFrame.set_child(m_NewEmuBox);
    m_NewEmuBox.set_margin(5);
    m_NewEmuBox.append(m_EmulationNotebook);
    m_NewEmuBox.append(m_EmuButtonsBox);
    m_HidCheckButton.set_group(m_EmMarineCheckButton);
    m_EmulationNotebook.set_hexpand();
    m_EmulationNotebook.append_page(m_EmMarineGrid, m_EmMarineCheckButton);
    m_EmulationNotebook.append_page(m_HidGrid, m_HidCheckButton);
    m_EmulationNotebook.signal_switch_page().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_emulation_switch_page));
    m_EmMarineCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_em_marine_toggled));
    m_HidCheckButton.signal_toggled().connect(sigc::mem_fun(*this, &CTemicDialog::on_hid_toggled));
    m_EmMarineGrid.set_margin(5);
    m_EmMarineGrid.set_column_spacing(5);
    m_EmMarineGrid.set_row_spacing(5);
    m_EmMarineGrid.attach(m_EmFacilityLabel, 0, 0);
    m_EmMarineGrid.attach(m_EmFacilityEntry, 0, 1);
    m_EmMarineGrid.attach(m_EmSeriesLabel, 1, 0);
    m_EmMarineGrid.attach(m_EmSeriesSpinButton, 1, 1);
    m_EmMarineGrid.attach(m_EmNumberLabel, 2, 0);
    m_EmMarineGrid.attach(m_EmNumberSpinButton, 2, 1);
    m_EmSeriesSpinButton.set_wrap();
    m_EmSeriesSpinButton.set_size_request(55, -1);
    m_EmNumberSpinButton.set_wrap();
    m_EmNumberSpinButton.set_size_request(55, -1);
    m_HidGrid.set_margin(5);
    m_HidGrid.set_column_spacing(5);
    m_HidGrid.set_row_spacing(5);
    m_HidGrid.attach(m_HidWiegandLabel, 0, 0);
    m_HidGrid.attach(m_HidWiegandSpinButton, 0, 1);
    m_HidGrid.attach(m_HidFacilityLabel, 1, 0);
    m_HidGrid.attach(m_HidFacilityEntry, 1, 1);
    m_HidGrid.attach(m_HidNumberLabel, 2, 0);
    m_HidGrid.attach(m_HidNumberSpinButton, 2, 1);
    m_HidWiegandSpinButton.set_size_request(55, -1);
    m_HidWiegandSpinButton.signal_value_changed().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_hid_wiegand_value_changed));
    m_HidNumberSpinButton.set_wrap();
    m_HidNumberSpinButton.set_size_request(55, -1);
    m_EmuButtonsBox.set_margin(5);
    m_EmuButtonsBox.set_spacing(5);
    m_EmuButtonsBox.append(m_EmuGetButton);
    m_EmuButtonsBox.append(m_EmuSetButton);

    m_EmuGetButton.signal_clicked().connect(sigc::mem_fun(*this, &CTemicDialog::on_button_emu_get));
    m_EmuSetButton.signal_clicked().connect(sigc::mem_fun(*this, &CTemicDialog::on_button_emu_set));

    m_BottomBox.set_margin(5);
    m_BottomBox.set_spacing(5);
    m_BottomBox.append(m_ReadButton);
    m_BottomBox.append(m_WriteButton);

    m_ReadButton.signal_clicked().connect(sigc::mem_fun(*this, &CTemicDialog::on_button_read));
    m_WriteButton.signal_clicked().connect(sigc::mem_fun(*this, &CTemicDialog::on_button_write));

    signal_hide().connect(sigc::mem_fun(*this, &CTemicDialog::on_hide));

    m_nNewCardType = g_AppSet.m_nTmcNewCardType;
    m_rNewEmMarineUid = g_AppSet.m_rNewEmMarineUid;
    m_rNewHidUid = g_AppSet.m_rNewHidUid;
    UpdateNewCardTypeCtrl();
}

CTemicDialog::~CTemicDialog() {
}

void CTemicDialog::Init(const ilr::CReader& oReader,
                        Glib::RefPtr<CTemicPasswordsSettings> refSettings) {
    m_oReader = oReader.Clone();
    m_refTmcPasswords = refSettings;
    ilr_card_info rInfo;
    m_oReader.GetCardInfo(rInfo);
    m_rTemicUid = rInfo.rUID;
    set_title(Glib::ustring::format("Temic ", ilr::CardUIDToStr(rInfo.nType, rInfo.rUID)));
    m_DataGrid.set_byte_format(g_AppSet.m_nDataFormat);
    switch (m_DataGrid.get_byte_format()) {
    case byte_format::BYTE_FORMAT_HEX:
        m_HexCheckButton.set_active();
        break;

    case byte_format::BYTE_FORMAT_DEC:
        m_DecCheckButton.set_active();
        break;

    case byte_format::BYTE_FORMAT_OCT:
        m_OctCheckButton.set_active();
        break;

    case byte_format::BYTE_FORMAT_BIN:
        m_BinCheckButton.set_active();
        break;
    }
    m_nActivePassword = -1;
    UpdateNewEmMarineCtrlData(false);
    UpdateNewHidCtrlData(false);
    ReadCardData();
}

uint CTemicDialog::on_data_get_byte_access(uint nAddress) {
    return (nAddress >= 32) ? 0 : 0xff;
}

void CTemicDialog::on_data_validate_byte(uint nAddress, uint8_t& b) {
    SetByte(nAddress / 4, nAddress % 4, b, false);
}

void CTemicDialog::on_data_checkbox_toggle(uint nBlockIdx) {
    auto fCheck = !m_DataGrid.get_checkbox_active(nBlockIdx);
    SET_BIT(m_nLocks, nBlockIdx, fCheck);
    m_DataGrid.set_checkbox_active(nBlockIdx, fCheck);
}

void CTemicDialog::on_serial_color_set() {
    g_AppSet.SetTmcSerialColor(m_SerialButton.get_rgba());
    m_DataGrid.add_or_set_byte_attrs(32, 8, CCardDataGrid::BYTE_FORMAT_UNDEF, Gdk::RGBA(0, 0, 0),
                                     m_SerialButton.get_rgba());
}

void CTemicDialog::on_password_color_set() {
    g_AppSet.SetTmcPasswordColor(m_PasswordButton.get_rgba());
    UpdateDataGridPasswordColor();
}

void CTemicDialog::on_user_data_color_set() {
    g_AppSet.SetTmcUserDataColor(m_UserDataButton.get_rgba());
    m_DataGrid.add_or_set_byte_attrs(4, 24, CCardDataGrid::BYTE_FORMAT_UNDEF, Gdk::RGBA(0, 0, 0),
                                     m_UserDataButton.get_rgba());
}

void CTemicDialog::on_config_color_set() {
    g_AppSet.SetTmcConfigColor(m_ConfigButton.get_rgba());
    m_DataGrid.add_or_set_byte_attrs(0, 4, CCardDataGrid::BYTE_FORMAT_UNDEF, Gdk::RGBA(0, 0, 0),
                                     m_ConfigButton.get_rgba());
}

void CTemicDialog::on_hex_toggled() {
    m_DataGrid.set_byte_format(CCardDataGrid::BYTE_FORMAT_HEX);
}

void CTemicDialog::on_dec_toggled() {
    m_DataGrid.set_byte_format(CCardDataGrid::BYTE_FORMAT_DEC);
}

void CTemicDialog::on_oct_toggled() {
    m_DataGrid.set_byte_format(CCardDataGrid::BYTE_FORMAT_OCT);
}

void CTemicDialog::on_bin_toggled() {
    m_DataGrid.set_byte_format(CCardDataGrid::BYTE_FORMAT_BIN);
}

void CTemicDialog::on_button_toggle_password() {
    auto fSetPassword = m_PasswordCheckButton.get_active();
    auto nByte = m_aNewBlocks[0][3];
    if (fSetPassword == ((nByte & 0x10) != 0))
        return;

    if (fSetPassword)
        nByte |= 0x10;
    else
        nByte &= ~0x10;
    SetByte(0, 3, nByte);

    if (fSetPassword) {
        CTemicPassword oInfo;
        oInfo.m_nPassword = *(uint32_t*)&m_aNewBlocks[7][0];
        auto nIdx = FindTemicPassword(m_refTmcPasswords->m_oPasswords, oInfo.m_nPassword);
        if (-1 == nIdx) {
            oInfo.m_sComment = ilr::kCardTypeNames[ILR_CARD_TEMIC];
            oInfo.m_sComment += ' ';
            oInfo.m_sComment += ilr::CardUIDToStr(ILR_CARD_TEMIC, m_rTemicUid);
            m_refTmcPasswords->m_oPasswords.emplace_back(std::move(oInfo));
            nIdx = m_refTmcPasswords->m_oPasswords.size() - 1;
            m_refTmcPasswords->SetModifiedFlag();
            UpdatePasswordDropDown();
        }
        m_PasswordDropDown.set_selected(nIdx);
    }
    UpdateDataGridPasswordColor();
    UpdatePasswordGroupCtrlState();
}

void CTemicDialog::on_password_selected_changed() {
    if (m_fLockUpdate || !m_PasswordCheckButton.get_active())
        return;
    auto nIdx = m_PasswordDropDown.get_selected();
    if (nIdx != GTK_INVALID_LIST_POSITION) {
        auto nNewPassword = m_refTmcPasswords->m_oPasswords[nIdx].m_nPassword;
        auto pPassword = (uint32_t*)&m_aNewBlocks[7][0];

        if (*pPassword != nNewPassword) {
            *pPassword = nNewPassword;
            m_DataGrid.redraw();
        }
    }
}

void CTemicDialog::on_button_edit_passwords() {
    auto nPassword = *(uint32_t*)&m_aNewBlocks[7][0];
    if (nullptr == m_refPasswordsDialog)
        m_refPasswordsDialog.reset(new CTemicPasswordsDialog());
    m_refPasswordsDialog->set_transient_for(*this);
    m_refPasswordsDialog->set_modal();
    m_refPasswordsDialog->set_hide_on_close();
    m_refPasswordsDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CTemicDialog::on_passwords_dialog_hide));

    ilr_reader_info rInfo;
    m_oReader.GetReaderInfo(rInfo);
    ilr::CILR oILR;
    auto oReader = oILR.GetReader(rInfo.nPortType, rInfo.pszPortName);
    m_refPasswordsDialog->Init(oReader, m_refTmcPasswords, nPassword);
    m_refPasswordsDialog->set_visible(true);
}

void CTemicDialog::on_passwords_dialog_hide() {
    if (m_refPasswordsDialog->m_fAccept) {
        if (m_PasswordCheckButton.get_active()) {
            auto nNewPassword = m_refPasswordsDialog->m_nCurrentPassword;
            auto pCurrentPassword = (uint32_t*)&m_aNewBlocks[7][0];
            if (*pCurrentPassword != nNewPassword) {
                *pCurrentPassword = nNewPassword;
                m_DataGrid.redraw();
            }
        }
        UpdatePasswordDropDown();
    }
    m_refPasswordsDialog = nullptr;
}

void CTemicDialog::on_mode_selected_changed() {
    if (m_fLockUpdate)
        return;
    auto nIdx = m_ModeDropDown.get_selected();
    if (nIdx != GTK_INVALID_LIST_POSITION) {
        auto pCurrentConfig = (uint32_t*)&m_aNewBlocks[0][0];
        auto nNewConfig = *pCurrentConfig;

        switch (nIdx) {
        case 0:
            SET_BIT(nNewConfig, 9, false);
            nNewConfig &= ~0xF;
            if (m_TestModeCheckButton.get_active())
                nNewConfig |= 6;
            else
                nNewConfig |= 9;
            break;

        case 1:
            SET_BIT(nNewConfig, 9, true);
            nNewConfig &= ~0xF;
            if (m_TestModeCheckButton.get_active())
                nNewConfig |= 9;
            else
                nNewConfig |= 6;
            break;
        }
        if (*pCurrentConfig != nNewConfig) {
            *pCurrentConfig = nNewConfig;
            m_DataGrid.redraw();
            UpdateBitRateDropDown();
            UpdateModulationDropDown();
        }
    }
}

void CTemicDialog::on_bitrate_selected_changed() {
    if (m_fLockUpdate)
        return;
    auto nIdx = m_BitRateDropDown.get_selected();
    if (nIdx != GTK_INVALID_LIST_POSITION) {
        auto pCurrentConfig = (uint32_t*)&m_aNewBlocks[0][0];
        auto nNewConfig = *pCurrentConfig;
        if (GetXMode())
            nNewConfig = (nNewConfig & ~0xFC00) | (((nIdx - 2) / 2) << 10);
        else
            nNewConfig = (nNewConfig & ~0x1C00) | (nIdx << 10);

        if (*pCurrentConfig != nNewConfig) {
            *pCurrentConfig = nNewConfig;
            m_DataGrid.redraw();
        }
    }
}

void CTemicDialog::on_modulation_selected_changed() {
    if (m_fLockUpdate)
        return;
    auto nIdx = m_ModulationDropDown.get_selected();
    if (nIdx != GTK_INVALID_LIST_POSITION) {
        auto pCurrentConfig = (uint32_t*)&m_aNewBlocks[0][0];
        auto nNewConfig = *pCurrentConfig;

        nNewConfig = (nNewConfig & ~0xF00100) | ((nIdx & 0xF) << 20) | ((nIdx >> 4) << 8);

        if (*pCurrentConfig != nNewConfig) {
            *pCurrentConfig = nNewConfig;
            m_DataGrid.redraw();
        }
    }
}

void CTemicDialog::on_pskcf_selected_changed() {
    if (m_fLockUpdate)
        return;
    auto nIdx = m_PskCfDropDown.get_selected();
    if (nIdx != GTK_INVALID_LIST_POSITION) {
        auto pCurrentConfig = (uint32_t*)&m_aNewBlocks[0][0];
        auto nNewConfig = *pCurrentConfig;

        nNewConfig = (nNewConfig & ~0xC0000) | (nIdx << 18);

        if (*pCurrentConfig != nNewConfig) {
            *pCurrentConfig = nNewConfig;
            m_DataGrid.redraw();
        }
    }
}

void CTemicDialog::on_maxblock_selected_changed() {
    if (m_fLockUpdate)
        return;
    auto nIdx = m_MaxBlockDropDown.get_selected();
    if (nIdx != GTK_INVALID_LIST_POSITION) {
        auto pCurrentConfig = (uint32_t*)&m_aNewBlocks[0][0];
        auto nNewConfig = *pCurrentConfig;

        nNewConfig = (nNewConfig & ~0xE0000000) | (nIdx << 29);

        if (*pCurrentConfig != nNewConfig) {
            *pCurrentConfig = nNewConfig;
            m_DataGrid.redraw();
        }
    }
}

void CTemicDialog::on_test_mode_toggle() {
    auto fChecked = m_TestModeCheckButton.get_active();
    auto nConfig = *(uint32_t*)&m_aNewBlocks[0][0];
    auto fXMode = GetXMode();

    nConfig &= ~0xF;
    if (fXMode) {
        if (fChecked)
            nConfig |= 9;
        else
            nConfig |= 6;
    }
    else if (fChecked)
        nConfig |= 6;
    else
        nConfig |= 9;

    if (*(uint32_t*)&m_aNewBlocks[0][0] != nConfig) {
        *(uint32_t*)&m_aNewBlocks[0][0] = nConfig;
        m_DataGrid.redraw();
    }
}

void CTemicDialog::on_aor_mode_toggle() {
    auto fChecked = m_AorModeCheckButton.get_active();
    auto nConfig = *(uint32_t*)&m_aNewBlocks[0][0];
    SET_BIT(nConfig, 17, fChecked);

    if (*(uint32_t*)&m_aNewBlocks[0][0] != nConfig) {
        *(uint32_t*)&m_aNewBlocks[0][0] = nConfig;
        m_DataGrid.redraw();
    }
}

void CTemicDialog::on_st_toggle() {
    auto fChecked = m_StCheckButton.get_active();
    auto nConfig = *(uint32_t*)&m_aNewBlocks[0][0];

    SET_BIT(nConfig, 27, fChecked);

    if (*(uint32_t*)&m_aNewBlocks[0][0] != nConfig) {
        *(uint32_t*)&m_aNewBlocks[0][0] = nConfig;
        m_DataGrid.redraw();
    }
}

void CTemicDialog::on_sst_toggle() {
    auto fChecked = m_SstCheckButton.get_active();
    auto nConfig = *(uint32_t*)&m_aNewBlocks[0][0];

    SET_BIT(nConfig, 27, fChecked);

    if (*(uint32_t*)&m_aNewBlocks[0][0] != nConfig) {
        *(uint32_t*)&m_aNewBlocks[0][0] = nConfig;
        m_DataGrid.redraw();
    }
}

void CTemicDialog::on_por_delay_toggle() {
    auto fChecked = m_PorDelayCheckButton.get_active();
    auto nConfig = *(uint32_t*)&m_aNewBlocks[0][0];

    SET_BIT(nConfig, 17, fChecked);

    if (*(uint32_t*)&m_aNewBlocks[0][0] != nConfig) {
        *(uint32_t*)&m_aNewBlocks[0][0] = nConfig;
        m_DataGrid.redraw();
    }
}

void CTemicDialog::on_otp_toggle() {
    auto fChecked = m_OtpCheckButton.get_active();
    auto nConfig = *(uint32_t*)&m_aNewBlocks[0][0];
    auto fXMode = GetXMode();

    SET_BIT(nConfig, 16, fXMode && fChecked);

    if (*(uint32_t*)&m_aNewBlocks[0][0] != nConfig) {
        *(uint32_t*)&m_aNewBlocks[0][0] = nConfig;
        m_DataGrid.redraw();
    }
}

void CTemicDialog::on_fast_write_toggle() {
    auto fChecked = m_FastWriteCheckButton.get_active();
    auto nConfig = *(uint32_t*)&m_aNewBlocks[0][0];
    auto fXMode = GetXMode();

    SET_BIT(nConfig, 26, fXMode && fChecked);

    if (*(uint32_t*)&m_aNewBlocks[0][0] != nConfig) {
        *(uint32_t*)&m_aNewBlocks[0][0] = nConfig;
        m_DataGrid.redraw();
    }
}

void CTemicDialog::on_inverse_data_toggle() {
    auto fChecked = m_InverseDataCheckButton.get_active();
    auto nConfig = *(uint32_t*)&m_aNewBlocks[0][0];
    auto fXMode = GetXMode();

    SET_BIT(nConfig, 25, fXMode && fChecked);

    if (*(uint32_t*)&m_aNewBlocks[0][0] != nConfig) {
        *(uint32_t*)&m_aNewBlocks[0][0] = nConfig;
        m_DataGrid.redraw();
    }
}

void CTemicDialog::on_emulation_switch_page(Widget* page, guint page_number) {
    switch (page_number) {
    case 0:
        m_nNewCardType = ILR_CARD_EM_MARINE;
        m_EmMarineCheckButton.set_active();
        m_HidCheckButton.set_active(false);
        break;

    case 1:
        m_nNewCardType = ILR_CARD_HID;
        m_HidCheckButton.set_active();
        m_EmMarineCheckButton.set_active(false);
        break;
    }
}

void CTemicDialog::on_em_marine_toggled() {
    if (m_EmMarineCheckButton.get_active()) {
        m_nNewCardType = ILR_CARD_EM_MARINE;
        m_EmulationNotebook.set_current_page(0);
    }
}

void CTemicDialog::on_hid_toggled() {
    if (m_HidCheckButton.get_active()) {
        m_nNewCardType = ILR_CARD_HID;
        m_EmulationNotebook.set_current_page(1);
    }
}

void CTemicDialog::on_hid_wiegand_value_changed() {
    if (m_fLockUpdate)
        return;
    auto value = m_HidWiegandSpinButton.get_value_as_int();

    UpdateNewHidCtrlData(true);
    m_nNewHidWiegand = value;
    UpdateNewHidCtrlData(false);
}

void CTemicDialog::on_button_emu_get() {
    ilr_card_type nCardType;
    ilr::CCardUID rUid;
    int nWiegand = 0;
    m_oReader.DecodeTemicEmMarine((uint32_t*)&m_aNewBlocks[0], 3, rUid);
    if (!rUid.IsEmpty())
        nCardType = ILR_CARD_EM_MARINE;
    else {
        m_oReader.DecodeTemicHid((uint32_t*)&m_aNewBlocks[0], 4, rUid, nWiegand);
        nCardType = ILR_CARD_HID;
    }

    if (rUid.IsEmpty())
        MessageDialog("Нет эмулируемой карты");
    else {
        m_nNewCardType = nCardType;
        if (ILR_CARD_EM_MARINE == nCardType) {
            m_rNewEmMarineUid = rUid;
            UpdateNewEmMarineCtrlData(false);
        }
        else {
            m_rNewHidUid = rUid;
            m_nNewHidWiegand = nWiegand;
            UpdateNewHidCtrlData(false);
        }
        UpdateNewCardTypeCtrl();
    }
}

void CTemicDialog::on_button_emu_set() {
    switch (m_nNewCardType) {
    case ILR_CARD_EM_MARINE:
        {
            UpdateNewEmMarineCtrlData(true);
            if (!m_fEntryEmFacilityValid) {
                m_EmFacilityEntry.grab_focus();
                MessageDialog("Некорректное значение кода производителя Em-Marine.",
                              Gtk::MessageType::ERROR);
                return;
            }
            m_oReader.EncodeTemicEmMarine(m_rNewEmMarineUid, (uint32_t*)&m_aNewBlocks[0][0], 3);
            m_DataGrid.redraw();
            UpdateConfigGroupData(false);
            break;
        }

    case ILR_CARD_HID:
        {
            UpdateNewHidCtrlData(true);
            if (!m_fEntryHidFacilityValid) {
                m_HidFacilityEntry.grab_focus();
                MessageDialog("Некорректное значение кода производителя HID.",
                              Gtk::MessageType::ERROR);
                return;
            }
            m_oReader.EncodeTemicHid(m_rNewHidUid, (uint32_t*)&m_aNewBlocks[0][0], 4,
                                     m_nNewHidWiegand);
            m_DataGrid.redraw();
            UpdateConfigGroupData(false);
            break;
        }
    }
}

void CTemicDialog::on_button_read() {
    ReadCardData();
}

void CTemicDialog::on_button_write() {
    if (m_nLocks != 0) {
        MessageDialog("Действительно хотите запретить перезапись Temic?",
                      Gtk::MessageType::QUESTION, Gtk::ButtonsType::OK_CANCEL,
                      sigc::mem_fun(*this, &CTemicDialog::on_confirm_lock_temic_dialog_response));
        return;
    }

    WriteCardData();
}

void CTemicDialog::on_confirm_lock_temic_dialog_response(int response_id) {
    m_refDialog = nullptr;
    if (response_id != GTK_RESPONSE_OK)
        return;
    WriteCardData();
}

void CTemicDialog::on_dialog_response(int) {
    m_refDialog = nullptr;
}

void CTemicDialog::on_hide() {
    m_oReader.Close();
    g_AppSet.SetDataFormat(m_DataGrid.get_byte_format());
    g_AppSet.SetTemicNewCardType(m_nNewCardType);
    UpdateNewEmMarineCtrlData(true);
    g_AppSet.SetTemicEmMarineUid(m_rNewEmMarineUid);
    UpdateNewHidCtrlData(true);
    g_AppSet.SetTemicHidUid(m_rNewHidUid);
}

void CTemicDialog::ReadCardData() {
    try {
        m_oReader.LoadTemicPassword(m_nActivePassword);
        m_oReader.ReadTemic(0, (uint32_t*)&m_aOldBlocks[0][0], std::size(m_aOldBlocks));
        memcpy(m_aNewBlocks, m_aOldBlocks, sizeof(m_aNewBlocks));
    }
    catch (const std::exception& e) {
        // std::cerr << e.what() << '\n';
        MessageDialog(e.what(), Gtk::MessageType::ERROR);
    }
    UpdateDataGridPasswordColor();
    m_DataGrid.redraw();
    m_PasswordCheckButton.set_active(GET_BIT(m_aNewBlocks[0][3], 4));
    UpdatePasswordDropDown();
    UpdatePasswordGroupCtrlState();
    UpdateConfigGroupData(false);
    UpdateEmulatedLabel();
}

void CTemicDialog::WriteCardData() {
    try {
        m_oReader.LoadTemicPassword(m_nActivePassword);
        // Сначала пишем данные User Data
        for (size_t i = 0; i < 6; i++) {
            if (memcmp(m_aNewBlocks[i], m_aOldBlocks[i], 4) != 0) {
                auto fLock = GET_BIT(m_nLocks, i);
                m_oReader.WriteTemic(i, (uint32_t*)m_aNewBlocks[i], 1, fLock);
                memcpy(m_aOldBlocks[i], m_aNewBlocks[i], 4);
            }
        }

        // Пишем пароль
        if (memcmp(m_aNewBlocks[7], m_aOldBlocks[7], 4) != 0) {
            auto fLock = GET_BIT(m_nLocks, 7);
            m_oReader.WriteTemic(7, (uint32_t*)m_aNewBlocks[7], 1, fLock);
            if (m_nActivePassword != -1)
                m_nActivePassword = *(uint32_t*)&m_aNewBlocks[7];
            memcpy(m_aOldBlocks[7], m_aNewBlocks[7], 4);
            m_oReader.LoadTemicPassword(m_nActivePassword);
        }

        // Пишем конфигурацию
        if (memcmp(m_aNewBlocks[0], m_aOldBlocks[0], 4) != 0) {
            auto fLock = GET_BIT(m_nLocks, 0);
            m_oReader.WriteTemic(0, (uint32_t*)m_aNewBlocks[0], 1, fLock);
            memcpy(m_aOldBlocks[0], m_aNewBlocks[0], 4);
            auto fSetPassword = ((m_aNewBlocks[0][3] & 0x10) != 0);
            if (fSetPassword) {
                if (fSetPassword)
                    m_nActivePassword = *(uint32_t*)&m_aNewBlocks[7];
                else
                    m_nActivePassword = -1;
                m_oReader.LoadTemicPassword(m_nActivePassword);
            }
        }
    }
    catch (const std::exception& e) {
        // std::cerr << e.what() << '\n';
        MessageDialog(e.what(), Gtk::MessageType::ERROR);
    }
    m_DataGrid.redraw();
}

void CTemicDialog::SetByte(int nBlockIdx, int nByteIdx, uint8_t nValue, bool fUpdateGrid) {
    if (m_aNewBlocks[nBlockIdx][nByteIdx] == nValue)
        return;
    m_aNewBlocks[nBlockIdx][nByteIdx] = nValue;
    if (fUpdateGrid)
        m_DataGrid.redraw();
    if (0 == nBlockIdx) {
        if (3 == nByteIdx) {
            m_PasswordCheckButton.set_active(GET_BIT(m_aNewBlocks[0][3], 4));
            UpdatePasswordGroupCtrlState();
        }
        UpdateConfigGroupData(false);
    }
}

bool CTemicDialog::GetXMode() const {
    auto nConfig = *(uint32_t*)&m_aNewBlocks[0][0];
    auto nMasterKey = (nConfig >> 4) & 0xF;
    return GET_BIT(nConfig, 9) && ((6 == nMasterKey) || (9 == nMasterKey));
}

void CTemicDialog::UpdateConfigGroupData(bool fSave) {
    auto nConfig = *(uint32_t*)&m_aNewBlocks[0][0];
    auto nMasterKey = (nConfig >> 4) & 0xF;
    auto fXMode = GET_BIT(nConfig, 9) && ((6 == nMasterKey) || (9 == nMasterKey));

    if (!fSave) {
        if (fXMode) {
            m_ModeDropDown.set_selected(1);
            m_TestModeCheckButton.set_active(9 == nMasterKey);
            m_SstCheckButton.set_active(GET_BIT(nConfig, 27));
            m_OtpCheckButton.set_active(GET_BIT(nConfig, 16));
            m_FastWriteCheckButton.set_active(GET_BIT(nConfig, 26));
            m_InverseDataCheckButton.set_active(GET_BIT(nConfig, 25));
        }
        else {
            m_ModeDropDown.set_selected(0);
            m_TestModeCheckButton.set_active(6 == nMasterKey);
            m_StCheckButton.set_active(GET_BIT(nConfig, 27));
        }
        m_StCheckButton.set_visible(!fXMode);
        m_SstCheckButton.set_visible(fXMode);
        m_OtpCheckButton.set_visible(fXMode);
        m_FastWriteCheckButton.set_visible(fXMode);
        m_InverseDataCheckButton.set_visible(fXMode);
        UpdateBitRateDropDown();
        UpdateModulationDropDown();
        m_PskCfDropDown.set_selected((nConfig >> 18) & 3);
        m_MaxBlockDropDown.set_selected((nConfig >> 29) & 7);
        m_AorModeCheckButton.set_active(GET_BIT(nConfig, 17));
        m_PorDelayCheckButton.set_active(GET_BIT(nConfig, 24));
    }
}

void CTemicDialog::UpdatePasswordGroupCtrlState() {
    bool f = m_PasswordCheckButton.get_active();
    m_PasswordDropDown.set_sensitive(f);
    m_PasswordEditButton.set_sensitive(f);
}

void CTemicDialog::UpdatePasswordDropDown() {
    Glib::OptionGroup::vecustrings list;
    for (auto& r : m_refTmcPasswords->m_oPasswords)
        list.emplace_back(TemicPasswordToString(r.m_nPassword));

    m_fLockUpdate = true;
    m_refPasswordStringList->splice(0, m_refPasswordStringList->get_n_items(), list);

    auto nCurrentPassword = *(uint32_t*)&m_aNewBlocks[7][0];
    auto nIdx = FindTemicPassword(m_refTmcPasswords->m_oPasswords, nCurrentPassword);
    if (nIdx != -1)
        m_PasswordDropDown.set_selected(nIdx);
    m_fLockUpdate = false;
}

void CTemicDialog::UpdateDataGridPasswordColor() {
    auto fPasswordSet = (m_aNewBlocks[0][3] & 0x10) != 0;
    m_DataGrid.add_or_set_byte_attrs(
        28, 4, CCardDataGrid::BYTE_FORMAT_UNDEF, Gdk::RGBA(0, 0, 0),
        fPasswordSet ? m_PasswordButton.get_rgba() : m_UserDataButton.get_rgba());
}

void CTemicDialog::UpdateBitRateDropDown() {
    Glib::OptionGroup::vecustrings oNewList;
    guint nIdx;
    auto nConfig = *(uint32_t*)&m_aNewBlocks[0][0];
    if (GetXMode()) {
        oNewList.reserve(64);
        for (size_t i = 0; i < 64; i++)
            oNewList.emplace_back(Glib::ustring::sprintf("RF/%u", (i * 2 + 2)));
        nIdx = ((nConfig >> 10) & 0x3F);
    }
    else {
        oNewList.reserve(std::size(s_aE5550BitRates));
        for (size_t i = 0; i < std::size(s_aE5550BitRates); i++)
            oNewList.emplace_back(Glib::ustring::sprintf("RF/%u", s_aE5550BitRates[i]));
        nIdx = ((nConfig >> 10) & 7);
    }
    m_refBitRateStringList->splice(0, m_refBitRateStringList->get_n_items(), oNewList);
    m_BitRateDropDown.set_selected(nIdx);
}

void CTemicDialog::UpdateModulationDropDown() {
    auto nConfig = *(uint32_t*)&m_aNewBlocks[0][0];
    guint nIdx = ((nConfig >> 20) & 0xF) | (((nConfig >> 8) & 1) << 4);

    Glib::OptionGroup::vecustrings oNewList;
    oNewList.reserve(32);
    if (GetXMode()) {
        for (size_t i = 0; i < std::size(kModulationNames); i++) {
            if (GET_BIT(kXModeModulations, i))
                oNewList.emplace_back(Glib::ustring::sprintf("%u %s", i, kModulationNames[i]));
            else
                oNewList.emplace_back(Glib::ustring::sprintf("%u", i));
        }
    }
    else {
        for (size_t i = 0; i < std::size(kModulationNames); i++) {
            if (GET_BIT(kE5550Modulations, i))
                oNewList.emplace_back(Glib::ustring::sprintf("%u %s", i, kModulationNames[i]));
            else
                oNewList.emplace_back(Glib::ustring::sprintf("%u", i));
        }
    }
    m_refModulationStringList->splice(0, m_refModulationStringList->get_n_items(), oNewList);
    m_ModulationDropDown.set_selected(nIdx);
}

void CTemicDialog::UpdateEmulatedLabel() {
    ilr_card_type nCardType;
    ilr::CCardUID rUid;
    int nWiegand = 0;
    m_oReader.DecodeTemicEmMarine((uint32_t*)&m_aNewBlocks[0], 3, rUid);
    if (!rUid.IsEmpty())
        nCardType = ILR_CARD_EM_MARINE;
    else {
        m_oReader.DecodeTemicHid((uint32_t*)&m_aNewBlocks[0], 4, rUid, nWiegand);
        nCardType = ILR_CARD_HID;
    }

    if (!rUid.IsEmpty()) {
        if (ILR_CARD_HID == nCardType)
            m_CurrentEmuLabel.set_label(
                Glib::ustring::sprintf("%s[W%d] %s", ilr::kCardTypeNames[nCardType], nWiegand,
                                       ilr::CardUIDToStr(nCardType, rUid)));
        else
            m_CurrentEmuLabel.set_label(Glib::ustring::sprintf(
                "%s %s", ilr::kCardTypeNames[nCardType], ilr::CardUIDToStr(nCardType, rUid)));
    }
}

void CTemicDialog::UpdateNewCardTypeCtrl() {
    switch (m_nNewCardType) {
    case ILR_CARD_EM_MARINE:
        m_EmulationNotebook.set_current_page(0);
        m_EmMarineCheckButton.set_active();
        m_HidCheckButton.set_active(false);
        break;

    case ILR_CARD_HID:
        m_EmulationNotebook.set_current_page(1);
        m_HidCheckButton.set_active();
        m_EmMarineCheckButton.set_active(false);
        break;
    }
}

void CTemicDialog::UpdateNewEmMarineCtrlData(bool fSave) {
    if (fSave) {
        // Сохраняет код производителя
        auto s = m_EmFacilityEntry.get_text();
        auto fHex = (s.compare(0, 2, "0x") == 0);
        const char* pStr = s.c_str();
        unsigned long n;
        char* p;
        if (fHex) {
            pStr += 2;
            errno = 0;
            n = strtoul(pStr, &p, 16);
        }
        else {
            errno = 0;
            n = strtoul(pStr, &p, 10);
        }
        m_fEntryEmFacilityValid = (0 == errno) && (p != pStr) && (n <= 0xffff);
        m_rNewEmMarineUid.em_marine.nFacility = (uint16_t)n;
        m_fEntryEmFacilityValid = true;

        // Сохраняет серию Em-Marine
        auto value = m_EmSeriesSpinButton.get_value_as_int();
        m_rNewEmMarineUid.em_marine.nSeries = (uint8_t)value;

        // Сохраняем номер Em-Marine
        value = m_EmNumberSpinButton.get_value_as_int();
        m_rNewEmMarineUid.em_marine.nNumber = (uint16_t)value;
    }
    else {
        std::stringstream ss;
        ss.imbue(std::locale("C"));
        ss << "0x" << std::hex << std::setfill('0') << std::setw(4)
           << (uint)m_rNewEmMarineUid.em_marine.nFacility;
        m_fLockUpdate = true;
        m_EmFacilityEntry.set_text(ss.str());
        m_EmSeriesSpinButton.set_value(m_rNewEmMarineUid.em_marine.nSeries);
        m_EmNumberSpinButton.set_value(m_rNewEmMarineUid.em_marine.nNumber);
        m_fLockUpdate = false;
        m_fEntryEmFacilityValid = true;
    }
}

void CTemicDialog::UpdateNewHidCtrlData(bool fSave) {
    if (fSave) {
        // Сохраняем код производителя HID
        auto s = m_HidFacilityEntry.get_text();
        auto fHex = (s.compare(0, 2, "0x") == 0);
        const char* pStr = s.c_str();
        unsigned long long n;
        char* p;
        if (fHex) {
            pStr += 2;
            errno = 0;
            n = strtoull(pStr, &p, 16);
        }
        else {
            errno = 0;
            n = strtoull(pStr, &p, 10);
        }
        auto nFSize = (m_nNewHidWiegand - 2 - 16);
        auto nFCount = (nFSize / 8);
        if (nFSize % 8)
            ++nFCount;
        m_fEntryHidFacilityValid = (0 == errno) && (p != pStr);
        memset(&m_rNewHidUid.aBytes[2], 0, std::size(m_rNewHidUid.aBytes) - 2);
        memcpy(&m_rNewHidUid.aBytes[2], &n, nFCount);
        m_rNewHidUid.nLength = (2 + nFCount);

        // Сохраняем номер HID
        auto value = m_HidNumberSpinButton.get_value_as_int();
        *(uint16_t*)&m_rNewHidUid.aBytes[0] = (uint16_t)value;
    }
    else {
        m_fLockUpdate = true;
        m_HidWiegandSpinButton.set_value(m_nNewHidWiegand);
        m_fLockUpdate = false;

        auto nFSize = (m_nNewHidWiegand - 2 - 16);
        auto nFCount = (nFSize / 8);
        if (nFSize % 8)
            ++nFCount;

        if (nFSize > 0) {
            std::stringstream ss;
            ss.imbue(std::locale("C"));
            ss << std::hex << std::showbase << std::setfill('0');
            for (size_t i = nFCount; i-- > 0;)
                ss << std::setw(2) << (uint)m_rNewHidUid.aBytes[2 + i] << std::noshowbase;

            m_fLockUpdate = true;
            m_HidFacilityEntry.set_text(ss.str());
            m_fLockUpdate = false;
            m_HidFacilityEntry.set_visible();
        }
        else
            m_HidFacilityEntry.set_visible(false);

        m_fLockUpdate = true;
        m_HidNumberSpinButton.set_value(*(uint16_t*)&m_rNewHidUid.aBytes[0]);
        m_fLockUpdate = false;
        m_fEntryHidFacilityValid = true;
    }
}

void CTemicDialog::MessageDialog(const std::string& sMessage, Gtk::MessageType nType,
                                 Gtk::ButtonsType nButtons,
                                 Gio::ActionMap::ActivateWithIntParameterSlot&& slot) {
    if (nullptr == m_refDialog)
        m_refDialog.reset(new Gtk::MessageDialog("Демо", false, nType, nButtons, true));
    m_refDialog->set_transient_for(*this);
    m_refDialog->set_message(sMessage);
    m_refDialog->show();
    if (slot.empty())
        m_refDialog->signal_response().connect(
            sigc::mem_fun(*this, &CTemicDialog::on_dialog_response));
    else
        m_refDialog->signal_response().connect(slot);
}
