#include "CReaderDialog.h"

#include <fstream>
#include <iostream>
#include <sstream>  // для std::stringstream

#include "CAppSettings.h"
#include "utils.h"

#define HAS_STYLE_PROVIDER_ADD_PROVIDER_FOR_DISPLAY GTKMM_CHECK_VERSION(4, 9, 1)

const char kAutoValue[] = {"Авто"};

CReaderDialog::CReaderDialog() :
    m_nModel(ILR_READER_MODEL_UNKNOWN),
    m_nReaderSn(0),
    m_fCanRwMfClassic(false),
    m_fCanRwMfPlus(false),
    m_fCanRwMfUltralight(false),
    m_fCanRwTemic(false),
    m_fLockUpdate(false),
    m_VBox(Gtk::Orientation::VERTICAL),
    m_StatusFrame("Состояние подключения"),
    m_ConnectButton("Подключиться"),
    m_DisconnectButton("Отключиться"),
    m_CardFrame("Текущая карта"),
    m_CardRefreshButton("Обновить"),
    m_OpenButton("Открыть..."),
    m_SlButton("Изменить уровень безопасности..."),
    m_TemicFrame("Параметры поиска карт Temic"),
    m_TsFrame("Тип и скорость"),
    m_TsComboBox(true),
    m_PasswordCheckButton("Пароль"),
    m_PasswEditButton("..."),
    m_TemicButtonsBox(Gtk::Orientation::VERTICAL),
    m_TemicScanButton("Сканировать"),
    m_TemicInitButton("Инициализировать Temic"),
    m_ScanTemicCheckButton("Автоматически сканировать карты Temic"),
    m_McKeysButton("Ключи Classic..."),
    m_MpKeysButton("Ключи Plus...") {
    set_destroy_with_parent(true);

    set_title("Считыватель");
    set_default_size(800, 250);

    m_VBox.set_margin(5);
    m_VBox.set_expand();
    set_child(m_VBox);

    m_StatusFrame.set_expand(false);
    m_VBox.append(m_StatusFrame);
    m_StatusBox.set_margin(5);
    m_StatusBox.set_spacing(5);
    m_StatusBox.set_expand();
    m_StatusFrame.set_child(m_StatusBox);

    m_StatusBox.append(m_StatusLabel);
    m_StatusLabel.set_hexpand(true);
    m_StatusLabel.set_halign(Gtk::Align::START);
    // m_StatusLabel.set_selectable();
    m_ConnectButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_button_connect));
    m_StatusBox.append(m_ConnectButton);
    m_DisconnectButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_button_disconnect));
    m_StatusBox.append(m_DisconnectButton);

    m_VBox.append(m_CardFrame);
    m_CardFrame.set_child(m_CardGrid);
    m_CardGrid.set_margin(5);
    m_CardGrid.set_row_spacing(5);
    m_CardGrid.set_column_spacing(5);
    m_CardGrid.attach(m_CardLabel, 0, 0, 1, 2);
    m_CardLabel.set_hexpand();
    m_CardGrid.attach(m_CardRefreshButton, 1, 0);
    m_CardGrid.attach(m_OpenButton, 2, 0);
    m_CardGrid.attach(m_SlButton, 1, 1, 2);
    m_CardLabel.set_halign(Gtk::Align::START);
    m_CardLabel.set_selectable();

    m_CardRefreshButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_button_card_refresh));
    m_OpenButton.signal_clicked().connect(sigc::mem_fun(*this, &CReaderDialog::on_button_open));
    m_SlButton.signal_clicked().connect(sigc::mem_fun(*this, &CReaderDialog::on_button_sl));

    m_VBox.append(m_TemicFrame);
    m_TemicFrame.set_child(m_TemicGrid);
    m_TemicGrid.set_margin(5);
    m_TemicGrid.set_row_spacing(5);
    m_TemicGrid.set_column_spacing(5);
    m_TemicGrid.attach(m_TsFrame, 0, 0);
    m_TemicGrid.attach(m_PasswordFrame, 1, 0);
    m_TemicGrid.attach(m_TemicButtonsBox, 2, 0, 1, 3);
    m_TemicGrid.attach(m_ScanTemicCheckButton, 0, 1, 3);
    m_TsFrame.set_margin(5);
    m_TsFrame.set_hexpand(false);
    m_TsFrame.set_child(m_TsComboBox);
    m_TsComboBox.set_margin(5);
    m_TsComboBox.append(kAutoValue);
    m_TsComboBox.append("2 0 Em-Marine");
    m_TsComboBox.append("1 1");
    m_TsComboBox.append("2 1");
    m_TsComboBox.append("1 0");
    m_TsComboBox.append("3 0 HID");
    m_TsComboBox.append("4 0");
    m_TsComboBox.signal_editing_done().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_ts_editing_done));
    m_TsComboBox.signal_changed().connect(sigc::mem_fun(*this, &CReaderDialog::on_ts_selected));
    m_PasswordFrame.set_child(m_PasswordBox);
    m_PasswordFrame.set_hexpand(true);
    m_PasswordFrame.set_label_widget(m_PasswordCheckButton);
    m_PasswordBox.set_margin(5);
    m_PasswordBox.set_spacing(5);
    m_PasswordBox.append(m_PasswordDropDown);
    m_PasswordBox.append(m_PasswEditButton);
    m_PasswordDropDown.set_hexpand(true);
    m_refPasswordStringList = Gtk::StringList::create({});
    m_PasswordDropDown.set_model(m_refPasswordStringList);
    m_PasswordDropDown.property_selected().signal_changed().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_password_selected_changed));
    m_TemicButtonsBox.set_margin(5);
    m_TemicButtonsBox.set_spacing(5);
    m_TemicButtonsBox.append(m_TemicScanButton);
    m_TemicButtonsBox.append(m_TemicInitButton);

    m_TemicScanButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_button_scan_temic));
    m_TemicInitButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_button_init_temic));
    m_PasswordCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_button_toggle_password));
    m_PasswEditButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_button_edit_passwords));
    m_ScanTemicCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_button_toggle_scan_temic));

    // Load extra CSS file.
    m_refCssProvider = Gtk::CssProvider::create();
#if HAS_STYLE_PROVIDER_ADD_PROVIDER_FOR_DISPLAY
    Gtk::StyleProvider::add_provider_for_display(get_display(), m_refCssProvider,
                                                 GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
#else
    Gtk::StyleContext::add_provider_for_display(get_display(), m_refCssProvider,
                                                GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
#endif
    std::string sCSS =
        ".password {"
        "  font-family:monospace; "
        "}"
        ".password popover {"
        "  font-family:monospace; "
        "}";
    m_refCssProvider->load_from_data(sCSS);

    m_PasswordDropDown.add_css_class("password");

    m_VBox.append(m_BottomBox);
    m_BottomBox.set_margin(5);
    m_BottomBox.set_spacing(5);
    m_BottomBox.append(m_McKeysButton);
    m_BottomBox.append(m_MpKeysButton);
    m_McKeysButton.signal_clicked().connect(sigc::mem_fun(*this, &CReaderDialog::on_button_mckeys));
    m_MpKeysButton.signal_clicked().connect(sigc::mem_fun(*this, &CReaderDialog::on_button_mpkeys));

    m_oDisp.connect(sigc::mem_fun(*this, &CReaderDialog::on_ilr));
    signal_hide().connect(sigc::mem_fun(*this, &CReaderDialog::on_hide));
}

CReaderDialog::~CReaderDialog() {
}

void CReaderDialog::Init(ilr::CReader& oReader) {
    m_oReader = std::move(oReader);
    m_oReader.EnableMessageQueue();
    m_oReader.SetMessageCallback(&on_ilr_message, this);
    ilr_reader_info rInfo;
    m_oReader.GetReaderInfo(rInfo);
    m_nModel = rInfo.nModel;
    m_nReaderSn = rInfo.nSn;
    m_refMfSet = nullptr;
    m_refTmcSet = nullptr;
    m_fCanRwMfClassic = false;
    m_fCanRwMfPlus = false;
    m_fCanRwMfUltralight = false;
    m_fCanRwTemic = false;
    switch (rInfo.nModel) {
    case ILR_READER_MODEL_Z2_RDALL:
    case ILR_READER_MODEL_Z2_E_HTZ_RF:
        m_fCanRwTemic = true;
        m_fCanRwMfUltralight = true;
        m_refTmcSet.reset(new CTemicReaderSettings);
        m_refTmcSet->Load(GetSettingFilePath());
        m_refTmcPasswords.reset(new CTemicPasswordsSettings());
        m_refTmcPasswords->Load(GetTemicPasswordsFilePath());
        break;

    case ILR_READER_MODEL_Z2_MF:
    case ILR_READER_MODEL_MATRIX3_NET:
    case ILR_READER_MODEL_CP_Z2_MFI:
    case ILR_READER_MODEL_Z2_MF_CCID:
        m_fCanRwMfClassic = true;
        m_fCanRwMfUltralight = true;
        m_refMfSet.reset(new CMifareReaderSettings);
        m_refMfSet->Load(GetSettingFilePath());
        m_refMcKeys.reset(new CMfClassicKeysSettings);
        m_refMcKeys->Load(GetMcKeysFilePath());
        break;

    case ILR_READER_MODEL_Z2_MFI:
    case ILR_READER_MODEL_MATRIX6:
        m_fCanRwMfClassic = true;
        m_fCanRwMfPlus = true;
        m_fCanRwMfUltralight = true;
        m_refMfSet.reset(new CMifareReaderSettings);
        m_refMfSet->Load(GetSettingFilePath());

        m_refMcKeys.reset(new CMfClassicKeysSettings);
        m_refMcKeys->Load(GetMcKeysFilePath());
        m_refMpKeys.reset(new CMfPlusKeysSettings);
        m_refMpKeys->Load(GetMpKeysFilePath());
        break;
    }
    m_TemicFrame.set_visible(m_fCanRwTemic);
    m_McKeysButton.set_visible(m_fCanRwMfClassic);
    m_MpKeysButton.set_visible(m_fCanRwMfPlus);
    m_SlButton.set_visible(m_fCanRwMfPlus);

    UpdateReaderInfo();
    UpdateConnectionStatus();
    UpdateCurrentCard();

    if (m_fCanRwTemic) {
        m_oReader.SetScanTemic(m_refTmcSet->m_fTemicAutoScan);
        m_ScanTemicCheckButton.set_active(m_refTmcSet->m_fTemicAutoScan);
        UpdateTemicPasswordDropDown();
        m_PasswordCheckButton.set_active(m_refTmcSet->m_fTemicPswEnabled);
        if (m_refTmcSet->m_fTemicPswEnabled)
            m_oReader.LoadTemicPassword(m_refTmcSet->m_nTemicPassword);
        else
            m_oReader.LoadTemicPassword(-1);
        switch (m_refTmcSet->m_nTemicScanParam) {
        case -1:
            m_TsComboBox.set_active(0);
            break;

        case 2:
            m_TsComboBox.set_active(1);
            break;

        case 0x0101:
            m_TsComboBox.set_active(2);
            break;

        case 0x0102:
            m_TsComboBox.set_active(3);
            break;

        case 1:
            m_TsComboBox.set_active(4);
            break;

        case 3:
            m_TsComboBox.set_active(5);
            break;

        case 4:
            m_TsComboBox.set_active(6);
            break;

        default:
            m_TsComboBox.set_active_text(Glib::ustring::sprintf(
                "%u %u", (uint)TYPE_FROM_TEMIC_SCAN_PARAM(m_refTmcSet->m_nTemicScanParam),
                (uint)SPEED_FROM_TEMIC_SCAN_PARAM(m_refTmcSet->m_nTemicScanParam)));
            break;
        }
    }
}

void CReaderDialog::on_button_connect() {
    m_oReader.Connect();
}

void CReaderDialog::on_button_disconnect() {
    m_oReader.Disconnect();
}

void CReaderDialog::on_button_card_refresh() {
    m_oReader.Scan(true);
}

void CReaderDialog::on_button_open() {
    switch (m_rCardInfo.nType) {
    case ILR_CARD_MF_ULTRALIGHT:
        ShowMfUltralightDialog();
        break;

    case ILR_CARD_MF_MINI:
    case ILR_CARD_MF_CLASSIC_1K:
    case ILR_CARD_MF_CLASSIC_2K:
    case ILR_CARD_MF_CLASSIC_4K:
        ShowMfClassicDialog();
        break;

    case ILR_CARD_MF_PLUS_1K:
    case ILR_CARD_MF_PLUS_2K:
    case ILR_CARD_MF_PLUS_4K:
    case ILR_CARD_SMART_MX_MF1K:
    case ILR_CARD_SMART_MX_MF4K:
        switch (m_rCardInfo.nSL) {
        case ILR_MF_PLUS_SL_SL1:
            ShowMfClassicDialog();
            break;

        case ILR_MF_PLUS_SL_SL3:
            ShowMfPlusDialog();
            break;
        }
        break;

    case ILR_CARD_TEMIC:
        ShowTemicDialog();
        break;
    }
}

void CReaderDialog::on_button_sl() {
    if (nullptr == m_refMfSlDialog)
        m_refMfSlDialog.reset(new CMfSecurityLevelDialog());
    m_refMfSlDialog->set_transient_for(*this);
    m_refMfSlDialog->set_modal();
    m_refMfSlDialog->set_hide_on_close();
    m_refMfSlDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_mfsl_dialog_hide));

    m_refMfSlDialog->Init(m_oReader);
    m_refMfSlDialog->set_visible(true);
    // TODO: Дописать
}

void CReaderDialog::on_mfsl_dialog_hide() {
    m_refMfSlDialog = nullptr;
}

void CReaderDialog::on_ts_editing_done() {
    int nScanParam;
    if (TryParseTemicScanParam(m_TsComboBox.get_active_text().c_str(), nScanParam))
        m_refTmcSet->SetTemicScanParam(nScanParam);
    else
        m_TsComboBox.set_active_text(TemicScanParamToString(m_refTmcSet->m_nTemicScanParam));
}

void CReaderDialog::on_ts_selected() {
    int nScanParam;
    if (TryParseTemicScanParam(m_TsComboBox.get_active_text().c_str(), nScanParam))
        m_refTmcSet->SetTemicScanParam(nScanParam);
}

void CReaderDialog::on_button_mckeys() {
    if (nullptr == m_refMcKeysDialog)
        m_refMcKeysDialog.reset(new CReaderMcKeysDialog());
    m_refMcKeysDialog->set_transient_for(*this);
    m_refMcKeysDialog->set_modal();
    m_refMcKeysDialog->set_hide_on_close();
    m_refMcKeysDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_mckeys_dialog_hide));

    m_refMcKeysDialog->Init(m_oReader, m_refMfSet);
    m_refMcKeysDialog->set_visible(true);
}

void CReaderDialog::on_mckeys_dialog_hide() {
    m_refMcKeysDialog = nullptr;
}

void CReaderDialog::on_button_toggle_password() {
    auto fEnabled = m_PasswordCheckButton.get_active();
    if (m_refTmcSet->m_fTemicPswEnabled != fEnabled) {
        m_refTmcSet->SetTemicPswEnabled(fEnabled);
        m_oReader.LoadTemicPassword(fEnabled ? m_refTmcSet->m_nTemicPassword : -1);
    }
}

void CReaderDialog::on_button_edit_passwords() {
    if (nullptr == m_refTemicPasswordsDialog)
        m_refTemicPasswordsDialog.reset(new CTemicPasswordsDialog());
    m_refTemicPasswordsDialog->set_transient_for(*this);
    m_refTemicPasswordsDialog->set_modal();
    m_refTemicPasswordsDialog->set_hide_on_close();
    m_refTemicPasswordsDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_temic_passwords_dialog_hide));

    ilr_reader_info rInfo;
    m_oReader.GetReaderInfo(rInfo);
    ilr::CILR oILR;
    auto oReader = oILR.GetReader(rInfo.nPortType, rInfo.pszPortName);
    m_refTemicPasswordsDialog->Init(oReader, m_refTmcPasswords, m_refTmcSet->m_nTemicPassword);
    m_refTemicPasswordsDialog->set_visible(true);
}

void CReaderDialog::on_temic_passwords_dialog_hide() {
    if (m_refTemicPasswordsDialog->m_fAccept) {
        UpdateTemicPasswordDropDown();
    }
    m_refTemicPasswordsDialog = nullptr;
}

void CReaderDialog::on_password_selected_changed() {
    if (m_fLockUpdate)
        return;
    auto nIdx = m_PasswordDropDown.get_selected();
    if (nIdx != GTK_INVALID_LIST_POSITION) {
        auto nPassword = m_refTmcPasswords->m_oPasswords[nIdx].m_nPassword;
        m_refTmcSet->SetTemicPassword(nPassword);
        if (m_refTmcSet->m_fTemicPswEnabled)
            m_oReader.LoadTemicPassword(nPassword);
    }
}

void CReaderDialog::on_button_scan_temic() {
    try {
        m_oReader.ScanTemic(m_refTmcSet->m_nTemicScanParam);
    }
    catch (const std::exception& e) {
        // std::cerr << e.what() << '\n';
        MessageDialog(e.what(), Gtk::MessageType::ERROR);
    }
}

void CReaderDialog::on_button_init_temic() {
    MessageDialog("Действительно хотите инициализировать Temic?", Gtk::MessageType::QUESTION,
                  Gtk::ButtonsType::OK_CANCEL,
                  sigc::mem_fun(*this, &CReaderDialog::on_confirm_init_temic_dialog_response));
}

void CReaderDialog::on_confirm_init_temic_dialog_response(int response_id) {
    m_refDialog = nullptr;
    if (response_id != GTK_RESPONSE_OK)
        return;
    try {
        const uint32_t nConfig = 0x40801400;
        int nScanParam = m_refTmcSet->m_nTemicScanParam;
        if (-1 == nScanParam)
            nScanParam = 2;
        m_oReader.WriteTemic(0, &nConfig, 1, false, nScanParam);
    }
    catch (const std::exception& e) {
        // std::cerr << e.what() << '\n';
        MessageDialog(e.what(), Gtk::MessageType::ERROR);
    }
}

void CReaderDialog::on_button_toggle_scan_temic() {
    m_refTmcSet->m_fTemicAutoScan = m_ScanTemicCheckButton.get_active();
    m_oReader.SetScanTemic(m_refTmcSet->m_fTemicAutoScan);
}

void ILR_CALL CReaderDialog::on_ilr_message(ilr_reader_msg nMsg, const void* pMsgData,
                                            void* pUserData) {
    auto p = (CReaderDialog*)pUserData;
    p->m_oDisp.emit();
}

void CReaderDialog::on_button_mpkeys() {
    if (nullptr == m_refMpKeysDialog)
        m_refMpKeysDialog.reset(new CReaderMpKeysDialog());
    m_refMpKeysDialog->set_transient_for(*this);
    m_refMpKeysDialog->set_modal();
    m_refMpKeysDialog->set_hide_on_close();
    m_refMpKeysDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_mpkeys_dialog_hide));

    m_refMpKeysDialog->Init(m_oReader, m_refMfSet);
    m_refMpKeysDialog->set_visible(true);
}

void CReaderDialog::on_mpkeys_dialog_hide() {
    m_refMpKeysDialog = nullptr;
}

void CReaderDialog::on_mfultralight_dialog_hide() {
    m_refMfUltralightDialog = nullptr;
}

void CReaderDialog::on_mfclassic_dialog_hide() {
    m_refMfClassicDialog = nullptr;
}

void CReaderDialog::on_mfplus_dialog_hide() {
    m_refMfPlusDialog = nullptr;
}

void CReaderDialog::on_temic_dialog_hide() {
    m_refTemicDialog = nullptr;
}

void CReaderDialog::on_dialog_response(int) {
    m_refDialog = nullptr;
}

void CReaderDialog::on_hide() {
    m_oReader = nullptr;
    SaveSettings();
}

void CReaderDialog::on_ilr() {
    ilr_reader_msg nMsg;
    const void* pMsgData;
    while (m_oReader.GetMessage(nMsg, pMsgData)) {
        switch (nMsg) {
        case ILR_READER_MSG_CONNECTION_CHANGED:
            UpdateConnectionStatus();
            if (m_oReader.GetConnectionStatus() == ILR_CONNECTION_CONNECTED)
                UpdateReaderInfo();
            break;

        case ILR_READER_MSG_CARD_FOUND:
        case ILR_READER_MSG_CARD_LOST:
            UpdateCurrentCard();
            break;
        }
    }
}

void CReaderDialog::ShowMfUltralightDialog() {
    if (nullptr == m_refMfUltralightDialog)
        m_refMfUltralightDialog.reset(new CMfUltralightDialog());
    m_refMfUltralightDialog->set_transient_for(*this);
    m_refMfUltralightDialog->set_modal();
    m_refMfUltralightDialog->set_hide_on_close();
    m_refMfUltralightDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_mfultralight_dialog_hide));

    m_refMfUltralightDialog->Init(m_oReader);
    m_refMfUltralightDialog->set_visible(true);
}

void CReaderDialog::ShowMfClassicDialog() {
    if (nullptr == m_refMfClassicDialog)
        m_refMfClassicDialog.reset(new CMifareClassicDialog());
    m_refMfClassicDialog->set_transient_for(*this);
    m_refMfClassicDialog->set_modal();
    m_refMfClassicDialog->set_hide_on_close();
    m_refMfClassicDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_mfclassic_dialog_hide));

    m_refMfClassicDialog->Init(m_oReader, m_refMfSet, m_refMcKeys);
    m_refMfClassicDialog->set_visible(true);
}

void CReaderDialog::ShowMfPlusDialog() {
    if (nullptr == m_refMfPlusDialog)
        m_refMfPlusDialog.reset(new CMifarePlusDialog());
    m_refMfPlusDialog->set_transient_for(*this);
    m_refMfPlusDialog->set_modal();
    m_refMfPlusDialog->set_hide_on_close();
    m_refMfPlusDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_mfplus_dialog_hide));

    m_refMfPlusDialog->Init(m_oReader, m_refMfSet, m_refMpKeys);
    m_refMfPlusDialog->set_visible(true);
}

void CReaderDialog::ShowTemicDialog() {
    if (nullptr == m_refTemicDialog)
        m_refTemicDialog.reset(new CTemicDialog());
    m_refTemicDialog->set_transient_for(*this);
    m_refTemicDialog->set_modal();
    m_refTemicDialog->set_hide_on_close();
    m_refTemicDialog->signal_hide().connect(
        sigc::mem_fun(*this, &CReaderDialog::on_temic_dialog_hide));

    m_refTemicDialog->Init(m_oReader, m_refTmcPasswords);
    m_refTemicDialog->set_visible(true);
}

void CReaderDialog::MessageDialog(const std::string& sMessage, Gtk::MessageType nType,
                                  Gtk::ButtonsType nButtons,
                                  Gio::ActionMap::ActivateWithIntParameterSlot&& slot) {
    if (nullptr == m_refDialog)
        m_refDialog.reset(new Gtk::MessageDialog("Демо", false, nType, nButtons, true));
    m_refDialog->set_transient_for(*this);
    m_refDialog->set_message(sMessage);
    m_refDialog->show();
    if (slot.empty())
        m_refDialog->signal_response().connect(
            sigc::mem_fun(*this, &CReaderDialog::on_dialog_response));
    else
        m_refDialog->signal_response().connect(slot);
}

void CReaderDialog::UpdateConnectionStatus() {
    auto nStatus = m_oReader.GetConnectionStatus();
    switch (nStatus) {
    case ILR_CONNECTION_CONNECTED:
        m_StatusLabel.set_text("Подключён");
        break;

    case ILR_CONNECTION_DISCONNECTED:
        m_StatusLabel.set_text("Отключён");
        break;

    case ILR_CONNECTION_CONNECTING:
        m_StatusLabel.set_text("Подключение...");
        break;
    }
}

void CReaderDialog::UpdateReaderInfo() {
    ilr_reader_info rInfo;
    m_oReader.GetReaderInfo(rInfo);
    std::stringstream ss;
    ss << "Считыватель (" << rInfo.pszPortName << "): " << ilr::kReaderModelNames[rInfo.nModel];
    if (rInfo.nSn != -1)
        ss << " с/н: " << rInfo.nSn;
    if (rInfo.nFwVersion != 0)
        ss << " прошивка:" << ilr::ReaderVersionToStr(rInfo.nFwVersion);
    set_title(ss.str());
}

void CReaderDialog::UpdateCurrentCard() {
    m_oReader.GetCardInfo(m_rCardInfo);
    if (m_rCardInfo.rUID.nLength != 0) {
        std::stringstream ss;
        ss << ilr::kCardTypeNames[m_rCardInfo.nType] << ' '
           << ilr::CardUIDToStr(m_rCardInfo.nType, m_rCardInfo.rUID);
        if (m_rCardInfo.nMpType != ILR_MF_PLUS_UNKNOWN)
            ss << ' ' << ilr::kMpTypeNames[m_rCardInfo.nMpType];
        if (m_rCardInfo.nSL != ILR_MF_PLUS_SL_UNKNOWN)
            ss << " SL" << std::dec << (int)m_rCardInfo.nSL;
        if (m_rCardInfo.rUID2.nLength != 0)
            ss << " (" << ilr::kCardTypeNames[m_rCardInfo.nType2] << ' '
               << ilr::CardUIDToStr(m_rCardInfo.nType2, m_rCardInfo.rUID2) << ')';
        m_CardLabel.set_label(ss.str());

        auto fOpen = false;
        auto fSL = false;
        switch (m_rCardInfo.nType) {
        case ILR_CARD_MF_ULTRALIGHT:
            fOpen = m_fCanRwMfUltralight;
            break;

        case ILR_CARD_MF_MINI:
        case ILR_CARD_MF_CLASSIC_1K:
        case ILR_CARD_MF_CLASSIC_2K:
        case ILR_CARD_MF_CLASSIC_4K:
            fOpen = m_fCanRwMfClassic;
            break;

        case ILR_CARD_MF_PLUS_1K:
        case ILR_CARD_MF_PLUS_2K:
        case ILR_CARD_MF_PLUS_4K:
        case ILR_CARD_SMART_MX_MF1K:
        case ILR_CARD_SMART_MX_MF4K:
            switch (m_rCardInfo.nSL) {
            case ILR_MF_PLUS_SL_SL0:
            case ILR_MF_PLUS_SL_SL2:
                fSL = m_fCanRwMfPlus;
                break;

            case ILR_MF_PLUS_SL_SL1:
                fOpen = m_fCanRwMfClassic;
                fSL = m_fCanRwMfPlus;
                break;

            case ILR_MF_PLUS_SL_SL3:
                fOpen = m_fCanRwMfPlus;
                break;
            }
            break;

        case ILR_CARD_TEMIC:
            fOpen = m_fCanRwTemic;
            break;
        }
        m_OpenButton.set_sensitive(fOpen);
        m_SlButton.set_sensitive(fSL);
    }
    else {
        m_CardLabel.set_label("Нет карты");
        m_SlButton.set_sensitive(false);
    }
}

void CReaderDialog::UpdateTemicPasswordDropDown() {
    Glib::OptionGroup::vecustrings list;
    for (auto& r : m_refTmcPasswords->m_oPasswords)
        list.emplace_back(TemicPasswordToString(r.m_nPassword));

    m_fLockUpdate = true;
    m_refPasswordStringList->splice(0, m_refPasswordStringList->get_n_items(), list);

    auto nIdx = FindTemicPassword(m_refTmcPasswords->m_oPasswords, m_refTmcSet->m_nTemicPassword);
    if (nIdx != -1)
        m_PasswordDropDown.set_selected(nIdx);
    m_fLockUpdate = false;
}

bool CReaderDialog::TryParseTemicScanParam(const char* pStr, int& nScanParam) {
    if (strcmp(pStr, kAutoValue) == 0) {
        nScanParam = -1;  // Авто
        return true;
    }
    uint nType, nSpeed;
    if ((sscanf(pStr, "%u %u", &nType, &nSpeed) == 2) &&
        (nType < std::numeric_limits<uint8_t>::max()) &&
        (nSpeed < std::numeric_limits<uint8_t>::max())) {
        nScanParam = MAKE_TEMIC_SCAN_PARAM(nType, nSpeed);
        return true;
    }
    return false;
}

std::string CReaderDialog::TemicScanParamToString(int nScanParam) {
    if (-1 == nScanParam)
        return kAutoValue;
    std::stringstream ss;
    ss << std::dec << TYPE_FROM_TEMIC_SCAN_PARAM(nScanParam) << ' '
       << SPEED_FROM_TEMIC_SCAN_PARAM(nScanParam);
    return ss.str();
}

void CReaderDialog::SaveSettings() {
    if ((m_refMfSet != nullptr) && m_refMfSet->m_fModified)
        m_refMfSet->Save(GetSettingFilePath());
    else if ((m_refTmcSet != nullptr) && m_refTmcSet->m_fModified)
        m_refTmcSet->Save(GetSettingFilePath());
}

std::filesystem::path CReaderDialog::GetSettingFilePath() const {
    char szCfgPath[PATH_MAX];
    auto count = readlink("/proc/self/exe", szCfgPath, std::size(szCfgPath) - 1);
    szCfgPath[count] = '\0';
    std::stringstream ss;
    ss << szCfgPath << "reader_" << m_nReaderSn << ".txt";
    return ss.str();
}
