#pragma once

#include <gdkmm/rgba.h>
#include <gtkmm.h>
#include <gtkmm/drawingarea.h>

#include <map>
#include <string>
#include <vector>

class CCardDataGrid : public Gtk::Scrollable, public Gtk::DrawingArea {
public:
    // Формат байта
    enum byte_format {
        BYTE_FORMAT_UNDEF,
        BYTE_FORMAT_HEX,  // Шестнадцатеричный
        BYTE_FORMAT_DEC,  // Десятичный
        BYTE_FORMAT_OCT,  // Восьмеричный
        BYTE_FORMAT_BIN,  // Двоичный
        BYTE_FORMAT_SIZE  // Размер списка
    };

    typedef struct byte_attrs {
        uint m_nCount;
        byte_format m_nFormat;
        Gdk::RGBA m_ForeColor;
        Gdk::RGBA m_BgColor;

        byte_attrs(uint nCount, byte_format nFormat, Gdk::RGBA ForeColor, Gdk::RGBA BgColor) :
            m_nCount(nCount),
            m_nFormat(nFormat),
            m_ForeColor(ForeColor),
            m_BgColor(BgColor) {
        }
    } byte_attrs;

    using CByteAttrsMap = std::map<uint, byte_attrs>;
    using CSectorSizeMap = std::map<uint, uint>;

    using type_signal_checkbox_toggle = sigc::signal<void(uint nBlockIdx)>;
    using type_signal_validate_byte = sigc::signal<void(uint nAddress, uint8_t& b)>;
    using type_signal_get_byte_access = sigc::signal<uint(uint nAddress)>;
    using type_signal_select_cell = sigc::signal<void()>;

protected:
    // Флаги состояния флажка checkbox
    typedef enum {
        CHECKBOX_F_VISIBLE = 1,   // Видимый
        CHECKBOX_F_ACTIVE = 2,    // Отмеченный
        CHECKBOX_F_SENSITIVE = 4  // Доступный
    } checkbox_flags;

public:
    // Количество фиксированных строк (горизонтальный заголовок)
    static constexpr int kFixedRows = 1;
    // Внутренние горизонтальные отступы ячейки (слева и справа)
    static constexpr int kFixedCellHPadding = 3;
    // Внутренние горизонтальные отступы ячейки (слева и справа)
    static constexpr int kDataCellHPadding = 8;
    // Внутренние вертикальные отступы ячейки (сверху и снизу)
    static constexpr int kCellVPadding = 2;
    // Отступ между элементами в ячейке (между флажком и текстовой меткой)
    static constexpr int kCellSpacing = 3;

public:
    CCardDataGrid();
    virtual ~CCardDataGrid();

    // Устанавливает ссылку на данные карты
    void set_data_pointer(uint8_t* pData, uint nSize);
    void set_old_data_pointer(uint8_t* pData, uint nSize);
    // Устанавливает размер блока в байтах (количество столбцов с байтами)
    void set_block_size(uint nSize);
    // Добавляет размер сектора, который начинается с блока nFromBlockIdx
    void add_sector_size(uint nFromBlockIdx, uint nSize);
    void clear_sector_sizes();
    // Устанавливает названия фиксированных столбцов, разделённые символом '\n'
    void set_fix_col_titles(const char** pTitles, size_t nCount);
    // Добавляет диапазон байт с особым форматированием
    void add_or_set_byte_attrs(uint nAddress, uint nCount, byte_format nFormat, Gdk::RGBA foreColor,
                               Gdk::RGBA bgColor, bool fRedraw = true);
    void remove_byte_attrs(uint nAddress);
    void clear_byte_attrs();
    // Устанавливает видимый диапазон. Если nBlockCount равно 0, то не используется
    void set_visible_range(uint nBlockIdx, uint nBlockCount, bool fUpdate = true);
    inline uint get_first_visible_block() const;
    inline uint get_visible_block_count() const;
    // Обновляет параметры
    void update();
    // Перерисовать
    inline void redraw();

    // Возвращает ячейку по адресу
    void address_to_cell(uint nAddress, int& nCol, int& nRow) const;
    // Возвращает адрес байта по адресу ячейки
    bool cell_to_address(int nCol, int nRow, uint& nAddress) const;
    // Возвращает текст ячейки
    std::string address_to_text(uint nAddress, bool fOldData = false) const;
    // Возвращает длину текста в ячейке
    int get_address_text_length(uint nAddress) const;

    // Возвращает текущий столбец
    inline int get_col() const;
    // Возвращает текущую строку
    inline int get_row() const;
    // Возвращает количество столбцов
    inline int get_col_count() const;
    // Возвращает количество строк
    inline int get_row_count() const;
    // Устанавливает ширину столбца
    void set_col_width(int nCol, int nWidth);
    // Устанавливает ширину столбца по умолчанию
    inline void set_default_col_width(int nWidth);
    // Делает ячейку текущей
    void set_current_cell(int nCol, int nRow, bool fScroll = true, bool fNotify = false);
    // Возвращает ячейку в координатах x, y
    void mouse_to_cell(int x, int y, int& nCol, int& nRow);
    // Возвращает адрес следующей ячейки
    bool get_next_tab_cell(uint nCurrCol, uint nCurrRow, int& nCol, int& nRow,
                           bool fPrevious = false);
    int get_next_tab_col(uint nCurrCol, bool fPrevious = false);
    int get_next_tab_row(uint nCurrRow, bool fPrevious = false);
    // Возвращает номер байта по номеру столбца
    int col_to_byte_idx(int nCol) const;
    // Возвращает номер столбца по номеру байта
    int byte_idx_to_col(uint nByteIdx) const;
    // Возвращает номер блока по номеру строки
    int row_to_block_idx(int nRow) const;
    // Возвращает номер строки по номеру блока
    int block_idx_to_row(uint nBlockIdx) const;

    // Устанавливает общий формат байтов
    void set_byte_format(byte_format nFormat, bool fUpdateColSizes = true);
    // Возвращает общий формат байтов
    inline byte_format get_byte_format() const;
    // Обновляет размеры столбцов байт
    void update_data_col_sizes();
    void set_checkbox_visible(uint nBlockIdx, bool fVisible = true);
    void set_checkbox_active(uint nBlockIdx, bool fActive = true);
    void set_checkbox_sensitive(uint nBlockIdx, bool fSensitive = true);
    void set_range_checkbox(uint nBlockIdx, uint nCount, bool fVisible = true, bool fActive = false,
                            bool fSensitive = true);
    bool get_checkbox_active(uint nBlockIdx) const;
    bool get_checkbox_sensitive(uint nBlockIdx) const;
    // Включает/выключает режим редактирования
    void EnableEditMode(bool fEnable = true);

    inline type_signal_checkbox_toggle signal_checkbox_toggle();
    inline type_signal_validate_byte signal_validate_byte();
    inline type_signal_get_byte_access signal_get_byte_access();
    inline type_signal_select_cell signal_select_cell();

protected:
    void on_draw(const Cairo::RefPtr<Cairo::Context>& cr, int width, int height);
    bool on_key_pressed(guint keyval, guint keycode, Gdk::ModifierType state);
    void on_mouse_click(int n_press, double x, double y);
    void on_vscroll();
    void on_hscroll();
    void on_resize(int width, int height);
    void on_property_vadjustment_change();
    void on_property_hadjustment_change();

    void set_fixed_col_count(uint nCount);
    // Устанавливает количество столбцов
    void set_col_count(uint nCount);
    // Устанавливает количество строк
    void set_row_count(uint nCount);
    uint get_block_count() const;
    uint get_total_block_count() const;
    // uint get_end_visible_address() const;
    void get_visible_addresses(uint& nBeginAddress, uint& nEndAddress) const;
    uint calc_sector_count(uint nBlockCount, uint& nMaxSBlocks) const;
    // Возвращает количество столбцов, которые умещаются в заданной ширине nWidth
    // dx - общая ширина умещающихся столбцов
    int get_cols_per_width(int nWidth, uint nStartCol, uint nEndCol, int& dx) const;
    // Обновляет размер холста
    void update_canvas_size();
    // Обновляет размеры фиксированных столбцов
    void Update_fix_col_widths();
    // Обновляет параметры полос прокрутки
    void update_scroll_info(bool fVert = true, bool fHorz = true);
    // Возвращает реальное количество фиксированных столбцов
    inline int get_real_fixed_col_count() const;
    // Возвращает расстояние от левой границы столбца nStart до левой границы столбца nCol
    int get_col_dx(int nCol, int nStart = 0) const;
    Gdk::Rectangle get_checkbox_rect(uint nBlockIdx) const;
    int get_checkbox_size() const;
    void draw_checkbox(const Cairo::RefPtr<Cairo::Context>& cr, int x, int y, int nSize,
                       bool fActive, bool fSensitive);
    // Возвращает номер столбца по координате X
    int get_col_by_x(int x) const;
    // Возвращает номер блока по координате Y
    int get_block_by_y(int y) const;
    // Возвращает номер строки по координате Y
    int get_row_by_y(int y) const;
    int mouse_to_edit_cell_pos(int x, int y) const;
    // Возвращает координаты ячейки
    Gdk::Rectangle get_cell_rect(int nCol, int nRow) const;
    CByteAttrsMap::const_iterator find_byte_attrs(uint nAddress) const;
    byte_format address_to_format(uint nAddress) const;
    Glib::ustring byte_to_string(uint nByte, byte_format nFormat) const;

protected:
    std::vector<int> m_oColWidths;  // Ширины столбцов
    int m_nDefaultColWidth;         // Ширина столбца по умолчанию
    int m_nRowCount;                // Количество строк
    int m_nFixedCols;  // Количество фиксированных столбцов (вертикальный заголовок)
    int m_nCurrCol;    // Текущая строка
    int m_nCurrRow;    // Текущий столбец
    int m_nRowHeight;  // Высота строки
    std::vector<std::string> m_oTitles;  // Название фиксированных столбцов
    uint8_t* m_pData;                    // Байты
    uint m_nDataSize;                    // Количество байт
    uint8_t* m_pOldData;                 // Старые байты
    uint m_nOldDataSize;                 // Количество старых байт
    byte_format m_nByteFormat;           // Общий формат байтов
    uint m_nBlockSize;  // Размер блока (количество байт в строке)
    uint m_nVisibleBlockIdx;    // Первый отображаемый блок
    uint m_nVisibleBlockCount;  // Количество отображаемых блоков
    bool m_fEditMode;           // True, включен режим редактирования
    int m_nEditCellPos;  // Позиция курсора в редактируемой ячейке
    uint m_nByteAccess;  // Биты, разрешенные для редактирования для текущей ячейки
    std::map<uint, byte_attrs> m_oAttrMap;
    std::vector<int8_t> m_oCheckBoxes;  // &1 установлен/снят, &2 видимый/скрыт
    CSectorSizeMap m_oSectorSizeMap;  // Размеры секторов: № блока > размер сектора

    type_signal_checkbox_toggle m_signal_checkbox_toggle;
    type_signal_validate_byte m_signal_validate_byte;
    type_signal_get_byte_access m_signal_get_byte_access;
    type_signal_select_cell m_signal_select_cell;

protected:
    Glib::RefPtr<Pango::Layout> m_pangoLayout;
};

inline uint CCardDataGrid::get_first_visible_block() const {
    return m_nVisibleBlockIdx;
}

inline uint CCardDataGrid::get_visible_block_count() const {
    return m_nVisibleBlockCount;
}

inline void CCardDataGrid::redraw() {
    queue_draw();
}

inline int CCardDataGrid::get_col() const {
    return m_nCurrCol;
}

inline int CCardDataGrid::get_row() const {
    return m_nCurrRow;
}

inline int CCardDataGrid::get_col_count() const {
    return static_cast<int>(m_oColWidths.size());
}

inline int CCardDataGrid::get_row_count() const {
    return m_nRowCount;
}

inline CCardDataGrid::byte_format CCardDataGrid::get_byte_format() const {
    return m_nByteFormat;
}

inline void CCardDataGrid::set_default_col_width(int nWidth) {
    m_nDefaultColWidth = nWidth;
}

inline CCardDataGrid::type_signal_checkbox_toggle CCardDataGrid::signal_checkbox_toggle() {
    return m_signal_checkbox_toggle;
}

inline CCardDataGrid::type_signal_validate_byte CCardDataGrid::signal_validate_byte() {
    return m_signal_validate_byte;
}

inline CCardDataGrid::type_signal_get_byte_access CCardDataGrid::signal_get_byte_access() {
    return m_signal_get_byte_access;
}

CCardDataGrid::type_signal_select_cell CCardDataGrid::signal_select_cell() {
    return m_signal_select_cell;
}

inline int CCardDataGrid::get_real_fixed_col_count() const {
    return std::min(get_col_count(), m_nFixedCols);
}