#include "CCardDataGrid.h"

#include <cairomm/context.h>

#include <codecvt>  // для std::codecvt_utf8
#include <iomanip>  // для std::wstring_convert
#include <iostream>

CCardDataGrid::CCardDataGrid() :
    Glib::ObjectBase(typeid(CCardDataGrid)),
    m_nDefaultColWidth(100),
    m_nRowCount(0),
    m_nFixedCols(1),
    m_nCurrCol(-1),
    m_nCurrRow(-1),
    m_nRowHeight(25),
    m_pData(nullptr),
    m_nDataSize(0),
    m_pOldData(nullptr),
    m_nOldDataSize(0),
    m_nVisibleBlockIdx(0),
    m_nVisibleBlockCount(0),
    m_nByteFormat(BYTE_FORMAT_BIN),
    m_nBlockSize(0),
    m_fEditMode(false),
    m_nEditCellPos(0),
    m_nByteAccess(0) {
    m_pangoLayout = Pango::Layout::create(get_pango_context());
    auto cxt = m_pangoLayout->get_context();
    // Pango::FontDescription fontd;
    auto fontd = cxt->get_font_description();
    fontd.set_family("Monospace");
    // fontd.set_size(15 * Pango::SCALE);
    cxt->set_font_description(fontd);

    set_draw_func(sigc::mem_fun(*this, &CCardDataGrid::on_draw));
    signal_resize().connect(sigc::mem_fun(*this, &CCardDataGrid::on_resize), true);

    set_focusable();

    auto controller = Gtk::EventControllerKey::create();
    controller->signal_key_pressed().connect(sigc::mem_fun(*this, &CCardDataGrid::on_key_pressed),
                                             false);
    add_controller(controller);

    auto mouseClick = Gtk::GestureClick::create();
    mouseClick->set_button(GDK_BUTTON_PRIMARY);
    mouseClick->signal_pressed().connect(sigc::mem_fun(*this, &CCardDataGrid::on_mouse_click));
    add_controller(mouseClick);

    property_vadjustment().signal_changed().connect(
        sigc::mem_fun(*this, &CCardDataGrid::on_property_vadjustment_change));
    property_hadjustment().signal_changed().connect(
        sigc::mem_fun(*this, &CCardDataGrid::on_property_hadjustment_change));
}

CCardDataGrid::~CCardDataGrid() {
}

void CCardDataGrid::set_fix_col_titles(const char** pTitles, size_t nCount) {
    m_oTitles.assign(pTitles, pTitles + nCount);
}

void CCardDataGrid::update() {
    m_nFixedCols = m_oSectorSizeMap.empty() ? 1 : 3;
    set_col_count(m_nFixedCols + m_nBlockSize);
    set_row_count(kFixedRows + get_block_count());
    int cx, cy;
    m_pangoLayout->set_text("Z");
    m_pangoLayout->get_pixel_size(cx, cy);
    m_nRowHeight = (cy + kCellVPadding + kCellVPadding);
    Update_fix_col_widths();
}

void CCardDataGrid::set_data_pointer(uint8_t* pData, uint nSize) {
    m_pData = pData;
    m_nDataSize = nSize;
}

void CCardDataGrid::set_old_data_pointer(uint8_t* pData, uint nSize) {
    m_pOldData = pData;
    m_nOldDataSize = nSize;
}

void CCardDataGrid::set_block_size(uint nSize) {
    if (m_nBlockSize == nSize)
        return;
    m_nBlockSize = nSize;
}

void CCardDataGrid::add_sector_size(uint nFromBlockIdx, uint nSize) {
    m_oSectorSizeMap.insert_or_assign(nFromBlockIdx, nSize);
}

void CCardDataGrid::clear_sector_sizes() {
    m_oSectorSizeMap.clear();
}

void CCardDataGrid::address_to_cell(uint nAddress, int& nCol, int& nRow) const {
    nCol = -1;
    nRow = -1;
    if ((get_col_count() <= m_nFixedCols) || (m_nRowCount < kFixedRows))
        return;
    uint nRowBytes = (get_col_count() - m_nFixedCols);

    nCol = static_cast<int>(m_nFixedCols + (nAddress % nRowBytes));
    nRow = static_cast<int>(m_nRowCount - 1 - (nAddress / nRowBytes));
}

bool CCardDataGrid::cell_to_address(int nCol, int nRow, uint& nAddress) const {
    auto nByteIdx = col_to_byte_idx(nCol);
    if (-1 == nByteIdx)
        return false;
    auto nBlockIdx = row_to_block_idx(nRow);
    if (-1 == nBlockIdx)
        return false;
    nAddress = (nBlockIdx * m_nBlockSize + nByteIdx);
    return nAddress < m_nDataSize;
}

std::string CCardDataGrid::address_to_text(uint nAddress, bool fOldData) const {
    uint8_t* p;
    if (fOldData) {
        if (nAddress < m_nOldDataSize)
            p = &m_pOldData[nAddress];
        else
            return std::string();
    }
    else {
        if (nAddress < m_nDataSize)
            p = &m_pData[nAddress];
        else
            return std::string();
    }
    auto nFormat = address_to_format(nAddress);
    return byte_to_string((unsigned char)*p, nFormat);
}

int CCardDataGrid::get_address_text_length(uint nAddress) const {
    auto nFormat = address_to_format(nAddress);
    switch (nFormat) {
    case BYTE_FORMAT_HEX:
        return 2;

    case BYTE_FORMAT_DEC:
    case BYTE_FORMAT_OCT:
        return 3;

    case BYTE_FORMAT_BIN:
        return 8;
    }
    return 0;
}

void CCardDataGrid::set_row_count(uint nCount) {
    if (m_nRowCount == nCount)
        return;
    m_nRowCount = nCount;
    if (m_nCurrRow >= m_nRowCount)
        m_nCurrRow = (m_nRowCount - 1);
}

uint CCardDataGrid::get_block_count() const {
    if (0 == m_nBlockSize)
        return 0;
    uint res = (m_nDataSize / m_nBlockSize);
    if ((m_nVisibleBlockCount != 0) && (res > m_nVisibleBlockCount))
        res = m_nVisibleBlockCount;
    return res;
}

uint CCardDataGrid::get_total_block_count() const {
    return (m_nBlockSize ? (m_nDataSize / m_nBlockSize) : 0);
}

void CCardDataGrid::get_visible_addresses(uint& nBeginAddress, uint& nEndAddress) const {
    if (m_nVisibleBlockCount != 0) {
        nBeginAddress = (m_nVisibleBlockIdx * m_nBlockSize);
        nEndAddress =
            std::min((m_nVisibleBlockIdx + m_nVisibleBlockCount) * m_nBlockSize, m_nDataSize);
    }
    else {
        nBeginAddress = 0;
        nEndAddress = m_nDataSize;
    }
}

uint CCardDataGrid::calc_sector_count(uint nBlockCount, uint& nMaxSBlocks) const {
    uint res = 0;
    nMaxSBlocks = 0;

    auto it = m_oSectorSizeMap.cbegin();
    auto nLeft = nBlockCount;
    uint c, nNext;
    CSectorSizeMap::const_iterator itNext;
    while (it != m_oSectorSizeMap.cend()) {
        itNext = std::next(it);
        nNext = (itNext != m_oSectorSizeMap.cend()) ? itNext->first : nBlockCount;
        if (nMaxSBlocks < it->second)
            nMaxSBlocks = it->second;
        c = nNext - it->first;
        if (c >= nLeft) {
            res += nLeft / it->second;
            break;
        }
        res += c / it->second;
        nLeft -= c;
        it = itNext;
    }

    return res;
}

int CCardDataGrid::get_cols_per_width(int nWidth, uint nStartCol, uint nEndCol, int& dx) const {
    dx = 0;
    int n;
    for (auto i = nStartCol; i < nEndCol; i++) {
        n = (dx + m_oColWidths[i]);
        if (n > nWidth)
            return (i - nStartCol);
        dx = n;
    }
    return (nEndCol - nStartCol);
}

void CCardDataGrid::set_col_count(uint nCount) {
    if (get_col_count() == nCount)
        return;
    m_oColWidths.resize(nCount, m_nDefaultColWidth);
    if (m_nCurrCol >= m_oColWidths.size())
        m_nCurrCol = ((int)m_oColWidths.size() - 1);
}

void CCardDataGrid::set_col_width(int nCol, int nWidth) {
    if (nWidth == m_oColWidths[nCol])
        return;
    m_oColWidths[nCol] = nWidth;
}

void CCardDataGrid::set_current_cell(int nCol, int nRow, bool fScroll, bool fNotify) {
    if ((m_nCurrCol == nCol) && (m_nCurrRow == nRow))
        return;

    m_nCurrCol = nCol;
    m_nCurrRow = nRow;
    m_nByteAccess = 0;
    uint nAddress;
    if (cell_to_address(nCol, nRow, nAddress)) {
        if (!m_signal_get_byte_access.empty())
            m_nByteAccess = m_signal_get_byte_access.emit(nAddress);
        if (m_fEditMode && (m_nEditCellPos != 0)) {
            auto nLength = get_address_text_length(nAddress);
            if (m_nEditCellPos >= nLength)
                m_nEditCellPos = 0;
        }
    }
    else
        m_nEditCellPos = 0;

    if (fScroll && (m_nCurrCol >= m_nFixedCols) && (m_nCurrRow >= kFixedRows)) {
        auto pAdj = get_hadjustment();
        int nPos = pAdj ? static_cast<int>(pAdj->get_value()) : 0;

        auto rect = get_cell_rect(nCol, nRow);
        auto nFixedWidth = get_col_dx(m_nFixedCols);
        auto nWidth = get_width();
        if (rect.get_x() < nFixedWidth) {
            nPos -= (nFixedWidth - rect.get_x());
            pAdj->set_value(nPos);
        }
        else if ((rect.get_x() + rect.get_width()) > nWidth) {
            nPos += ((rect.get_x() + rect.get_width()) - nWidth);
            pAdj->set_value(nPos);
        }

        pAdj = get_vadjustment();
        nPos = pAdj ? static_cast<int>(pAdj->get_value()) : 0;
        auto nFixedHeight = (kFixedRows * m_nRowHeight);
        auto nHeight = get_height();
        if (rect.get_y() < nFixedHeight) {
            nPos -= (nFixedHeight - rect.get_y());
            pAdj->set_value(nPos);
        }
        else if ((rect.get_y() + rect.get_height()) > nHeight) {
            nPos += ((rect.get_y() + rect.get_height()) - nHeight);
            pAdj->set_value(nPos);
        }
    }
    queue_draw();

    if (fNotify && !m_signal_select_cell.empty())
        m_signal_select_cell.emit();
}

void CCardDataGrid::mouse_to_cell(int x, int y, int& nCol, int& nRow) {
    nCol = get_col_by_x(x);
    nRow = get_row_by_y(y);
}

bool CCardDataGrid::get_next_tab_cell(uint nCurrCol, uint nCurrRow, int& nCol, int& nRow,
                                      bool fPrevious) {
    nCol = -1;
    nRow = -1;
    if (0 == m_nDataSize)
        return false;

    auto nByteIdx = (nCurrCol != -1) ? col_to_byte_idx(nCurrCol) : -1;
    auto nBlockIdx = (nCurrRow != -1) ? row_to_block_idx(nCurrRow) : -1;
    uint nBeginAddress, nEndAddress;
    get_visible_addresses(nBeginAddress, nEndAddress);

    uint nAddress;
    auto nCount = (nEndAddress - nBeginAddress);
    if ((nBlockIdx != -1) && (nByteIdx != -1)) {
        nAddress = nBlockIdx * m_nBlockSize + nByteIdx;

        if (fPrevious) {
            if (nAddress == nBeginAddress)
                nAddress = (nEndAddress - 1);
            else
                --nAddress;
        }
        else if (++nAddress == nEndAddress)
            nAddress = nBeginAddress;
        --nCount;
    }
    else
        nAddress = fPrevious ? (nEndAddress - 1) : 0;

    uint nByteAccess;
    for (uint i = 0; i < nCount; i++) {
        if (!m_fEditMode || m_signal_get_byte_access.empty() ||
            ((nByteAccess = m_signal_get_byte_access.emit(nAddress)) != 0)) {
            nCol = byte_idx_to_col(nAddress % m_nBlockSize);
            nRow = block_idx_to_row(nAddress / m_nBlockSize);
            return true;
        }
        if (fPrevious) {
            if (nAddress == nBeginAddress)
                nAddress = (nEndAddress - 1);
            else
                --nAddress;
        }
        else if (++nAddress == nEndAddress)
            nAddress = nBeginAddress;
    }
    return false;
}

int CCardDataGrid::get_next_tab_col(uint nCurrCol, bool fPrevious) {
    if ((0 == m_nDataSize) || (-1 == m_nCurrRow))
        return -1;

    auto nByteIdx = (nCurrCol != -1) ? col_to_byte_idx(nCurrCol) : -1;
    auto nBlockIdx = (m_nCurrRow != -1) ? row_to_block_idx(m_nCurrRow) : -1;
    auto nCount = m_nBlockSize;
    uint nBeginAddress, nEndAddress;
    get_visible_addresses(nBeginAddress, nEndAddress);

    uint nAddress, nStart;
    if ((nBlockIdx != -1) && (nByteIdx != -1)) {
        nAddress = ((nBlockIdx * m_nBlockSize) + nByteIdx);
        nStart = (nAddress - nByteIdx);

        if (fPrevious) {
            if (nAddress == nStart)
                nAddress += (m_nBlockSize - 1);
            else
                --nAddress;
        }
        else if (++nAddress == nEndAddress)
            nAddress = nStart;
        --nCount;
    }
    else {
        nAddress = 0;
        nStart = nAddress;
    }
    nEndAddress = std::min(nEndAddress, nStart + m_nBlockSize);

    uint nByteAccess;
    for (uint i = 0; i < nCount; i++) {
        if (!m_fEditMode || m_signal_get_byte_access.empty() ||
            ((nByteAccess = m_signal_get_byte_access.emit(nAddress)) != 0))
            return byte_idx_to_col(nAddress % m_nBlockSize);

        if (fPrevious) {
            if (nAddress == nStart)
                nAddress += (m_nBlockSize - 1);
            else
                --nAddress;
        }
        else if (++nAddress == nEndAddress)
            nAddress = nStart;
    }
    return -1;
}

int CCardDataGrid::get_next_tab_row(uint nCurrRow, bool fPrevious) {
    if (m_nDataSize <= m_nBlockSize)
        return -1;

    auto nByteIdx = (m_nCurrCol != -1) ? col_to_byte_idx(m_nCurrCol) : -1;
    auto nBlockIdx = (nCurrRow != -1) ? row_to_block_idx(nCurrRow) : -1;
    uint nBeginAddress, nEndAddress;
    get_visible_addresses(nBeginAddress, nEndAddress);
    auto nBlockCount = (nEndAddress - nBeginAddress) / m_nBlockSize;
    auto nCount = nBlockCount;

    uint nAddress;
    if ((nBlockIdx != -1) && (nByteIdx != -1)) {
        nAddress = nBlockIdx * m_nBlockSize + nByteIdx;

        if (fPrevious) {
            if (nAddress < (nBeginAddress + m_nBlockSize))
                nAddress = nBeginAddress + ((nBlockCount - 1) * m_nBlockSize) + nByteIdx;
            else
                nAddress -= m_nBlockSize;
        }
        else {
            nAddress += m_nBlockSize;
            if (nAddress > nEndAddress)
                nAddress = nBeginAddress + (nAddress % nEndAddress);
        }
        --nCount;
    }
    else
        nAddress = 0;

    uint nByteAccess;
    for (uint i = 0; i < nCount; i++) {
        if (!m_fEditMode || m_signal_get_byte_access.empty() ||
            ((nByteAccess = m_signal_get_byte_access.emit(nAddress)) != 0))
            return block_idx_to_row(nAddress / m_nBlockSize);

        if (fPrevious) {
            if (nAddress < (nBeginAddress + m_nBlockSize))
                nAddress = nBeginAddress + ((nBlockCount - 1) * m_nBlockSize) + nByteIdx;
            else
                nAddress -= m_nBlockSize;
        }
        else {
            nAddress += m_nBlockSize;
            if (nAddress > nEndAddress)
                nAddress = nBeginAddress + (nAddress % nEndAddress);
        }
    }
    return -1;
}

int CCardDataGrid::row_to_block_idx(int nRow) const {
    if ((nRow < 0) || (nRow < kFixedRows) || (nRow >= m_nRowCount))
        return -1;
    return (m_nVisibleBlockIdx + m_nRowCount - 1 - nRow);
}

int CCardDataGrid::block_idx_to_row(uint nBlockIdx) const {
    auto nBlockCount = get_block_count();
    if (nBlockIdx >= (m_nVisibleBlockIdx + nBlockCount))
        return -1;
    return (kFixedRows + nBlockCount - 1 - nBlockIdx + m_nVisibleBlockIdx);
}

int CCardDataGrid::col_to_byte_idx(int nCol) const {
    if ((nCol < 0) || (nCol < m_nFixedCols) || (nCol >= m_oColWidths.size()))
        return -1;
    return (nCol - m_nFixedCols);
}

int CCardDataGrid::byte_idx_to_col(uint nByteIdx) const {
    if (nByteIdx >= m_nBlockSize)
        return -1;
    return (m_nFixedCols + nByteIdx);
}

void CCardDataGrid::set_fixed_col_count(uint nCount) {
    m_nFixedCols = nCount;
}

void CCardDataGrid::set_byte_format(byte_format nFormat, bool fUpdateColSizes) {
    if (m_nByteFormat == nFormat)
        return;
    m_nByteFormat = nFormat;
    if (m_fEditMode)
        m_nEditCellPos = 0;
    if (fUpdateColSizes)
        update_data_col_sizes();
}

void CCardDataGrid::add_or_set_byte_attrs(uint nAddress, uint nCount, byte_format nFormat,
                                          Gdk::RGBA foreColor, Gdk::RGBA bgColor, bool fRedraw) {
    m_oAttrMap.insert_or_assign(nAddress, byte_attrs(nCount, nFormat, foreColor, bgColor));
    if (fRedraw)
        queue_draw();
}

void CCardDataGrid::remove_byte_attrs(uint nAddress) {
    auto it = m_oAttrMap.find(nAddress);
    if (it != m_oAttrMap.end())
        m_oAttrMap.erase(it);
    queue_draw();
}

void CCardDataGrid::clear_byte_attrs() {
    m_oAttrMap.clear();
    queue_draw();
}

void CCardDataGrid::set_visible_range(uint nBlockIdx, uint nBlockCount, bool fUpdate) {
    if ((m_nVisibleBlockIdx = nBlockIdx) && (m_nVisibleBlockCount = nBlockCount))
        return;
    m_nVisibleBlockIdx = nBlockIdx;
    m_nVisibleBlockCount = nBlockCount;
    if (fUpdate)
        update();
}

void CCardDataGrid::update_data_col_sizes() {
    m_pangoLayout->set_text("Z");
    int nTextWidth, nTextHeight;
    m_pangoLayout->get_pixel_size(nTextWidth, nTextHeight);

    auto nDataPadding = (kDataCellHPadding + kDataCellHPadding);
    if (m_oAttrMap.empty()) {
        int nColWidth = (nTextWidth * get_address_text_length(0)) + nDataPadding;
        for (int i = m_nFixedCols; i < m_oColWidths.size(); i++)
            m_oColWidths[i] = nColWidth;
    }
    else {
        uint nBeginAddress, nEndAddress;
        get_visible_addresses(nBeginAddress, nEndAddress);
        uint nAddress;
        int nMaxLength, nLength;
        for (auto i = m_nFixedCols; i < m_oColWidths.size(); i++) {
            nMaxLength = 0;
            nAddress = nBeginAddress + (i - m_nFixedCols);
            while (nAddress < nEndAddress) {
                nLength = get_address_text_length(nAddress);
                if (nMaxLength < nLength)
                    nMaxLength = nLength;
                nAddress += m_nBlockSize;
            }
            m_oColWidths[i] = nMaxLength * nTextWidth + nDataPadding;
        }
    }
    update_canvas_size();
}

void CCardDataGrid::set_checkbox_visible(uint nBlockIdx, bool fVisible) {
    if (m_oCheckBoxes.size() <= nBlockIdx)
        m_oCheckBoxes.resize(nBlockIdx + 1);
    uint nState = m_oCheckBoxes[nBlockIdx];
    if (((nState & CHECKBOX_F_VISIBLE) != 0) == fVisible)
        return;
    if (fVisible)
        nState |= CHECKBOX_F_VISIBLE;
    else
        nState &= ~CHECKBOX_F_VISIBLE;
    m_oCheckBoxes[nBlockIdx] = (uint8_t)nState;
    queue_draw();
}

void CCardDataGrid::set_checkbox_active(uint nBlockIdx, bool fActive) {
    if (m_oCheckBoxes.size() <= nBlockIdx)
        m_oCheckBoxes.resize(nBlockIdx + 1);
    uint nState = m_oCheckBoxes[nBlockIdx];
    if (((nState & CHECKBOX_F_ACTIVE) != 0) == fActive)
        return;
    if (fActive)
        nState |= CHECKBOX_F_ACTIVE;
    else
        nState &= ~CHECKBOX_F_ACTIVE;
    m_oCheckBoxes[nBlockIdx] = (uint8_t)nState;
    queue_draw();
}

void CCardDataGrid::set_checkbox_sensitive(uint nBlockIdx, bool fSensitive) {
    if (m_oCheckBoxes.size() <= nBlockIdx)
        m_oCheckBoxes.resize(nBlockIdx + 1);
    uint nState = m_oCheckBoxes[nBlockIdx];
    if (((nState & CHECKBOX_F_SENSITIVE) != 0) == fSensitive)
        return;
    if (fSensitive)
        nState |= CHECKBOX_F_SENSITIVE;
    else
        nState &= ~CHECKBOX_F_SENSITIVE;
    m_oCheckBoxes[nBlockIdx] = (uint8_t)nState;
    queue_draw();
}

void CCardDataGrid::set_range_checkbox(uint nBlockIdx, uint nCount, bool fVisible, bool fActive,
                                       bool fSensitive) {
    auto nEnd = (nBlockIdx + nCount);
    if (m_oCheckBoxes.size() < nEnd)
        m_oCheckBoxes.resize(nEnd);
    uint nState = 0;
    if (fVisible)
        nState |= CHECKBOX_F_VISIBLE;
    if (fActive)
        nState |= CHECKBOX_F_ACTIVE;
    if (fSensitive)
        nState |= CHECKBOX_F_SENSITIVE;
    for (size_t i = nBlockIdx; i < nEnd; i++)
        m_oCheckBoxes[i] = nState;
    queue_draw();
}

bool CCardDataGrid::get_checkbox_active(uint nBlockIdx) const {
    return (nBlockIdx < m_oCheckBoxes.size()) ? (m_oCheckBoxes[nBlockIdx] & CHECKBOX_F_ACTIVE) :
                                                false;
}

bool CCardDataGrid::get_checkbox_sensitive(uint nBlockIdx) const {
    return (nBlockIdx < m_oCheckBoxes.size()) ? (m_oCheckBoxes[nBlockIdx] & CHECKBOX_F_SENSITIVE) :
                                                false;
}

void CCardDataGrid::EnableEditMode(bool fEnable) {
    if (m_fEditMode == fEnable)
        return;
    m_fEditMode = fEnable;
    if (fEnable)
        m_nEditCellPos = 0;
    queue_draw();
}

void CCardDataGrid::on_draw(const Cairo::RefPtr<Cairo::Context>& cr, int width, int height) {
    auto adj = get_vadjustment();
    int nVPos = adj ? static_cast<int>(adj->get_value()) : 0;
    adj = get_hadjustment();
    int nHPos = adj ? static_cast<int>(adj->get_value()) : 0;

    cr->save();

    // Закрашиваем фиксированные области
    auto nFixedCols = get_real_fixed_col_count();
    auto nFixedHeight = (m_nRowHeight * kFixedRows);
    cr->set_source_rgb(0.9, 0.9, 0.9);
    cr->rectangle(0, 0, width, nFixedHeight);
    cr->fill();
    auto nFixedWidth = get_col_dx(nFixedCols);
    cr->rectangle(0, 0, nFixedWidth, height);
    cr->fill();

    auto nColCount = get_col_count();

    int nX;
    // Первый отображаемый столбец
    auto nFirstCol = nFixedCols + get_cols_per_width(nHPos, nFixedCols, nColCount, nX);
    // Левая координата первого столбца
    auto nFirstColLeft = (nFixedWidth + nX - nHPos);
    // Последний отображаемый столбец
    auto n = width - nFirstColLeft;
    auto nEndCol = nFirstCol + get_cols_per_width(n, nFirstCol, nColCount, nX);
    // Правая координата последнего столбца
    auto nLastColRight = nFirstColLeft + nX;
    // Если есть ещё свободное место и есть ещё столбцы, не полностью отображаемые
    if ((nX < n) && (nEndCol < nColCount)) {
        nLastColRight += m_oColWidths[nEndCol];
        ++nEndCol;
    }

    auto nFullBlockCount = (m_nRowCount - kFixedRows);
    // Высота всех блоков
    auto nFullDataHeight = (nFullBlockCount * m_nRowHeight);
    // Высота для отображения блоков
    auto nDataHeight = (height - nFixedHeight);
    // Первый отображаемый блок
    auto nFirstBlock = (int)m_nVisibleBlockIdx +
                       std::max((nFullDataHeight - nVPos - nDataHeight) / m_nRowHeight, 0);
    // Последняя отображаемая строка
    auto nLastRow = (m_nRowCount - 1 - nFirstBlock + (int)m_nVisibleBlockIdx);
    // Верхняя граница последней строки
    auto nLastRowTop = (nFixedHeight + nFullDataHeight - nVPos -
                        ((nFirstBlock - (int)m_nVisibleBlockIdx) + 1) * m_nRowHeight);
    // Количество отображаемых блоков
    auto nY = (nLastRowTop - nFixedHeight);
    auto nVisBlockCount = (nY / m_nRowHeight + 1);
    if (nY % m_nRowHeight)
        ++nVisBlockCount;
    // Первая отображаемая строка
    auto nFirstRow =
        m_nRowCount - 1 - ((nFirstBlock - (int)m_nVisibleBlockIdx) + nVisBlockCount - 1);
    // Верхняя граница первой строки
    auto nFirstRowTop = nLastRowTop - (nVisBlockCount - 1) * m_nRowHeight;

    // Рисуем сетку
    cr->set_source_rgb(0.6, 0.6, 0.6);
    cr->set_line_width(0.5);

    cr->move_to(0, nFixedHeight);
    cr->line_to(width, nFixedHeight);
    cr->stroke();

    nX = 0;
    for (uint i = 0; i < nFixedCols; i++) {
        nX += m_oColWidths[i];
        cr->move_to(nX, 0);
        cr->line_to(nX, height);
        cr->stroke();
    }

    cr->save();
    cr->rectangle(0, std::max(nFirstRowTop, (int)nFixedHeight), width, nDataHeight);
    cr->clip();
    // Рисуем горизонтальные разделительные линии строк данных
    nY = nFirstRowTop;
    for (uint i = nFirstRow; i <= nLastRow; i++) {
        cr->move_to(0, nY);
        cr->line_to(width, nY);
        cr->stroke();
        nY += m_nRowHeight;
    }
    cr->restore();

    cr->save();
    // Ограничиваем область рисования прямоугольником
    cr->rectangle(std::max(nFirstColLeft, (int)nFixedWidth), 0, nLastColRight - nFirstColLeft,
                  height);
    cr->clip();
    // Рисуем вертикальные разделительные линии столбцов данных
    nX = nFirstColLeft;
    for (auto i = nFirstCol; i < nEndCol; i++) {
        cr->move_to(nX, 0);
        cr->line_to(nX, height);
        cr->stroke();
        nX += m_oColWidths[i];
    }
    cr->restore();

    // Рисуем названия фиксированных заголовков
    cr->set_source_rgb(0.1, 0.1, 0.1);
    nX = 0;
    int nColWidth, nTextWidth, nTextHeight;
    auto c = std::min(nFixedCols, (int)m_oTitles.size());
    for (int i = 0; i < c; i++) {
        nColWidth = m_oColWidths[i];
        m_pangoLayout->set_text(m_oTitles[i]);
        m_pangoLayout->get_pixel_size(nTextWidth, nTextHeight);
        cr->move_to(nX + (nColWidth - nTextWidth) / 2, (m_nRowHeight - nTextHeight) / 2);
        m_pangoLayout->show_in_cairo_context(cr);
        nX += nColWidth;
    }

    // Рисуем заголовки столбцов байтов
    cr->save();
    // Ограничиваем область рисования прямоугольником
    nX = std::max(nFirstColLeft, (int)nFixedWidth);
    cr->rectangle(nX, 0, nLastColRight - nX, m_nRowHeight);
    cr->clip();
    nX = nFirstColLeft;
    uint nByteIdx = (nFirstCol - nFixedCols);
    for (auto i = nFirstCol; i < nEndCol; i++) {
        nColWidth = m_oColWidths[i];
        m_pangoLayout->set_text(Glib::ustring::format(nByteIdx));
        m_pangoLayout->get_pixel_size(nTextWidth, nTextHeight);
        cr->move_to(nX + (nColWidth - nTextWidth) / 2, (m_nRowHeight - nTextHeight) / 2);
        m_pangoLayout->show_in_cairo_context(cr);
        nX += nColWidth;
        ++nByteIdx;
    }
    cr->restore();

    // Рисуем заголовки строк в столбце "Блок"
    cr->save();
    // Ограничиваем область рисования прямоугольником
    cr->rectangle(0, std::max(nFirstRowTop, (int)nFixedHeight), width, nDataHeight);
    cr->clip();
    nY = nLastRowTop;
    // Размер флажка (ширина = высота checkbox)
    auto nCheckboxSize = get_checkbox_size();
    // Смещение по высоте флажка
    auto nCheckboxDy = (m_nRowHeight - nCheckboxSize) / 2;
    auto nEndBlock = nFirstBlock + nVisBlockCount;
    for (auto i = nFirstBlock; i < nEndBlock; i++) {
        nX = kFixedCellHPadding;
        if (!m_oCheckBoxes.empty()) {
            if (i < m_oCheckBoxes.size()) {
                auto nState = m_oCheckBoxes[i];
                if (nState & CHECKBOX_F_VISIBLE)
                    draw_checkbox(cr, nX, nY + nCheckboxDy, nCheckboxSize,
                                  (nState & CHECKBOX_F_ACTIVE), (nState & CHECKBOX_F_SENSITIVE));
            }
            nX += nCheckboxSize + kCellSpacing;
        }
        m_pangoLayout->set_text(Glib::ustring::format(i));
        m_pangoLayout->get_pixel_size(nTextWidth, nTextHeight);
        nColWidth = (m_oColWidths[0] - nX);
        cr->move_to(nX + (nColWidth - nTextWidth) / 2, nY + (m_nRowHeight - nTextHeight) / 2);
        m_pangoLayout->show_in_cairo_context(cr);
        nY -= m_nRowHeight;
    }
    // Рисуем заголовки строк для столбцов "Сектор" и "Блок сектора"
    if (!m_oSectorSizeMap.empty()) {
        // Ищем номер сектора, номер блока в секторе и размер сектора для первого видимого блока
        uint nSectorIdx = 0;  // Номер сектора
        uint nSBlock = 0;     // Номер блока в секторе
        uint nSectorSize = 0;  // Размер сектора (количество блоков в секторе)
        uint nNext = 0;  // Номер следующего блока, с которого изменяется размер сектора
        auto it = m_oSectorSizeMap.cbegin();
        if ((it != m_oSectorSizeMap.cend()) && (nFirstBlock < it->first))
            nNext = it->first;
        else {
            while (it != m_oSectorSizeMap.cend()) {
                auto itNext = std::next(it);
                if (itNext != m_oSectorSizeMap.cend()) {
                    if (nFirstBlock < itNext->first) {
                        nSectorIdx += (nFirstBlock - it->first) / it->second;
                        nSBlock = (nFirstBlock - it->first) % it->second;
                        nSectorSize = it->second;
                        nNext = itNext->first;
                        break;
                    }
                    else
                        nSectorIdx += (itNext->first - it->first) / it->second;
                }
                else {
                    nSectorIdx += (nFirstBlock - it->first) / it->second;
                    nSBlock = (nFirstBlock - it->first) % it->second;
                    nSectorSize = it->second;
                    nNext = std::numeric_limits<uint>::max();
                    break;
                }
                it = itNext;
            }
        }

        nX = m_oColWidths[0];
        auto nX2 = (nX + m_oColWidths[1]);
        nY = nLastRowTop + m_nRowHeight;
        int nY2;
        auto fDrawSector = true;
        for (auto i = nFirstBlock; i < nEndBlock; i++) {
            if (i == nNext) {
                ++it;
                nSectorSize = it->second;
                nNext = (it != m_oSectorSizeMap.cend()) ? it->first : nFullBlockCount;
            }
            if (fDrawSector) {
                fDrawSector = false;
                nY2 = (nY - ((nSectorSize - nSBlock) * m_nRowHeight));

                cr->set_source_rgb(0.9, 0.9, 0.9);
                cr->rectangle(nX + 1, nY2 + 2, m_oColWidths[1] - 2, nY - nY2 - 4);
                cr->fill();
                cr->stroke();

                cr->set_source_rgb(0.1, 0.1, 0.1);
                m_pangoLayout->set_text(Glib::ustring::format(nSectorIdx));
                m_pangoLayout->get_pixel_size(nTextWidth, nTextHeight);
                cr->move_to(nX + (m_oColWidths[1] - nTextWidth) / 2,
                            nY2 + (nY - nY2 - nTextHeight) / 2);
                m_pangoLayout->show_in_cairo_context(cr);

                cr->move_to(0, nY);
                cr->line_to(width, nY);
                cr->stroke();
            }

            nY -= m_nRowHeight;
            m_pangoLayout->set_text(Glib::ustring::format(nSBlock));
            m_pangoLayout->get_pixel_size(nTextWidth, nTextHeight);
            cr->move_to(nX2 + (m_oColWidths[2] - nTextWidth) / 2,
                        nY + (m_nRowHeight - nTextHeight) / 2);
            m_pangoLayout->show_in_cairo_context(cr);

            if (++nSBlock == nSectorSize) {
                nSBlock = 0;
                ++nSectorIdx;
                fDrawSector = true;
            }
        }
    }
    cr->restore();

    // Рисуем ячейки данных
    cr->save();
    // Ограничиваем область рисования прямоугольником
    cr->rectangle(nFixedWidth, nFixedHeight, width - nFixedWidth, nDataHeight);
    cr->clip();

    // Рисуем значения байт
    uint8_t* p;
    uint8_t* pOld;
    uint nAddress = (nFirstBlock * m_nBlockSize);
    uint nAddress2;
    nY = nLastRowTop;
    int nY2, nX2;
    byte_format nFormat;
    CByteAttrsMap::const_iterator itA;
    Glib::ustring s, s2;
    std::u32string sU;
    uint nByte, nOldByte, nChanges;
    for (auto i = nFirstBlock; i < nEndBlock; i++) {
        nX = nFirstColLeft;
        nY2 = (nY - m_nRowHeight);
        nAddress2 = nAddress;
        nAddress += (nFirstCol - m_nFixedCols);
        p = (nAddress < m_nDataSize) ? &m_pData[nAddress] : nullptr;
        pOld = (nAddress < m_nOldDataSize) ? &m_pOldData[nAddress] : nullptr;
        for (auto j = nFirstCol; j < nEndCol; j++) {
            nX2 = (nX + m_oColWidths[j]);
            if (nAddress < m_nDataSize) {
                nFormat = m_nByteFormat;
                itA = find_byte_attrs(nAddress);
                if (itA != m_oAttrMap.cend()) {
                    if (itA->second.m_nFormat != BYTE_FORMAT_UNDEF)
                        nFormat = itA->second.m_nFormat;
                    if (!itA->second.m_BgColor.is_clear()) {
                        cr->save();
                        cr->set_source_rgb(itA->second.m_BgColor.get_red(),
                                           itA->second.m_BgColor.get_green(),
                                           itA->second.m_BgColor.get_blue());
                        cr->rectangle(nX + 1, nY + 2, nX2 - nX - 2, m_nRowHeight - 4);
                        cr->fill();
                        cr->restore();
                    }
                    if (!itA->second.m_ForeColor.is_clear())
                        cr->set_source_rgb(itA->second.m_ForeColor.get_red(),
                                           itA->second.m_ForeColor.get_green(),
                                           itA->second.m_ForeColor.get_blue());
                }
                s = byte_to_string((unsigned char)*p, nFormat);

                nByte = (unsigned char)*p;
                if (nAddress < m_nOldDataSize)
                    nOldByte = (unsigned char)*pOld++;
                else
                    nOldByte = nByte;

                // Если байт изменился, то выделяем измененные цифры
                if (nByte != nOldByte) {
                    cr->save();
                    std::wstring_convert<std::codecvt_utf8<char32_t>, char32_t> cv;
                    sU = cv.from_bytes(s);
                    s2.clear();
                    switch (nFormat) {
                    case BYTE_FORMAT_HEX:
                        {
                            nChanges = (nByte ^ nOldByte);
                            uint nMask = (0xf << (4 * (sU.length() - 1)));
                            size_t nPos = 0;
                            bool fChangeMode = false;
                            for (size_t k = 0; k < sU.length(); k++) {
                                if (fChangeMode == !(nChanges & nMask)) {
                                    fChangeMode = !fChangeMode;
                                    if (nPos != k) {
                                        s2.append(cv.to_bytes(sU.substr(nPos, k - nPos)));
                                        nPos = k;
                                    }
                                    if (fChangeMode)
                                        s2.append("<span color=\"#FF0000\">");
                                    else
                                        s2.append("</span>");
                                }
                                nMask >>= 4;
                            }
                            if (nPos < sU.length())
                                s2.append(cv.to_bytes(sU.substr(nPos, sU.length() - nPos)));
                            if (fChangeMode)
                                s2.append("</span>");
                            break;
                        }

                    case BYTE_FORMAT_DEC:
                        {
                            bool fChange;
                            uint nDiv = pow(10, sU.length() - 1);
                            auto fChangeMode = false;
                            size_t nPos = 0;
                            for (size_t k = 0; k < sU.length(); k++) {
                                fChange = ((nByte / nDiv) % 10) != ((nOldByte / nDiv) % 10);
                                if (fChangeMode != fChange) {
                                    fChangeMode = !fChangeMode;
                                    if (nPos != k) {
                                        s2.append(cv.to_bytes(sU.substr(nPos, k - nPos)));
                                        nPos = k;
                                    }
                                    if (fChangeMode)
                                        s2.append("<span color=\"#FF0000\">");
                                    else
                                        s2.append("</span>");
                                }
                                nDiv /= 10;
                            }
                            if (nPos < sU.length())
                                s2.append(cv.to_bytes(sU.substr(nPos, sU.length() - nPos)));
                            if (fChangeMode)
                                s2.append("</span>");
                            break;
                        }

                    case BYTE_FORMAT_OCT:
                        {
                            nChanges = (nByte ^ nOldByte);
                            uint nMask = (7 << (3 * (sU.length() - 1)));
                            size_t nPos = 0;
                            bool fChangeMode = false;
                            for (size_t k = 0; k < sU.length(); k++) {
                                if (fChangeMode == !(nChanges & nMask)) {
                                    fChangeMode = !fChangeMode;
                                    if (nPos != k) {
                                        s2.append(cv.to_bytes(sU.substr(nPos, k - nPos)));
                                        nPos = k;
                                    }
                                    if (fChangeMode)
                                        s2.append("<span color=\"#FF0000\">");
                                    else
                                        s2.append("</span>");
                                }
                                nMask >>= 3;
                            }
                            if (nPos < sU.length())
                                s2.append(cv.to_bytes(sU.substr(nPos, sU.length() - nPos)));
                            if (fChangeMode)
                                s2.append("</span>");
                            break;
                        }

                    case BYTE_FORMAT_BIN:
                        {
                            nChanges = (nByte ^ nOldByte);
                            uint nMask = (1 << (sU.length() - 1));
                            size_t nPos = 0;
                            bool fChangeMode = false;
                            for (size_t k = 0; k < sU.length(); k++) {
                                if (fChangeMode == !(nChanges & nMask)) {
                                    fChangeMode = !fChangeMode;
                                    if (nPos != k) {
                                        s2.append(cv.to_bytes(sU.substr(nPos, k - nPos)));
                                        nPos = k;
                                    }
                                    if (fChangeMode)
                                        s2.append("<span color=\"#FF0000\">");
                                    else
                                        s2.append("</span>");
                                }
                                nMask >>= 1;
                            }
                            if (nPos < sU.length())
                                s2.append(cv.to_bytes(sU.substr(nPos, sU.length() - nPos)));
                            if (fChangeMode)
                                s2.append("</span>");
                            break;
                        }
                    }
                    m_pangoLayout->set_markup(s2);
                    cr->restore();
                }
                else
                    m_pangoLayout->set_markup(s);

                m_pangoLayout->get_pixel_size(nTextWidth, nTextHeight);
                cr->move_to((nX2 + nX - nTextWidth) / 2, nY + (m_nRowHeight - nTextHeight) / 2);
                m_pangoLayout->show_in_cairo_context(cr);
                ++p;
            }
            ++nAddress;
            nX = nX2;
        }
        nAddress = nAddress2 + m_nBlockSize;
        nY = nY2;
    }
    m_pangoLayout->set_markup(Glib::Markup::escape_text(s));

    // Если выделена ячейка, то рисуем рамку для этой ячейки
    if ((m_nCurrCol != -1) && (m_nCurrRow != -1)) {
        auto rect = get_cell_rect(m_nCurrCol, m_nCurrRow);
        cr->set_source_rgb(0.1, 0.1, 0.1);
        cr->set_line_width(1.0);
        cr->rectangle(rect.get_x(), rect.get_y(), rect.get_width(), rect.get_height());
        cr->stroke();

        // Если включен режим редактирования, рисуем каретку ввода
        if (m_fEditMode) {
            uint nAddress;
            if (cell_to_address(m_nCurrCol, m_nCurrRow, nAddress)) {
                auto p = &m_pData[nAddress];
                nFormat = address_to_format(nAddress);
                s = byte_to_string((unsigned char)*p, nFormat);
                if ((m_nEditCellPos >= 0) && (m_nEditCellPos < s.length())) {
                    m_pangoLayout->set_text(s);
                    m_pangoLayout->get_pixel_size(nTextWidth, nTextHeight);
                    nX = rect.get_x() + (rect.get_width() - nTextWidth) / 2;
                    n = nTextWidth / s.length();
                    nY = rect.get_y() + (rect.get_height() - nTextHeight) / 2 + nTextHeight;
                    nX += m_nEditCellPos * n;

                    bool fReadOnly;
                    switch (nFormat) {
                    case BYTE_FORMAT_BIN:
                        fReadOnly = !(m_nByteAccess & (1 << m_nEditCellPos));
                        break;

                    case BYTE_FORMAT_OCT:
                        fReadOnly = !(m_nByteAccess & (7 << (m_nEditCellPos * 3)));
                        break;

                    case BYTE_FORMAT_HEX:
                        fReadOnly = !(m_nByteAccess & (0xF << (m_nEditCellPos * 4)));
                        break;

                    default:
                        fReadOnly = (0 == m_nByteAccess);
                        break;
                    }
                    if (fReadOnly)
                        cr->set_source_rgb(0.5, 0.5, 0.5);
                    else
                        cr->set_source_rgb(0, 0, 0);
                    cr->move_to(nX, nY);
                    cr->line_to(nX + n, nY);
                    cr->stroke();
                }
            }
        }
    }
    cr->restore();

    cr->restore();
}

bool CCardDataGrid::on_key_pressed(guint keyval, guint keycode, Gdk::ModifierType state) {
    switch (keyval) {
    case GDK_KEY_Home:  // Клавиша Home
        switch (state & (Gdk::ModifierType::SHIFT_MASK | Gdk::ModifierType::CONTROL_MASK |
                         Gdk::ModifierType::ALT_MASK)) {
        case Gdk::ModifierType::CONTROL_MASK:  // Ctrl + Home
            int nCol, nRow;
            if (get_next_tab_cell(-1, -1, nCol, nRow, true)) {
                set_current_cell(nCol, nRow, true, true);
                return true;
            }
            break;

        case (Gdk::ModifierType)0:
            if (m_fEditMode) {
                if ((m_nCurrCol != -1) && (m_nCurrRow != -1) && (m_nEditCellPos > 0)) {
                    m_nEditCellPos = 0;
                    queue_draw();
                }
                return true;
            }
            if (m_nCurrRow != -1)
                set_current_cell(m_nFixedCols, m_nCurrRow, true, true);
            return true;
        }
        break;

    case GDK_KEY_End:  // Клавиша End
        switch (state & (Gdk::ModifierType::SHIFT_MASK | Gdk::ModifierType::CONTROL_MASK |
                         Gdk::ModifierType::ALT_MASK)) {
        case Gdk::ModifierType::CONTROL_MASK:  // Ctrl + End
            {
                int nCol, nRow;
                if (get_next_tab_cell(-1, -1, nCol, nRow)) {
                    set_current_cell(nCol, nRow, true, true);
                    return true;
                }
                break;
            }

        case (Gdk::ModifierType)0:
            if (m_fEditMode) {
                uint nAddress = 0;
                if ((m_nCurrCol != -1) && (m_nCurrRow != -1) &&
                    cell_to_address(m_nCurrCol, m_nCurrRow, nAddress)) {
                    int nLength = get_address_text_length(nAddress);
                    if ((m_nEditCellPos + 1) < nLength) {
                        m_nEditCellPos = (nLength - 1);
                        queue_draw();
                    }
                }
                return true;
            }
            if (m_nCurrRow != -1)
                set_current_cell(get_col_count() - 1, m_nCurrRow, true, true);
            return true;
        }
        break;

    case GDK_KEY_Left:  // Клавиша Left Arrow (стрелка влево)
        if (m_fEditMode) {
            if (m_nEditCellPos > 0) {
                --m_nEditCellPos;
                queue_draw();
                return true;
            }
            else {
                auto nCol = get_next_tab_col(m_nCurrCol, true);
                if (nCol != -1) {
                    set_current_cell(nCol, m_nCurrRow, true, true);
                    uint nAddress;
                    if (cell_to_address(m_nCurrCol, m_nCurrRow, nAddress)) {
                        m_nEditCellPos = get_address_text_length(nAddress) - 1;
                        queue_draw();
                        return true;
                    }
                }
            }
        }
        else {
            auto nCol = get_next_tab_col(m_nCurrCol, true);
            if (nCol != -1) {
                set_current_cell(nCol, m_nCurrRow, true, true);
                return true;
            }
        }
        break;

    case GDK_KEY_Right:  // Клавиша Right Arrow (стрелка вправо)
        if (m_fEditMode) {
            uint nAddress;
            if (cell_to_address(m_nCurrCol, m_nCurrRow, nAddress)) {
                if ((m_nEditCellPos + 1) < get_address_text_length(nAddress)) {
                    ++m_nEditCellPos;
                    queue_draw();
                    return true;
                }
                else {
                    auto nCol = get_next_tab_col(m_nCurrCol);
                    if (nCol != -1) {
                        m_nEditCellPos = 0;
                        set_current_cell(nCol, m_nCurrRow, true, true);
                        return true;
                    }
                }
            }
        }
        else {
            auto nCol = get_next_tab_col(m_nCurrCol);
            if (nCol != -1) {
                set_current_cell(nCol, m_nCurrRow, true, true);
                return true;
            }
        }
        break;

    case GDK_KEY_Up:
        {
            auto nRow = get_next_tab_row(m_nCurrRow);
            if (nRow != -1) {
                set_current_cell(m_nCurrCol, nRow, true, true);
                return true;
            }
            break;
        }

    case GDK_KEY_Down:
        {
            auto nRow = get_next_tab_row(m_nCurrRow, true);
            if (nRow != -1) {
                set_current_cell(m_nCurrCol, nRow, true, true);
                return true;
            }
            break;
        }

    case GDK_KEY_Tab:
        switch (state & (Gdk::ModifierType::SHIFT_MASK | Gdk::ModifierType::CONTROL_MASK |
                         Gdk::ModifierType::ALT_MASK)) {
        case Gdk::ModifierType::SHIFT_MASK:
            {
                int nCol, nRow;
                if (get_next_tab_cell(m_nCurrCol, m_nCurrRow, nCol, nRow, true)) {
                    set_current_cell(nCol, nRow, true, true);
                    return true;
                }
                break;
            }

        case (Gdk::ModifierType)0:
            {
                int nCol, nRow;
                if (get_next_tab_cell(m_nCurrCol, m_nCurrRow, nCol, nRow)) {
                    set_current_cell(nCol, nRow, true, true);
                    return true;
                }
                break;
            }
        }
        break;

    case GDK_KEY_Return:  // Клавиша Enter
        if ((state & (Gdk::ModifierType::SHIFT_MASK | Gdk::ModifierType::CONTROL_MASK |
                      Gdk::ModifierType::ALT_MASK)) == (Gdk::ModifierType)0) {
            if (m_nByteAccess != 0)
                EnableEditMode();
            return true;
        }
        break;

    case GDK_KEY_Escape:  // Клавиша ESC
        if (m_fEditMode) {
            EnableEditMode(false);
            return true;
        }
        break;

    case GDK_KEY_BackSpace:  // Клавиша Backspace
        if (m_fEditMode) {
            uint nAddress;
            if (m_nEditCellPos > 0) {
                --m_nEditCellPos;
                queue_draw();
                if (!cell_to_address(m_nCurrCol, m_nCurrRow, nAddress))
                    return false;
            }
            else {
                int nCol, nRow;
                if (!get_next_tab_cell(m_nCurrCol, m_nCurrRow, nCol, nRow, true))
                    return false;
                set_current_cell(nCol, nRow, true, true);
                if (!cell_to_address(m_nCurrCol, m_nCurrRow, nAddress))
                    return false;
                m_nEditCellPos = get_address_text_length(nAddress) - 1;
                queue_draw();
            }
            if ((m_nByteAccess != 0) && (nAddress < m_nOldDataSize)) {
                auto nValue = (uint)m_pData[nAddress];
                auto nOldValue = (uint)m_pOldData[nAddress];
                if (nValue != nOldValue) {
                    auto nNewValue = nValue;
                    auto nFormat = address_to_format(nAddress);
                    uint nMask;
                    switch (nFormat) {
                    case BYTE_FORMAT_BIN:
                        if (m_nEditCellPos < 8) {
                            nMask = (1 << (7 - m_nEditCellPos));
                            if (nOldValue & nMask)
                                nNewValue |= nMask;
                            else
                                nNewValue &= ~nMask;
                        }
                        break;

                    case BYTE_FORMAT_DEC:
                        {
                            std::wstring_convert<std::codecvt_utf8<char32_t>, char32_t> cv;
                            auto sNew = cv.from_bytes(byte_to_string(nNewValue, nFormat));
                            auto sOld = cv.from_bytes(byte_to_string(nOldValue, nFormat));
                            if (m_nEditCellPos < sNew.length()) {
                                sNew[m_nEditCellPos] = sOld[m_nEditCellPos];
                                auto n = strtoul(cv.to_bytes(sNew).c_str(), nullptr, 10);
                                if (n != ULONG_MAX)
                                    nNewValue = n;
                            }
                            break;
                        }

                    case BYTE_FORMAT_HEX:
                        if (m_nEditCellPos < 2) {
                            nMask = (0xF << ((1 - m_nEditCellPos) * 4));
                            nNewValue = (nNewValue & ~nMask) | (nOldValue & nMask);
                        }
                        break;

                    case BYTE_FORMAT_OCT:
                        if (m_nEditCellPos < 3) {
                            nMask = (7 << ((2 - m_nEditCellPos) * 3));
                            nNewValue = (nNewValue & ~nMask) | (nOldValue & nMask);
                        }
                        break;
                    }
                    if (nNewValue != nValue) {
                        auto b = (uint8_t)nNewValue;
                        if (!m_signal_validate_byte.empty())
                            m_signal_validate_byte.emit(nAddress, b);
                        m_pData[nAddress] = b;
                    }
                }
            }
            return true;
        }
        break;

    case GDK_KEY_0:
    case GDK_KEY_1:
    case GDK_KEY_2:
    case GDK_KEY_3:
    case GDK_KEY_4:
    case GDK_KEY_5:
    case GDK_KEY_6:
    case GDK_KEY_7:
    case GDK_KEY_8:
    case GDK_KEY_9:
    case GDK_KEY_a:
    case GDK_KEY_A:
    case GDK_KEY_b:
    case GDK_KEY_B:
    case GDK_KEY_c:
    case GDK_KEY_C:
    case GDK_KEY_d:
    case GDK_KEY_D:
    case GDK_KEY_e:
    case GDK_KEY_E:
    case GDK_KEY_f:
    case GDK_KEY_F:
        if ((m_nCurrCol != -1) && (m_nCurrRow != -1) && (m_nByteAccess != 0)) {
            uint nAddress;
            if (cell_to_address(m_nCurrCol, m_nCurrRow, nAddress)) {
                if (!m_fEditMode)
                    EnableEditMode();
                auto fOk = false;
                auto nValue = (uint)m_pData[nAddress];
                auto nNewValue = nValue;
                int nTextLength = 0;
                auto nFormat = address_to_format(nAddress);
                switch (nFormat) {
                case BYTE_FORMAT_BIN:
                    nTextLength = 8;
                    if (((GDK_KEY_0 == keyval) || (GDK_KEY_1 == keyval)) &&
                        (m_nEditCellPos < nTextLength)) {
                        uint nMask = (1 << (7 - m_nEditCellPos));
                        if (GDK_KEY_1 == keyval)
                            nNewValue |= nMask;
                        else
                            nNewValue &= ~nMask;
                        fOk = true;
                    }
                    break;

                case BYTE_FORMAT_DEC:
                    nTextLength = 3;
                    if ((keyval >= GDK_KEY_0) && (keyval <= GDK_KEY_9) &&
                        (m_nEditCellPos < nTextLength)) {
                        std::wstring_convert<std::codecvt_utf8<char32_t>, char32_t> cv;
                        auto s = cv.from_bytes(byte_to_string(nNewValue, nFormat));
                        if (m_nEditCellPos < s.length()) {
                            s[m_nEditCellPos] = '0' + (keyval - GDK_KEY_0);
                            auto n = strtoul(cv.to_bytes(s).c_str(), nullptr, 10);
                            if ((n != ULONG_MAX) && (n < std::numeric_limits<uint8_t>::max())) {
                                nNewValue = n;
                                fOk = true;
                            }
                        }
                    }
                    break;

                case BYTE_FORMAT_HEX:
                    nTextLength = 2;
                    if (m_nEditCellPos < nTextLength) {
                        uint a, nBitOffs;
                        if ((keyval >= GDK_KEY_0) && (keyval <= GDK_KEY_9))
                            a = (keyval - GDK_KEY_0);
                        else if ((keyval >= GDK_KEY_A) && (keyval <= GDK_KEY_F))
                            a = 10 + (keyval - GDK_KEY_A);
                        else
                            a = 10 + (keyval - GDK_KEY_a);
                        nBitOffs = (1 - m_nEditCellPos) * 4;
                        nNewValue = (nNewValue & ~(0xf << nBitOffs)) | (a << nBitOffs);
                        fOk = true;
                    }
                    break;

                case BYTE_FORMAT_OCT:
                    nTextLength = 3;
                    if ((keyval >= GDK_KEY_0) && (keyval <= GDK_KEY_7) &&
                        (m_nEditCellPos < nTextLength)) {
                        uint a = (keyval - GDK_KEY_0);
                        uint nBitOffs = (2 - m_nEditCellPos) * 3;
                        nNewValue = (nNewValue & ~(7 << nBitOffs)) | (a << nBitOffs);
                        fOk = true;
                    }
                    break;
                }

                if (fOk) {
                    if (nValue != nNewValue) {
                        auto b = (uint8_t)nNewValue;
                        if (!m_signal_validate_byte.empty())
                            m_signal_validate_byte.emit(nAddress, b);
                        m_pData[nAddress] = b;
                        queue_draw();
                        fOk = (b == (uint8_t)nNewValue);
                    }
                    if (++m_nEditCellPos >= nTextLength) {
                        m_nEditCellPos = 0;
                        int nCol, nRow;
                        if (get_next_tab_cell(m_nCurrCol, m_nCurrRow, nCol, nRow))
                            set_current_cell(nCol, nRow, true, true);
                    }
                }
                if (!fOk)
                    error_bell();
                return true;
            }
        }
        break;
    }
    // the event has not been handled
    return false;
}

void CCardDataGrid::on_mouse_click(int n_press, double x, double y) {
    // Если двойной клик,
    if (n_press > 1) {
        int nCol, nRow;
        mouse_to_cell(x, y, nCol, nRow);
        if ((nCol >= m_nFixedCols) && (nRow >= kFixedRows)) {
            grab_focus();
            set_current_cell(nCol, nRow, true, true);
            if (!m_fEditMode && (m_nByteAccess != 0))
                EnableEditMode();
            if (m_fEditMode) {
                auto nPos = mouse_to_edit_cell_pos(x, y);
                if ((nPos != -1) && (m_nEditCellPos != nPos)) {
                    m_nEditCellPos = nPos;
                    queue_draw();
                }
            }
        }
        return;
    }

    if ((y >= m_nRowHeight) && (y < get_height())) {
        auto nBlockIdx = get_block_by_y(y);

        if (x < m_oColWidths[0])
            if ((nBlockIdx < m_oCheckBoxes.size()) &&
                ((m_oCheckBoxes[nBlockIdx] & (CHECKBOX_F_VISIBLE | CHECKBOX_F_SENSITIVE)) ==
                 (CHECKBOX_F_VISIBLE | CHECKBOX_F_SENSITIVE)))
                if (get_checkbox_rect(nBlockIdx).contains_point(x, y)) {
                    if (!m_signal_checkbox_toggle.empty())
                        m_signal_checkbox_toggle.emit(nBlockIdx);
                    return;
                }
        int nFixedWidth = get_col_dx(m_nFixedCols);
        if (x >= nFixedWidth) {
            int nCol, nRow;
            mouse_to_cell(x, y, nCol, nRow);
            if ((nCol != -1) && (nRow != -1)) {
                grab_focus();
                set_current_cell(nCol, nRow, true, true);

                if (m_fEditMode) {
                    auto nPos = mouse_to_edit_cell_pos(x, y);
                    if ((nPos != -1) && (m_nEditCellPos != nPos)) {
                        m_nEditCellPos = nPos;
                        queue_draw();
                    }
                }
            }
        }
    }
}

void CCardDataGrid::on_vscroll() {
    queue_draw();
}

void CCardDataGrid::on_hscroll() {
    queue_draw();
}

void CCardDataGrid::on_resize(int width, int height) {
    update_scroll_info();
}

void CCardDataGrid::on_property_vadjustment_change() {
    auto adj = get_vadjustment();
    if (adj != nullptr) {
        adj->signal_value_changed().connect(sigc::mem_fun(*this, &CCardDataGrid::on_vscroll));
        update_scroll_info(true, false);
    }
}

void CCardDataGrid::on_property_hadjustment_change() {
    auto adj = get_hadjustment();
    if (adj != nullptr) {
        adj->signal_value_changed().connect(sigc::mem_fun(*this, &CCardDataGrid::on_hscroll));
        update_scroll_info(false, true);
    }
}

void CCardDataGrid::update_canvas_size() {
    update_scroll_info();
    queue_draw();
}

int CCardDataGrid::get_col_dx(int nCol, int nStart) const {
    int res = 0;
    for (auto i = nStart; i < nCol; i++)
        res += m_oColWidths[i];
    return res;
}

Gdk::Rectangle CCardDataGrid::get_checkbox_rect(uint nBlockIdx) const {
    auto adj = get_vadjustment();
    int nVPos = adj ? static_cast<int>(adj->get_value()) : 0;
    // Размер флажка (ширина = высота checkbox)
    auto nCheckboxSize = get_checkbox_size();
    // Смещение по высоте флажка
    auto nCheckboxDy = (m_nRowHeight - nCheckboxSize) / 2;

    Gdk::Rectangle rect;
    rect.set_x(kFixedCellHPadding);
    rect.set_y(((m_nRowCount - 1 - nBlockIdx) * m_nRowHeight) - nVPos + nCheckboxDy);
    rect.set_width(nCheckboxSize);
    rect.set_height(nCheckboxSize);
    return rect;
}

int CCardDataGrid::get_checkbox_size() const {
    return (m_nRowHeight - kCellVPadding - kCellVPadding - m_nRowHeight / 4);
}

void CCardDataGrid::draw_checkbox(const Cairo::RefPtr<Cairo::Context>& cr, int x, int y, int nSize,
                                  bool fActive, bool fSensitive) {
    cr->save();
    cr->set_line_width(1.0);
    if (fSensitive)
        cr->set_source_rgb(1.0, 1.0, 1.0);
    else
        cr->set_source_rgb(0.6, 0.6, 0.6);
    cr->rectangle(x, y, nSize, nSize);
    cr->fill();
    if (fSensitive)
        cr->set_source_rgb(0, 0, 0);
    else
        cr->set_source_rgb(0.1, 0.1, 0.1);
    cr->rectangle(x, y, nSize, nSize);
    cr->stroke();
    if (fActive) {
        auto nCrossSize = (nSize / 4);
        auto nX1 = (x + nCrossSize);
        auto nX2 = (x + nSize - nCrossSize);
        auto nY1 = (y + nCrossSize);
        auto nY2 = (y + nSize - nCrossSize);
        cr->move_to(nX1, nY1);
        cr->line_to(nX2, nY2);
        cr->move_to(nX2, nY1);
        cr->line_to(nX1, nY2);
    }
    cr->stroke();
    cr->restore();
}

int CCardDataGrid::get_col_by_x(int x) const {
    auto nFixedWidth = get_col_dx(m_nFixedCols);
    int dx;
    if (x < nFixedWidth)
        return get_cols_per_width(x, 0, m_nFixedCols, dx);

    auto adj = get_hadjustment();
    int nHPos = adj ? static_cast<int>(adj->get_value()) : 0;

    auto left = nFixedWidth - nHPos;
    for (size_t i = m_nFixedCols; i < m_oColWidths.size(); i++) {
        left += m_oColWidths[i];
        if (x < left)
            return i;
    }
    return -1;
}

int CCardDataGrid::get_block_by_y(int y) const {
    auto adj = get_vadjustment();
    int nVPos = adj ? static_cast<int>(adj->get_value()) : 0;
    return (int)m_nVisibleBlockIdx + ((m_nRowCount * m_nRowHeight) - nVPos - y) / (int)m_nRowHeight;
}

int CCardDataGrid::get_row_by_y(int y) const {
    if (y < (kFixedRows * m_nRowHeight))
        return 0;
    auto nBlockCount = get_visible_block_count();
    auto adj = get_vadjustment();
    int nVPos = adj ? static_cast<int>(adj->get_value()) : 0;
    int nBlockIdx = ((m_nRowCount * m_nRowHeight) - nVPos - y) / (int)m_nRowHeight;
    return (nBlockIdx < nBlockCount) ? (kFixedRows + nBlockCount - 1 - nBlockIdx) : -1;
}

int CCardDataGrid::mouse_to_edit_cell_pos(int x, int y) const {
    uint nAddress;
    if (cell_to_address(m_nCurrCol, m_nCurrRow, nAddress)) {
        auto s = address_to_text(nAddress);
        m_pangoLayout->set_text(s);
        int nTextWidth, nTextHeight;
        m_pangoLayout->get_pixel_size(nTextWidth, nTextHeight);
        auto rect = get_cell_rect(m_nCurrCol, m_nCurrRow);
        auto nX = rect.get_x() + (rect.get_width() - nTextWidth) / 2;
        auto n = nTextWidth / s.length();
        auto nY = rect.get_y() + (rect.get_height() - nTextHeight) / 2;
        if ((x >= nX) && (x < (nX + nTextWidth)) && (y >= nY) && (y < (nY + nTextHeight)))
            return (x - nX) / n;
    }
    return -1;
}

Gdk::Rectangle CCardDataGrid::get_cell_rect(int nCol, int nRow) const {
    Gdk::Rectangle rect;
    if (nCol < m_nFixedCols)
        rect.set_x(get_col_dx(nCol));
    else {
        auto adj = get_hadjustment();
        int nHPos = adj ? static_cast<int>(adj->get_value()) : 0;
        rect.set_x(get_col_dx(nCol) - nHPos);
    }

    if (nRow < kFixedRows)
        rect.set_y(m_nRowHeight * nRow);
    else {
        auto adj = get_vadjustment();
        int nVPos = adj ? static_cast<int>(adj->get_value()) : 0;

        rect.set_y(nRow * m_nRowHeight - nVPos);
    }
    rect.set_width(m_oColWidths[nCol]);
    rect.set_height(m_nRowHeight);
    return rect;
}

CCardDataGrid::CByteAttrsMap::const_iterator CCardDataGrid::find_byte_attrs(uint nAddress) const {
    for (auto it = m_oAttrMap.cbegin(); it != m_oAttrMap.cend(); it++) {
        if (nAddress < it->first)
            break;
        if (nAddress < (it->first + it->second.m_nCount))
            return it;
    }
    return m_oAttrMap.cend();
}

CCardDataGrid::byte_format CCardDataGrid::address_to_format(uint nAddress) const {
    auto nFormat = m_nByteFormat;
    auto itA = find_byte_attrs(nAddress);
    if ((itA != m_oAttrMap.cend()) && (itA->second.m_nFormat != BYTE_FORMAT_UNDEF))
        nFormat = itA->second.m_nFormat;
    return nFormat;
}

Glib::ustring CCardDataGrid::byte_to_string(uint nByte, byte_format nFormat) const {
    Glib::ustring s;
    switch (nFormat) {
    case BYTE_FORMAT_HEX:
        s = Glib::ustring::sprintf("%.2X", nByte);
        break;

    case BYTE_FORMAT_DEC:
        s = Glib::ustring::sprintf("%.3u", nByte);
        break;

    case BYTE_FORMAT_OCT:
        s = Glib::ustring::sprintf("%.3o", nByte);
        break;

    case BYTE_FORMAT_BIN:
        {
            s.reserve(8);
            uint nMask = 0x80;
            for (uint i = 0; i < 8; i++) {
                s.append((nByte & nMask) ? "1" : "0");
                nMask >>= 1;
            }
            break;
        }
    }
    return s;
}

void CCardDataGrid::Update_fix_col_widths() {
    auto c = std::min(m_nFixedCols, (int)m_oColWidths.size());
    auto nBlockCount = get_total_block_count();
    uint nMaxSBlocks;
    auto nSectorCount = calc_sector_count(nBlockCount, nMaxSBlocks);
    auto nPadding = kFixedCellHPadding + kFixedCellHPadding;
    int cx, cy;
    Glib::ustring s;
    for (int i = 0; i < c; i++) {
        m_oColWidths[i] = 0;
        if (i < m_oTitles.size()) {
            m_pangoLayout->set_text(m_oTitles[i]);
            m_pangoLayout->get_pixel_size(cx, cy);
            m_oColWidths[i] = (cx + nPadding);
        }
        switch (i) {
        case 0:
            s = Glib::ustring::format(nBlockCount);
            break;

        case 1:
            s = Glib::ustring::format(nSectorCount);
            break;

        case 2:
            s = Glib::ustring::format(nMaxSBlocks);
            break;

        default:
            s.clear();
            break;
        }
        if (!s.empty()) {
            m_pangoLayout->set_text(s);
            m_pangoLayout->get_pixel_size(cx, cy);
            if (!m_oCheckBoxes.empty() && (0 == i))
                cx += (get_checkbox_size() + kCellSpacing);

            m_oColWidths[i] = std::max(cx + nPadding, m_oColWidths[i]);
        }
    }
    update_canvas_size();
}

void CCardDataGrid::update_scroll_info(bool fVert, bool fHorz) {
    if (fVert) {
        auto adj = get_vadjustment();
        if (adj != nullptr) {
            auto nFixedHeight = (kFixedRows * m_nRowHeight);
            auto nUpper = (m_nRowCount * m_nRowHeight) - nFixedHeight;
            auto pos = adj->get_value();
            if (pos > nUpper)
                pos = nUpper;
            adj->configure(pos, 0, nUpper, m_nRowHeight, 10.0,
                           std::max(get_height() - (int)nFixedHeight, 0));
        }
    }
    if (fHorz) {
        auto adj = get_hadjustment();
        if (adj != nullptr) {
            auto nFixedCols = get_real_fixed_col_count();
            auto nFixedWidth = get_col_dx(nFixedCols);
            auto nUpper = get_col_dx(get_col_count(), nFixedCols);
            auto pos = adj->get_value();
            if (pos > nUpper)
                pos = nUpper;
            adj->configure(pos, 0, nUpper, 1.0, 10.0, std::max(get_width() - (int)nFixedWidth, 0));
        }
    }
}
