#include "ilr_cpp_helpers.h"

#include <time.h>

#include <iomanip>  // для std::put_time
#include <memory>
#include <sstream>

namespace ilr {

CILRException::CILRException(ilr_status nCode) :
    m_nCode(nCode) {
}

const char* CILRException::what() const throw() {
    return ilr_get_error_text(m_nCode);
}

CCardUID::CCardUID() {
    Clear();
}

CCardUID::CCardUID(const ilr_card_uid& rUID) {
    nLength = rUID.nLength;
    memcpy(aBytes, rUID.aBytes, sizeof(aBytes));
}

CCardUID::CCardUID(const uint64_t& number, size_t nSize) {
    Assign(&number, nSize);
}

CCardUID::CCardUID(const void* pData, size_t nSize) {
    Assign(pData, nSize);
}

CCardUID::CCardUID(uint8_t nEmSeries, uint16_t nEmNumber, uint16_t nFacility) {
    em_marine.nSeries = nEmSeries;
    em_marine.nNumber = nEmNumber;
    em_marine.nFacility = nFacility;
    nLength = 5;
}

std::string CCardUID::ToString() const {
    std::string s;
    auto len = GetLength();
    s.resize(len * 2 + 1);
    size_t j = 0;
    for (int i = (len - 1); i >= 0; i--, j += 2)
        snprintf(&s[j], s.size() - j, "%.2X", aBytes[i]);
    s.resize(len * 2);
    return s;
}

bool CCardUID::TryParse(const char* pStr) {
    auto nStrLen = strlen(pStr);
    if (0 == nStrLen)
        return false;

    uint8_t aNewBytes[15] = {0};
    uint nNewLength = 0;
    bool fHi = false;
    uint n;

    for (const char* p = (pStr + nStrLen); p-- != pStr;) {
        if ((*p >= '0') && (*p <= '9'))
            n = (*p - '0');
        else if ((*p >= 'A') && (*p <= 'F'))
            n = 10 + (*p - 'A');
        else if ((*p >= 'a') && (*p <= 'f'))
            n = 10 + (*p - 'a');
        else
            return false;
        if (fHi) {
            aNewBytes[nNewLength] |= (n << 4);
            if (++nNewLength == std::size(aNewBytes))
                break;
        }
        else
            aNewBytes[nNewLength] = n;
        fHi = !fHi;
    }
    Clear();
    Assign(aNewBytes, nNewLength);
    return true;
}

const ilr_mf_classic_key CMifareClassicKey::kDefault = 0xffffffffffff;

CMifareClassicKey::CMifareClassicKey() {
    (ilr_mf_classic_key&)(*this) = kDefault;
}

CMifareClassicKey::CMifareClassicKey(const void* pData, size_t nSize) {
    m_nKey = 0;
    if (nSize > 6)
        nSize = 6;
    if (nSize != 0)
        memcpy(&m_nKey, pData, nSize);
}

CMifareClassicKey::CMifareClassicKey(const ilr_mf_classic_key& other) {
    (ilr_mf_classic_key&)(*this) = other;
}

std::string CMifareClassicKey::ToString() const {
    char szBuf[24];
    auto c = snprintf(szBuf, std::size(szBuf), "%.12llx", m_nKey);
    if (c <= 0)
        return std::string();
    return std::string(szBuf, c);
}

bool CMifareClassicKey::TryParse(const char* pStr) {
    char* p = nullptr;
    errno = 0;
    auto n = strtoull(pStr, &p, 16);
    if ((errno != 0) || (p == pStr) || (n > 0xffffffffffff))
        return false;
    m_nKey = n;
    return true;
}

CMifarePlusKey::CMifarePlusKey() {
    memset(a, 0xff, sizeof(a));
}

CMifarePlusKey::CMifarePlusKey(const ilr_mf_plus_key& other) {
    *this = other;
}

CMifarePlusKey::CMifarePlusKey(const uint64_t& lo, const uint64_t& hi) {
    ll.lo = lo;
    ll.hi = hi;
}

CMifarePlusKey::CMifarePlusKey(const ilr_mf_classic_key& other) {
    *this = other;
}

int CMifarePlusKey::Compare(const CMifarePlusKey& other) const {
    return memcmp(a, other.a, sizeof(a));
}

std::string CMifarePlusKey::ToString(const char* pPartDelimiter) const {
    std::stringstream ss;
    ss.imbue(std::locale("C"));
    ss << std::hex << std::setfill('0') << std::setw(16) << ll.hi;
    if (pPartDelimiter != nullptr)
        ss << pPartDelimiter;
    ss << std::hex << std::setfill('0') << std::setw(16) << ll.lo;
    return ss.str();
}

bool CMifarePlusKey::TryParse(const char* pStr) {
    char* p = nullptr;
    errno = 0;
    auto n = strtoull(pStr, &p, 16);
    if ((errno != 0) || (p == pStr))
        return false;
    if ('\0' == *p) {
        ll.hi = 0;
        ll.lo = n;
        return true;
    }
    while ((*p != '\0') && !isxdigit((int)*p))
        ++p;
    ll.hi = n;
    pStr = p;
    n = strtoull(pStr, &p, 16);
    if ((errno != 0) || (p == pStr))
        return false;
    ll.lo = n;
    return true;
}

CILRHandle::CILRHandle() :
    m_h(nullptr) {
}

CILRHandle::CILRHandle(ilr_handle h) :
    m_h(h) {
}

CILRHandle::CILRHandle(CILRHandle&& other) {
    m_h = other.m_h;
    other.m_h = nullptr;
}

CILRHandle::~CILRHandle() {
    if (m_h != nullptr)
        ilr_close_handle(m_h);
}

CILRHandle& CILRHandle::operator=(CILRHandle&& other) {
    if (m_h != nullptr)
        ilr_close_handle(m_h);
    m_h = other.m_h;
    other.m_h = nullptr;
    return *this;
}

CILRHandle::operator ilr_handle() const {
    return m_h;
}

CILRHandle::operator bool() const {
    return m_h != nullptr;
}

void CILRHandle::Swap(CILRHandle& other) noexcept {
    std::swap(m_h, other.m_h);
}

ilr_handle CILRHandle::Get() const {
    return m_h;
}

void CILRHandle::Close() {
    if (m_h != nullptr) {
        ILRCheck(ilr_close_handle(m_h));
        m_h = nullptr;
    }
}

void CILRHandle::Attach(ilr_handle h) {
    if (m_h != nullptr)
        ILRCheck(ilr_close_handle(m_h));
    m_h = h;
}

ilr_handle CILRHandle::Detach() {
    ilr_handle res = m_h;
    m_h = nullptr;
    return res;
}

CAsyncCommand::CAsyncCommand() {
}

CAsyncCommand::CAsyncCommand(ilr_handle h) :
    CILRHandle(h) {
}

CAsyncCommand::CAsyncCommand(CAsyncCommand&& other) :
    CILRHandle(other.m_h) {
    other.m_h = nullptr;
}

CAsyncCommand::~CAsyncCommand() {
}

CAsyncCommand& CAsyncCommand::operator=(CAsyncCommand&& other) {
    Attach(other.Detach());
    return *this;
}

CReaderSearch::CReaderSearch() {
}

CReaderSearch::CReaderSearch(ilr_handle h) :
    CILRHandle(h) {
}

CReaderSearch::CReaderSearch(CReaderSearch&& other) :
    CILRHandle(other.m_h) {
    other.m_h = nullptr;
}

CReaderSearch::~CReaderSearch() {
}

CReaderSearch& CReaderSearch::operator=(CReaderSearch&& other) {
    Attach(other.Detach());
    return *this;
}

void CReaderSearch::GetListenPorts(std::vector<unsigned short>& oPorts) const {
    size_t nSize = ILR_AUTOALLOCATE;
    unsigned short* p = nullptr;
    ILRCheck(ilr_search_get_listen_ports(m_h, (unsigned short*)&p, &nSize));

    std::unique_ptr<void, decltype(ilr_free_memory)*> o(p, ilr_free_memory);
    oPorts.assign(p, p + nSize);
}

CReader::CReader() {
}

CReader::~CReader() {
}

CReader& CReader::operator=(CReader&& other) {
    Attach(other.Detach());
    return *this;
}

CReader::CReader(ilr_handle h) :
    CILRHandle(h) {
}

CReader::CReader(CReader&& other) {
    m_h = other.m_h;
    other.m_h = nullptr;
}

void CReader::MfRats(std::vector<uint8_t>& oAts) {
    uint8_t* pAtsBuf = nullptr;
    size_t nSize = ILR_AUTOALLOCATE;
    ILRCheck(ilr_reader_mf_rats(m_h, (uint8_t*)&pAtsBuf, &nSize));
    std::unique_ptr<void, decltype(ilr_free_memory)*> o(pAtsBuf, ilr_free_memory);
    oAts.assign(pAtsBuf, pAtsBuf + nSize);
}

uint32_t CReader::GetSupportedCardTypes(ilr_reader_model nReaderModel) {
    uint32_t res;
    switch (nReaderModel) {
    case ILR_READER_MODEL_Z2_RDALL:
        res = (1 << ILR_CARD_EM_MARINE) | (1 << ILR_CARD_HID) | (1 << ILR_CARD_MF_ULTRALIGHT) |
              (1 << ILR_CARD_MF_ULTRALIGHT_C) | (1 << ILR_CARD_MF_MINI) |
              (1 << ILR_CARD_MF_CLASSIC_1K) | (1 << ILR_CARD_MF_CLASSIC_2K) |
              (1 << ILR_CARD_MF_CLASSIC_4K) | (1 << ILR_CARD_MF_DESFIRE) | (1 << ILR_CARD_MF_PROX);
        break;

    case ILR_READER_MODEL_Z2_MF:
    case ILR_READER_MODEL_Z2_MFI:
    case ILR_READER_MODEL_MATRIX3_NET:
    case ILR_READER_MODEL_CP_Z2_MFI:
    case ILR_READER_MODEL_Z2_MF_CCID:
    case ILR_READER_MODEL_MATRIX6:
        res = (1 << ILR_CARD_MF_ULTRALIGHT) | (1 << ILR_CARD_MF_ULTRALIGHT_C) |
              (1 << ILR_CARD_MF_MINI) | (1 << ILR_CARD_MF_CLASSIC_1K) |
              (1 << ILR_CARD_MF_CLASSIC_2K) | (1 << ILR_CARD_MF_CLASSIC_4K) |
              (1 << ILR_CARD_MF_PLUS) | (1 << ILR_CARD_MF_PLUS_1K) | (1 << ILR_CARD_MF_PLUS_2K) |
              (1 << ILR_CARD_MF_PLUS_4K) | (1 << ILR_CARD_SMART_MX_MF1K) |
              (1 << ILR_CARD_SMART_MX_MF4K) | (1 << ILR_CARD_MF_DESFIRE) | (1 << ILR_CARD_MF_PROX);
        break;

    case ILR_READER_MODEL_Z2_E_HTZ_RF:
        res = (1 << ILR_CARD_EM_MARINE) | (1 << ILR_CARD_HID) | (1 << ILR_CARD_MF_ULTRALIGHT) |
              (1 << ILR_CARD_MF_ULTRALIGHT_C) | (1 << ILR_CARD_MF_MINI) |
              (1 << ILR_CARD_MF_CLASSIC_1K) | (1 << ILR_CARD_MF_CLASSIC_2K) |
              (1 << ILR_CARD_MF_CLASSIC_4K) | (1 << ILR_CARD_MF_DESFIRE) | (1 << ILR_CARD_MF_PROX) |
              (1 << ILR_CARD_TEMIC) | (1 << ILR_CARD_DALLAS) | (1 << ILR_CARD_COD433) |
              (1 << ILR_CARD_COD433_FIX);
        break;

    case ILR_READER_MODEL_Z2_E_HT_HOTEL:
        res = (1 << ILR_CARD_EM_MARINE) | (1 << ILR_CARD_TEMIC);
        break;

    case ILR_READER_MODEL_MATRIX5_E_S_RF:
        res = (1 << ILR_CARD_EM_MARINE) | (1 << ILR_CARD_CAME433) | (1 << ILR_CARD_COD433) |
              (1 << ILR_CARD_COD433_FIX);
        break;

    default:
        res = 0;
    }
    return res;
}

uint32_t CReader::GetSupportedRewrittenCardTypes(ilr_reader_model nReaderModel,
                                                 uint32_t nFwVersion) {
    uint32_t res = 0;
    switch (nReaderModel) {
    case ILR_READER_MODEL_Z2_RDALL:
        res = ILR_RWCT_F_MF_ULTRALIGHT;
        if (nFwVersion >= 106)
            res |= ILR_RWCT_F_TEMIC;
        break;

    case ILR_READER_MODEL_Z2_E_HTZ_RF:
        res = ILR_RWCT_F_TEMIC;
        break;

    case ILR_READER_MODEL_MATRIX3_RDALL:
        res = ILR_RWCT_F_MF_ULTRALIGHT;
        break;

    case ILR_READER_MODEL_Z2_MF:
    case ILR_READER_MODEL_MATRIX3_NET:
    case ILR_READER_MODEL_CP_Z2_MFI:
    case ILR_READER_MODEL_Z2_MF_CCID:
        res = ILR_RWCT_F_MF_ULTRALIGHT | ILR_RWCT_F_MF_CLASSIC;
        break;

    case ILR_READER_MODEL_Z2_MFI:
    case ILR_READER_MODEL_MATRIX6:
        res = ILR_RWCT_F_MF_ULTRALIGHT | ILR_RWCT_F_MF_CLASSIC | ILR_RWCT_F_MF_PLUS_SL3;
        break;

    case ILR_READER_MODEL_Z1_N_Z:         // Em-Marine, Dallas
    case ILR_READER_MODEL_Z2_E_HT_HOTEL:  // Em-Marine, Temic
    case ILR_READER_MODEL_MATRIX5_E_S_RF:
    case ILR_READER_MODEL_UNKNOWN:
    case ILR_READER_MODEL_SIZE:
        break;
    }
    return res;
}

CILR::CILR(bool fInit) {
    if (fInit) {
        ILRCheck(ilr_init());
        m_fInit = true;
    }
}

CILR::~CILR() {
    if (m_fInit)
        ilr_cleanup();
}

void CILR::Init() {
    if (m_fInit)
        return;
    ILRCheck(ilr_init());
    m_fInit = true;
}

void CILR::Finalize() {
    if (m_fInit) {
        ilr_cleanup();
        m_fInit = false;
    }
}

void ILRCheck(ilr_status nCode) {
    if (ILR_FAILED(nCode))
        throw CILRException(nCode);
}

const char* kPortTypeNames[] = {
    "",        // 1
    "COM",     // 2
    "CCID",    // 3
    "Server",  // 4
    "Client",  // 5
    "Proxy"    // 6
};
static_assert(ILR_PORT_TYPE_SIZE == 6);

const char* kReaderModelNames[] = {
    "",                       // 1
    "Z-2 RD_ALL",             // 2
    "Z-2 USB MF",             // 3
    "Z-2 MF-I",               // 4
    "Z-2 EHR",                // 5
    "Z-2 Base",               // 6
    "RF-1996",                // 7
    "Matrix III Rd-All",      // 8
    "Matrix III Net",         // 9
    "CP-Z 2MF",               // 10
    "Matrix V",               // 11
    "Matrix-VI (NFC K Net)",  // 12
    "Z-2 MF CCID"             // 13
};
static_assert(ILR_READER_MODEL_SIZE == 13);

const char* kCardTypeNames[] = {
    "",                                 // 1
    "Em-Marine",                        // 2
    "HID",                              // 3
    "iCode",                            // 4
    "Cod433",                           // 5
    "Cod433 Fix",                       // 6
    "CAME",                             // 7
    "Dallas",                           // 8
    "Temic",                            // 9
    "Mifare UltraLight",                // 10
    "Mifare UltraLight C",              // 11
    "Mifare Mini",                      // 12
    "Mifare Classic 1K",                // 13
    "Mifare Classic 2K",                // 14
    "Mifare Classic 4K",                // 15
    "Mifare Plus",                      // 16
    "Mifare Plus 1K",                   // 17
    "Mifare Plus 2K",                   // 18
    "Mifare Plus 4K",                   // 19
    "Smart MX with Mifare Classic 1K",  // 20
    "Smart MX with Mifare Classic 4K",  // 21
    "Mifare DESFire",                   // 22
    "Mifare ProX"                       // 23
};
static_assert(ILR_CARD_TYPE_SIZE == 23);

const char* kMpTypeNames[] = {
    "",     // 1
    "S",    // 2
    "X",    // 3
    "SE",   // 4
    "EV1",  // 5
    "EV2"   // 6
};
static_assert(ILR_MF_PLUS_TYPE_SIZE == 6);

std::string SdkVersionToStr(uint nVersion) {
    std::ostringstream s;
    if (nVersion != 0) {
        s << ((nVersion >> 24) & 0xff) << "." << ((nVersion >> 16) & 0xff) << "."
          << ((nVersion >> 8) & 0xff);

        auto nType = (nVersion >> 6) & 3;
        if (nType != 3) {
            s << "-";
            switch (nType) {
            case 0:
                s << "alpha";
                break;

            case 1:
                s << "beta";
                break;

            case 2:
                s << "rc";
                break;
            }
            auto revision = (nVersion & 0x3f);
            if (revision != 0)
                s << "." << revision;
        }
    }
    return s.str();
}

std::string ReaderVersionToStr(uint nVersion) {
    std::ostringstream s;
    if (0 == nVersion)
        return s.str();  // Версия не известна

    // Если старшее слово = 0, то формат версии "123" (одно число), иначе "1.2.3.0"
    if (nVersion & 0xffff0000) {
        s << ((nVersion >> 24) & 0xff) << "." << ((nVersion >> 16) & 0xff);
        if (nVersion & 0xffff) {
            s << "." << ((nVersion >> 8) & 0xff);
            if (nVersion & 0xff)
                s << "." << (nVersion & 0xff);
        }
    }
    else
        s << nVersion;

    return s.str();
}

std::string TimeToStr(const int64_t& tTime) {
    struct tm* pTemp = localtime(&tTime);
    if (nullptr == pTemp)
        return std::string();
    std::ostringstream s;
    s << std::put_time(pTemp, "%c");
    return s.str();
}

bool IsMfClassic(ilr_card_type nType) {
    return (nType >= ILR_CARD_MF_MINI) && (nType <= ILR_CARD_MF_CLASSIC_4K);
}

bool IsMfClassicMode(ilr_card_type nType, ilr_mf_plus_sl nSL) {
    return IsMfClassic(nType) || (IsMfPlus(nType) && (ILR_MF_PLUS_SL_SL1 == nSL));
}

bool IsMfPlus(ilr_card_type nType) {
    return (nType >= ILR_CARD_MF_PLUS) && (nType <= ILR_CARD_SMART_MX_MF4K);
}

bool IsMfPlusSL3Mode(ilr_card_type nType, ilr_mf_plus_sl nSL) {
    return IsMfPlus(nType) && (ILR_MF_PLUS_SL_SL3 == nSL);
}

void GetMfBlockInfo(size_t nBlockIdx, size_t& nSectorIdx, size_t& nSBlockIdx,
                    size_t& nSBlockCount) {
    if (nBlockIdx < 128) {
        nSectorIdx = (nBlockIdx / 4);
        nSBlockIdx = (nBlockIdx % 4);
        nSBlockCount = 4;
    }
    else {
        nSectorIdx = (32 + ((nBlockIdx - 128) / 16));
        nSBlockIdx = (nBlockIdx % 16);
        nSBlockCount = 16;
    }
}

std::string CardUIDToStr(ilr_card_type nType, const CCardUID& rUID) {
    std::string s;
    switch (nType) {
    case ILR_CARD_EM_MARINE:
    case ILR_CARD_ICODE:
    case ILR_CARD_COD433:
    case ILR_CARD_COD433_FIX:
    case ILR_CARD_CAME433:
        {
            s.resize(32);
            auto c = snprintf(s.data(), s.size(), "[%.2X%.2X] %.3u,%.5u", rUID.aBytes[4],
                              rUID.aBytes[3], rUID.aBytes[2], *(const uint16_t*)rUID.aBytes);
            if (c < 0)
                throw std::system_error(EFAULT, std::system_category());
            s.resize(static_cast<size_t>(c));
            break;
        }
    case ILR_CARD_HID:
        {
            uint nFacility = 0;
            size_t nFLen = std::min((uint)rUID.nLength - 2, 4u);
            memcpy(&nFacility, &rUID.aBytes[2], nFLen);
            s.resize(32);
            auto c = snprintf(s.data(), s.size(), "[%.*X] %u", (int)nFLen * 2, nFacility,
                              *(const uint16_t*)rUID.aBytes);
            if (c < 0)
                throw std::system_error(EFAULT, std::system_category());
            s.resize(static_cast<size_t>(c));
            break;
        }

    default:
        {
            auto len = rUID.GetLength();
            s.resize(len * 2 + 1);
            size_t j = 0;
            for (int i = (len - 1); i >= 0; i--, j += 2)
                snprintf(&s[j], s.size() - j, "%.2X", rUID.aBytes[i]);
            s.resize(len * 2);
            break;
        }
    }
    return s;
}

uint32_t GetMfAccessBits(const ilr_mf_block_data& rTrailerData) {
    uint32_t res = 0;
    memcpy(&res, &rTrailerData.a[6], 3);
    return res;
}

void SetMfAccessBits(uint32_t nSectorAccess, ilr_mf_block_data& rTrailerData) {
    memcpy(&rTrailerData.a[6], &nSectorAccess, 3);
}

size_t GetMfAreaByBlockIdx(size_t nBlockIdx) {
    return (nBlockIdx < 128) ? (nBlockIdx % 4) : (nBlockIdx % 16) / 5;
}

bool TryGetMfAreaAccess(uint32_t nSectorAccess, size_t nArea, uint32_t& nAreaAccess) {
    uint32_t nSectorAccessBits = (nSectorAccess >> 12);
    uint32_t nSectorValidBits = (nSectorAccessBits ^ (nSectorAccess & 0xFFF));
    nAreaAccess = ((nSectorAccessBits >> nArea) & 1) |
                  (((nSectorAccessBits >> (nArea + 4)) & 1) << 1) |
                  ((((nSectorAccessBits >> (nArea + 8))) & 1) << 2);
    return ((nSectorValidBits >> nArea) & 1) && ((nSectorValidBits >> (nArea + 4)) & 1) &&
           (((nSectorValidBits >> (nArea + 8))) & 1);
}

void SetMfAreaAccess(size_t nArea, uint32_t nAreaAccess, uint32_t& nSectorAccess) {
    auto f = (nAreaAccess & 1) != 0;
    SET_BIT(nSectorAccess, 12 + nArea, f);
    SET_BIT(nSectorAccess, nArea, !f);
    f = (nAreaAccess & 2) != 0;
    SET_BIT(nSectorAccess, 16 + nArea, f);
    SET_BIT(nSectorAccess, 4 + nArea, !f);
    f = (nAreaAccess & 4) != 0;
    SET_BIT(nSectorAccess, 20 + nArea, f);
    SET_BIT(nSectorAccess, 8 + nArea, !f);
}

bool TryGetMfValue(const ilr_mf_block_data& rData, int& nValue) {
    auto fRes =
        (*(int*)&rData.a[0] == *(int*)&rData.a[8] == (*(uint32_t*)&rData.a[4] ^ 0xffffffff));
    if (fRes)
        nValue = *(int*)&rData.a[0];
    return fRes;
}

void SetMfValue(ilr_mf_block_data& rData, int nValue) {
    *(int*)&rData.a[0] = nValue;
    *(int*)&rData.a[4] = (uint32_t)nValue ^ 0xffffffff;
    *(int*)&rData.a[8] = nValue;
}

bool TryGetMfValueAddress(const ilr_mf_block_data& rData, uint8_t& nAddress) {
    auto fRes = (rData.a[12] == (rData.a[13] ^ 0xff) == rData.a[14] == (rData.a[15] ^ 0xff));
    if (fRes)
        nAddress = rData.a[12];
    return fRes;
}

void SetMfValueAddress(ilr_mf_block_data& rData, uint8_t nAddress) {
    rData.a[12] = nAddress;
    rData.a[13] = (nAddress ^ 0xff);
    rData.a[14] = rData.a[12];
    rData.a[15] = rData.a[13];
}

};  // namespace ilr
