#include "CTemicPasswordsDialog.h"

#include "utils.h"

#define HAS_STYLE_PROVIDER_ADD_PROVIDER_FOR_DISPLAY GTKMM_CHECK_VERSION(4, 9, 1)

CTemicPasswordsDialog::CTemicPasswordsDialog() :
    m_nCurrentPassword(-1),
    m_fAccept(false),
    m_fModifed(false),
    m_VBox(Gtk::Orientation::VERTICAL),
    m_pNewPasswordLabel(nullptr),
    m_pCommentLabel(nullptr),
    m_SortButton("Сортировать"),
    m_CreateButton("Создать"),
    m_DeleteButton("Удалить"),
    m_CancelButton("Отмена"),
    m_OkButton("OK") {
    set_destroy_with_parent(true);

    set_title("Пароли Temic");
    set_default_size(450, 300);

    m_VBox.set_margin(5);
    m_VBox.set_expand();
    set_child(m_VBox);

    m_VBox.append(m_ScrolledWindow);
    m_VBox.append(m_BottomBox);

    m_ScrolledWindow.set_child(m_ColumnView);
    m_ScrolledWindow.set_expand();
    m_ListStore = Gio::ListStore<ModelColumns>::create();

    // Set list model and selection model.
    auto selection_model = Gtk::MultiSelection::create(m_ListStore);
    m_ColumnView.set_model(selection_model);
    m_ColumnView.add_css_class("data-table");  // high density table

    // Add the ColumnView's columns:

    // Столбец "№"
    auto factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(sigc::mem_fun(*this, &CTemicPasswordsDialog::on_setup_label));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CTemicPasswordsDialog::on_bind_idx));
    auto column = Gtk::ColumnViewColumn::create("№", factory);
    m_ColumnView.append_column(column);

    // Столбец "Пароль"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(
        sigc::mem_fun(*this, &CTemicPasswordsDialog::on_setup_password));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CTemicPasswordsDialog::on_bind_password));
    factory->signal_unbind().connect(
        sigc::mem_fun(*this, &CTemicPasswordsDialog::on_unbind_password));
    column = Gtk::ColumnViewColumn::create("Пароль", factory);
    m_ColumnView.append_column(column);

    // Столбец "Комментарий"
    factory = Gtk::SignalListItemFactory::create();
    factory->signal_setup().connect(sigc::mem_fun(*this, &CTemicPasswordsDialog::on_setup_comment));
    factory->signal_bind().connect(sigc::mem_fun(*this, &CTemicPasswordsDialog::on_bind_comment));
    factory->signal_unbind().connect(
        sigc::mem_fun(*this, &CTemicPasswordsDialog::on_unbind_comment));
    column = Gtk::ColumnViewColumn::create("Комментарий", factory);
    m_ColumnView.append_column(column);

    // Load extra CSS file.
    m_refCssProvider = Gtk::CssProvider::create();
#if HAS_STYLE_PROVIDER_ADD_PROVIDER_FOR_DISPLAY
    Gtk::StyleProvider::add_provider_for_display(get_display(), m_refCssProvider,
                                                 GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
#else
    Gtk::StyleContext::add_provider_for_display(get_display(), m_refCssProvider,
                                                GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
#endif
    std::string sCSS =
        ".password {"
        "  font-family:monospace; "
        "}";
    m_refCssProvider->load_from_data(sCSS);

    m_BottomBox.set_margin(5);
    m_BottomBox.set_spacing(5);
    m_BottomBox.set_hexpand();
    m_BottomBox.set_halign(Gtk::Align::END);
    m_BottomBox.append(m_SortButton);
    m_BottomBox.append(m_CreateButton);
    m_BottomBox.append(m_DeleteButton);
    m_BottomBox.append(m_CancelButton);
    m_BottomBox.append(m_OkButton);

    m_SortButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CTemicPasswordsDialog::on_button_sort));
    m_CreateButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CTemicPasswordsDialog::on_button_create));
    m_DeleteButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CTemicPasswordsDialog::on_button_delete));
    m_CancelButton.set_halign(Gtk::Align::END);
    m_OkButton.set_halign(Gtk::Align::END);
    m_CancelButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CTemicPasswordsDialog::on_button_cancel));
    m_OkButton.signal_clicked().connect(sigc::mem_fun(*this, &CTemicPasswordsDialog::on_button_ok));

    set_default_widget(m_OkButton);
    signal_hide().connect(sigc::mem_fun(*this, &CTemicPasswordsDialog::on_hide));
}

CTemicPasswordsDialog::~CTemicPasswordsDialog() {
}

void CTemicPasswordsDialog::Init(ilr::CReader& oReader,
                                 Glib::RefPtr<CTemicPasswordsSettings> refSettings,
                                 const int64_t& nCurrentPassword) {
    m_oReader = std::move(oReader);
    m_refSettings = refSettings;
    m_nCurrentPassword = nCurrentPassword;
    m_fModifed = false;
    if (m_oReader) {
        UpdateView();
        auto pModel = dynamic_cast<Gtk::MultiSelection*>(m_ColumnView.get_model().get());
        auto nIdx = FindTemicPassword(m_refSettings->m_oPasswords, m_nCurrentPassword);
        if (nIdx != -1)
            pModel->select_item((guint)nIdx, true);
    }
}

void CTemicPasswordsDialog::on_setup_label(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    list_item->set_child(*Gtk::make_managed<Gtk::Label>());
}

void CTemicPasswordsDialog::on_setup_password(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto pLabel = Gtk::make_managed<Gtk::EditableLabel>();
    list_item->set_child(*pLabel);
    pLabel->add_css_class("password");
    pLabel->property_editing().signal_changed().connect(
        sigc::bind(sigc::mem_fun(*this, &CTemicPasswordsDialog::on_password_label_editing_change),
                   list_item.get()));
}

void CTemicPasswordsDialog::on_setup_comment(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto pLabel = Gtk::make_managed<Gtk::EditableLabel>();
    list_item->set_child(*pLabel);
    pLabel->set_editable(false);
    pLabel->property_editing().signal_changed().connect(
        sigc::bind(sigc::mem_fun(*this, &CTemicPasswordsDialog::on_comment_label_editing_change),
                   list_item.get()));
}

void CTemicPasswordsDialog::on_bind_idx(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::Label*>(list_item->get_child());
    if (!label)
        return;
    label->set_label(Glib::ustring::format(list_item->get_position()));
}

void CTemicPasswordsDialog::on_bind_password(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::EditableLabel*>(list_item->get_child());
    if (!label)
        return;
    if (col->m_property_password.get_value() == -1)
        m_pNewPasswordLabel = label;
    Glib::Binding::bind_property_value(
        col->m_property_password.get_proxy(), label->property_text(),
        Glib::Binding::Flags::SYNC_CREATE,
        sigc::mem_fun(*this, &CTemicPasswordsDialog::password_transform_to),
        sigc::mem_fun(*this, &CTemicPasswordsDialog::password_transform_from));
}

void CTemicPasswordsDialog::on_unbind_password(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::EditableLabel*>(list_item->get_child());
    if (!label)
        return;
    if (label == m_pNewPasswordLabel)
        m_pNewPasswordLabel = nullptr;
}

bool CTemicPasswordsDialog::password_transform_to(const GValue* a, GValue* b) {
    if (a->g_type != G_TYPE_LONG)
        return false;
    auto nSrc = a->data->v_long;
    if (b->g_type != G_TYPE_STRING)
        return false;
    if (nSrc != -1)
        g_value_set_string(b, TemicPasswordToString(nSrc).c_str());
    else
        g_value_set_string(b, "");
    return true;
}

bool CTemicPasswordsDialog::password_transform_from(const GValue* a, GValue* b) {
    if (a->g_type != G_TYPE_STRING)
        return false;
    if (b->g_type != G_TYPE_LONG)
        return false;
    auto nDest = b->data->v_long;
    auto pValue = g_value_get_string(a);
    int64_t nPassword;
    if (TryParseTemicPassword(pValue, nPassword)) {
        nDest = (glong)nPassword;
        return true;
    }
    return false;
}

void CTemicPasswordsDialog::on_password_label_editing_change(Gtk::ListItem* list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::EditableLabel*>(list_item->get_child());
    if (!label)
        return;
    if (!label->get_editing()) {
        int64_t nNewPassword;
        if (TryParseTemicPassword(label->get_text().c_str(), nNewPassword)) {
            if (col->m_property_password.get_value() != nNewPassword) {
                col->m_property_password = nNewPassword;
                m_fModifed = true;
                if (m_pCommentLabel != nullptr)
                    m_pCommentLabel->set_editable();
                auto nCount = m_ListStore->get_n_items();
                auto item = m_ListStore->get_item(nCount - 1);
                if (item->m_property_password.get_value() != -1) {
                    m_ListStore->append(ModelColumns::create(-1, ""));
                }
            }
        }
    }
}

void CTemicPasswordsDialog::on_bind_comment(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::EditableLabel*>(list_item->get_child());
    if (!label)
        return;
    m_pCommentLabel = label;
    auto fValidPassword = (col->m_property_password.get_value() != -1);
    label->set_editable(fValidPassword);
    Glib::Binding::bind_property(col->m_property_comment.get_proxy(), label->property_text(),
                                 Glib::Binding::Flags::SYNC_CREATE);
}

void CTemicPasswordsDialog::on_unbind_comment(const Glib::RefPtr<Gtk::ListItem>& list_item) {
    m_pCommentLabel = nullptr;
}

void CTemicPasswordsDialog::on_comment_label_editing_change(Gtk::ListItem* list_item) {
    auto col = std::dynamic_pointer_cast<ModelColumns>(list_item->get_item());
    if (!col)
        return;
    auto label = dynamic_cast<Gtk::EditableLabel*>(list_item->get_child());
    if (!label)
        return;
    if (!label->get_editing()) {
        auto text = label->get_text();
        if (col->m_property_comment.get_value() != text) {
            col->m_property_comment.set_value(text);
            m_fModifed = true;
        }
    }
}

void CTemicPasswordsDialog::on_button_sort() {
    m_ListStore->sort(sigc::mem_fun(*this, &CTemicPasswordsDialog::on_sort_compare));
    m_fModifed = true;
}

int CTemicPasswordsDialog::on_sort_compare(const Glib::RefPtr<const ModelColumns>& a,
                                           const Glib::RefPtr<const ModelColumns>& b) {
    auto nCmp = (a->m_property_password.get_value() - b->m_property_password.get_value());
    if (nCmp != 0) {
        if (a->m_property_password.get_value() == -1)
            nCmp = 1;
        else if (b->m_property_password.get_value() == -1)
            nCmp = -1;
        else
            nCmp = (nCmp < 0) ? -1 : 1;
    }
    return (int)nCmp;
}

void CTemicPasswordsDialog::on_button_delete() {
    auto pModel = dynamic_cast<Gtk::MultiSelection*>(m_ColumnView.get_model().get());
    auto nCount = pModel->get_n_items();
    auto pSelection = pModel->get_selection(0, nCount);
    if (pSelection->is_empty())
        return;

    int nOffs = 0;
    for (auto it = pSelection->cbegin(); it != pSelection->cend(); it++) {
        auto nIdx = (*it - nOffs);
        auto item = m_ListStore->get_item(nIdx).get();
        if (item->m_property_password.get_value() == -1)
            continue;
        m_ListStore->remove(nIdx);
        ++nOffs;
    }
    m_fModifed = true;
}

void CTemicPasswordsDialog::on_button_create() {
    if (m_pNewPasswordLabel != nullptr)
        m_pNewPasswordLabel->start_editing();
}

void CTemicPasswordsDialog::on_button_cancel() {
    set_visible(false);
}

void CTemicPasswordsDialog::on_button_ok() {
    SaveViewData();
    auto pModel = std::dynamic_pointer_cast<Gtk::MultiSelection>(m_ColumnView.get_model());
    auto nCount = pModel->get_n_items();
    auto pSelection = pModel->get_selection(0, nCount);
    if (!pSelection->is_empty()) {
        auto item = m_ListStore->get_item(*pSelection->cbegin()).get();
        auto nPassword = item->m_property_password.get_value();
        if (nPassword != -1)
            m_nCurrentPassword = nPassword;
    }
    m_fAccept = true;
    set_visible(false);
}

void CTemicPasswordsDialog::on_hide() {
    m_oReader = nullptr;
    m_refSettings = nullptr;
}

void CTemicPasswordsDialog::UpdateView() {
    m_ListStore->remove_all();

    for (size_t i = 0; i < m_refSettings->m_oPasswords.size(); i++) {
        auto& rInfo = m_refSettings->m_oPasswords[i];
        m_ListStore->append(ModelColumns::create(rInfo.m_nPassword, rInfo.m_sComment));
    }

    m_ListStore->append(ModelColumns::create(-1, ""));
}

void CTemicPasswordsDialog::SaveViewData() {
    if (m_fModifed) {
        CTemicPasswordList oPasswords;
        CTemicPassword rInfo;
        auto nCount = m_ListStore->get_n_items();
        for (size_t i = 0; i < nCount; i++) {
            auto item = m_ListStore->get_item(i);
            auto nPassword = item->m_property_password.get_value();
            if (-1 == nPassword)
                continue;
            rInfo.m_nPassword = (uint)nPassword;
            rInfo.m_sComment = item->m_property_comment.get_value();
            oPasswords.emplace_back(std::move(rInfo));
        }
        m_refSettings->m_oPasswords = std::move(oPasswords);
        m_refSettings->Save(GetTemicPasswordsFilePath());
        m_fModifed = false;
    }
}
