/** @file ilr_cpp_helpers.h
 *  @brief Заголовочный файл SDK Readers с классами-помощниками C++.
 *
 *  Классы-обертки для функций библиотеки из ilreaders.h, и
 *  вспомогательными функциями для разработчиков.
 */
#pragma once

#include <chrono>
#include <exception>
#include <string>
#include <vector>

#include "ilreaders.h"

namespace ilr {

/**
 * @brief Класс исключения SDK Readers.
 * @sa @ref ilr_status
 */
class CILRException : public std::exception {
public:
    /**
     * @brief Конструктор из кода возврата SDK.
     *
     * @param nCode Код возврата SDK.
     */
    CILRException(ilr_status nCode);

    /// Возвращает описание ошибки
    virtual const char* what() const _GLIBCXX_TXN_SAFE_DYN _GLIBCXX_NOTHROW;

public:
    ilr_status m_nCode;  ///< Код возврата SDK
};

/**
 * @brief Бросает исключение если код возврата является ошибкой.
 *
 * Проверяет код возврата SDK и если код соответствует ошибке вызывает исключение CILRException.
 *
 * @param[in]  nCode Код возврата.
 * @exception  CILRException Код возврата не равен `ILR_OK`.
 */
void ILRCheck(ilr_status nCode);

/**
 * @brief Класс ID карты.
 */
class CCardUID : public ilr_card_uid {
public:
    /// Конструктор по умолчанию.
    CCardUID();

    /// Конструктор копирования.
    CCardUID(const ilr_card_uid& rUID);

    /**
     * @brief Конструктор из целого числа.
     * @param[in] number Число.
     * @param[in] nSize  Длина номер в байтах.
     */
    CCardUID(const uint64_t& number, size_t nSize = 8);

    /**
     * @brief Конструктор номера из байтов.
     * @param[in] pData Байты номера.
     * @param[in] nSize Количество байт номера.
     */
    CCardUID(const void* pData, size_t nSize);

    /**
     * @brief Конструктор номера Em-Marine.
     *
     * @param[in] nEmSeries Серия Em-Marine.
     * @param[in] nEmNumber Номер Em-Marine.
     * @param[in] nFacility Код производителя Em-Marine.
     */
    CCardUID(uint8_t nEmSeries, uint16_t nEmNumber, uint16_t nFacility = 0);

    /**
     * @brief Очищает ID карты.
     */
    void Clear() {
        memset(aBytes, 0, sizeof(aBytes));
        nLength = 0;
    }

    /**
     * @brief Проверяет пустой ли ID карты.
     */
    bool IsEmpty() const {
        return (0 == nLength);
    }

    /**
     * @brief Возвращает размер ID карты в байтах.
     */
    int GetLength() const {
        return nLength;
    }

    /**
     * @brief Устанавливает номера из байтов номера.
     *
     * @param[in] pData Байты номера.
     * @param[in] nSize Количество байт номера.
     */
    void Assign(const void* pData, size_t nSize) {
        nLength = static_cast<uint8_t>((nSize > sizeof(aBytes)) ? sizeof(aBytes) : nSize);
        if (nLength != 0)
            memcpy(aBytes, pData, nLength);
    }

    /**
     * @brief Оператор копирования из 4-байтного числа.
     *
     * @param[in] other Другой номер.
     */
    /// Оператор копирования
    const CCardUID& operator=(const uint32_t& other) {
        Assign(&other, sizeof(other));
        return *this;
    }

    /**
     * @brief Оператор копирования из 8-байтного числа.
     *
     * @param[in] other Другой номер.
     */
    const CCardUID& operator=(const uint64_t& other) {
        Assign(&other, sizeof(other));
        return *this;
    }

    /**
     * @brief Сравнивает ID номеров.
     *
     * @param[in] other Другой номер, с которым сравниваем.
     */
    int Compare(const ilr_card_uid& other) const {
        int n = (nLength - other.nLength);
        if (0 == n)
            n = memcmp(aBytes, other.aBytes, nLength);
        return n;
    }

    /**
     * @brief Оператор сравнения на равенство.
     *
     * @param[in] other Другой номер.
     *
     * @return True, номера равны.
     */
    bool operator==(const ilr_card_uid& other) const {
        return (nLength == other.nLength) && (memcmp(aBytes, other.aBytes, nLength) == 0);
    }

    /**
     * @brief Оператор сравнения на не равенство.
     *
     * @param[in] other Другой номер.
     *
     * @return True, номера не равны.
     */
    bool operator!=(const ilr_card_uid& other) const {
        return (nLength != other.nLength) || (memcmp(aBytes, other.aBytes, nLength) != 0);
    }

    /**
     * @brief Оператор сравнения "больше".
     *
     * @param[in] other Другой номер.
     *
     * @return True, этот номер больше другого номера.
     */
    bool operator>(const ilr_card_uid& other) const {
        return Compare(other) > 0;
    }

    /**
     * @brief Оператор сравнения "больше или равно".
     *
     * @param[in] other Другой номер.
     *
     * @return True, этот номер больше другого номера, или они равны.
     */
    bool operator>=(const ilr_card_uid& other) const {
        return Compare(other) >= 0;
    }

    /**
     * @brief Оператор сравнения "меньше".
     *
     * @param[in] other Другой номер.
     *
     * @return True, этот номер меньше другого номера.
     */
    bool operator<(const ilr_card_uid& other) const {
        return Compare(other) < 0;
    }

    /**
     * @brief Оператор сравнения "меньше или равно".
     *
     * @param[in] other Другой номер.
     *
     * @return True, этот номер меньше другого номера, или они равны.
     */
    bool operator<=(const ilr_card_uid& other) const {
        return Compare(other) <= 0;
    }

    /**
     * @brief Преобразует номер карты в строку.
     */
    std::string ToString() const;

    /**
     * @brief Преобразует строку в номер карты.
     */
    bool TryParse(const char* pStr);
};

/**
 * @brief Класс ключа аутентификации Mifare Classic.
 *
 * Это класс-обертка для ilr_mf_classic_key.
 */
class CMifareClassicKey {
public:
    /// Конструктор по умолчанию.
    CMifareClassicKey();

    /**
     * @brief Конструктор из данных.
     *
     * @param[in] pData Ссылка на данные.
     * @param[in] nSize Размер данных. Если больше 6, то используется 6.
     */
    CMifareClassicKey(const void* pData, size_t nSize);

    /**
     * @brief Конструктор из ilr_mf_classic_key.
     *
     * @param[in] other Значение ilr_mf_classic_key.
     */
    CMifareClassicKey(const ilr_mf_classic_key& other);

    /// Устанавливает значение ключа по умолчанию.
    inline void Reset() {
        (ilr_mf_classic_key&)(*this) = kDefault;
    }

    /// Оператор приведения типа к ilr_mf_classic_key.
    operator ilr_mf_classic_key() const {
        return m_nKey;
    }

    /**
     * @brief Оператор копирования ilr_mf_classic_key.
     *
     * @param[in] other Другое значение ilr_mf_classic_key.
     */
    const CMifareClassicKey& operator=(const ilr_mf_classic_key& other) {
        (ilr_mf_classic_key&)(*this) = other;
        return *this;
    }

    /**
     * @brief Преобразует номер ключа в строку.
     */
    std::string ToString() const;

    /**
     * @brief Преобразует строку в номер ключа.
     */
    bool TryParse(const char* pStr);

public:
    /** Значение по умолчанию для ключа аутентификации Mifare Classic. */
    static const ilr_mf_classic_key kDefault;

public:
    ilr_mf_classic_key m_nKey;  ///< Ключ аутентификации Mifare Classic.
};

/**
 * @brief Ключ аутентификации Mifare Plus.
 *
 * Это класс-обертка для ilr_mf_plus_key.
 */
class CMifarePlusKey : public ilr_mf_plus_key {
public:
    /// Конструктор по умолчанию
    CMifarePlusKey();

    /**
     * @brief Конструктор из ilr_mf_plus_key.
     *
     * @param[in] other Значение ilr_mf_plus_key.
     */
    CMifarePlusKey(const ilr_mf_plus_key& other);

    /**
     * @brief Конструктор из двух 8-байтных чисел.
     *
     * @param[in] lo Число, которое копируется в младшие 8 байт ключа Mifare Plus.
     * @param[in] hi Число, которое копируется в старшие 8 байт ключа Mifare Plus.
     */
    CMifarePlusKey(const uint64_t& lo, const uint64_t& hi);

    /**
     * @brief Конструктор из ключа Mifare Classic.
     *
     * @param[in] other Значение ключа Mifare Classic.
     */
    CMifarePlusKey(const ilr_mf_classic_key& other);

    /// Устанавливает значение ключа по умолчанию
    inline void Reset() {
        memset(a, 0xff, sizeof(a));
    }

    /// Оператор приведения типа к ilr_mf_classic_key.
    operator ilr_mf_classic_key() const {
        return ll.lo & 0xffffffffffff;
    }

    /**
     * @brief Оператор присвоения ключа Mifare Plus.
     *
     * @param[in] other Значение ключа.
     */
    const CMifarePlusKey& operator=(const ilr_mf_plus_key& other) {
        (ilr_mf_plus_key&)(*this) = other;
        return *this;
    }

    /**
     * @brief Оператор присвоения ключа Mifare Classic.
     *
     * @param[in] other Значение ключа.
     */
    const CMifarePlusKey& operator=(const ilr_mf_classic_key& other) {
        ll.lo = other & 0xffffffffffff;
        ll.hi = 0;
        return *this;
    }

    /**
     * @brief Оператор сравнение ключей на равенство.
     *
     * @param[in] other Другой ключ.
     * @return True ключи равны.
     */
    bool operator==(const CMifarePlusKey& other) const {
        return memcmp(a, other.a, sizeof(a)) == 0;
    }

    /**
     * @brief Оператор сравнение ключей на неравенство.
     *
     * @param[in] other Другой ключ.
     * @return True ключи не равны.
     */
    bool operator!=(const CMifarePlusKey& other) const {
        return memcmp(a, other.a, sizeof(a)) != 0;
    }

    /**
     * @brief Оператор сравнение ключей "меньше".
     *
     * @param[in] other Другой ключ.
     * @return True этот ключ меньше, чем другой.
     */
    bool operator<(const CMifarePlusKey& other) const {
        return memcmp(a, other.a, sizeof(a)) < 0;
    }

    /**
     * @brief Оператор сравнение ключей "больше".
     *
     * @param[in] other Другой ключ.
     * @return True этот ключ больше, чем другой.
     */
    bool operator>(const CMifarePlusKey& other) const {
        return memcmp(a, other.a, sizeof(a)) > 0;
    }

    /**
     * @brief Оператор сравнение ключей "меньше или равно".
     *
     * @param[in] other Другой ключ.
     * @return True этот ключ меньше, чем другой, или равен ему.
     */
    bool operator<=(const CMifarePlusKey& other) const {
        return memcmp(a, other.a, sizeof(a)) <= 0;
    }

    /**
     * @brief Оператор сравнение ключей "больше или равно".
     *
     * @param[in] other Другой ключ.
     * @return True этот ключ больше, чем другой, или равен ему.
     */
    bool operator>=(const CMifarePlusKey& other) const {
        return memcmp(a, other.a, sizeof(a)) >= 0;
    }

    int Compare(const CMifarePlusKey& other) const;

    /**
     * @brief Преобразует номер ключа в строку.
     */
    std::string ToString(const char* pPartDelimiter = nullptr) const;

    /**
     * @brief Преобразует строку в номер ключа.
     */
    bool TryParse(const char* pStr);
};

/**
 * @brief Класс дескриптора SDK.
 *
 * Это класс-обертка для дескриптора SDK (ilr_handle).
 */
class CILRHandle {
    friend class CILR;
    friend class CReaderSearch;
    friend class CReader;

public:
    /// Конструктор по умолчанию.
    CILRHandle();

    /**
     * @brief Конструктор из дескриптора SDK.
     *
     * @param[in] h Значение дескриптора.
     */
    CILRHandle(ilr_handle h);

    /// Запрещаем копирование этого класса (т.к. он содержит указатель = дескриптор).
    CILRHandle(const CILRHandle&) = delete;

    /**
     * @brief Конструктор перемещения.
     *
     * @param[in out] other Другой дескриптор.
     */
    CILRHandle(CILRHandle&& other);

    /// Деструктор
    virtual ~CILRHandle();

    /// Запрещаем копирование этого класса (т.к. он содержит указатель = дескриптор).
    CILRHandle& operator=(const CILRHandle&) = delete;

    /**
     * @brief Оператор перемещения.
     *
     * @param[in out] other Другой дескриптор.
     */
    CILRHandle& operator=(CILRHandle&& other);

    /// Оператор приведения типа к ilr_handle.
    operator ilr_handle() const;

    /// Проверяет дескриптор на валидность.
    explicit operator bool() const;

    /**
     * @brief Обменивается значением с другим дескриптором.
     *
     * @param[in] other Другой дескриптор.
     */
    void Swap(CILRHandle& other) noexcept;

    /// Возвращает дескриптор SDK.
    ilr_handle Get() const;

    /**
     * @brief Закрывает дескриптор SDK.
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    void Close();

    /**
     * @brief Присоединяет дескриптор к этому объекту.
     *
     * @param[in] h Значение дескриптора SDK.
     */
    void Attach(ilr_handle h);

    /**
     * @brief Отсоединяет дескриптор от этого объекту.
     *
     * @return Значение дескриптора SDK.
     */
    ilr_handle Detach();

protected:
    ilr_handle m_h;  ///< Дескриптор SDK
};

/**
 * @brief Класс команды SDK.
 *
 * Класс для управления асинхронными запросом.
 */
class CAsyncCommand : public CILRHandle {
public:
    /// Конструктор по умолчанию.
    CAsyncCommand();

    /**
     * @brief Конструктор класса из дескриптора команды.
     *
     * @param[in] h Значение дескриптора SDK.
     */
    CAsyncCommand(ilr_handle h);

    /**
     * @brief Конструктор перемещения.
     *
     * @param[in out] other Другой дескриптор команды.
     */
    CAsyncCommand(CAsyncCommand&& other);

    /// Деструктор
    virtual ~CAsyncCommand();

    /// Оператор перемещения
    CAsyncCommand& operator=(CAsyncCommand&& other);

    /**
     * @brief Отменяет команду. Устанавливает статус `ILR_E_ABORT`.
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline void Cancel();

    /**
     * @brief Возвращает состояние команды.
     *
     * @return Состояние команды: =`ILR_E_PENDING` команда ещё выполняется, иначе - завершена.
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline ilr_status GetStatus() const;

    /**
     * @brief Возвращает состояние прогресса выполнения команды.
     *
     * @param[out] nCurrent  Текущий шаг.
     * @param[out] nTotal    Всего шагов.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline void GetProgress(size_t& nCurrent, size_t& nTotal) const;
};

inline void CAsyncCommand::Cancel() {
    ILRCheck(ilr_command_cancel(m_h));
}

inline ilr_status CAsyncCommand::GetStatus() const {
    ilr_status res;
    ILRCheck(ilr_command_get_status(m_h, &res));
    return res;
}

inline void CAsyncCommand::GetProgress(size_t& nCurrent, size_t& nTotal) const {
    ILRCheck(ilr_command_get_progress(m_h, &nCurrent, &nTotal));
}

/**
 * @brief Класс поиска считывателей.
 *
 * Класс-обёртка для дескриптора поиска считывателей.
 */
class CReaderSearch : public CILRHandle {
public:
    /// Конструктор по умолчанию.
    CReaderSearch();

    /**
     * @brief Конструктор класса из дескриптора поиска считывателей.
     *
     * @param[in] h Значение дескриптора SDK.
     */
    CReaderSearch(ilr_handle h);

    /**
     * @brief Конструктор перемещения.
     *
     * @param[in out] other Другой поиск считывателей.
     */
    CReaderSearch(CReaderSearch&& other);

    /// Деструктор
    virtual ~CReaderSearch();

    /// Оператор перемещения
    CReaderSearch& operator=(CReaderSearch&& other);

    /**
     * @brief Устанавливает функцию обратного вызова для уведомлений поиска считывателей.
     *
     * Устанавливает функцию для получения сообщений от дескриптора поиска считывателей.
     *
     * @param[in] pCallback Указатель на функцию, которую библиотека будет вызывать при
     * возникновении события поиска считывателей.
     * @param[in] pUserData Указатель на данные пользователя, который будет передаваться в функцию.
     *
     * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в `try catch`.
     * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
     * из которого вызвана эта callback-функция, иначе будет ошибка
     * ``ILR_E_BLOCKING_CALL_NOT_ALLOWED``.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline void SetMessageCallback(ilr_search_message_callback pCallback,
                                   void* pUserData = nullptr);

    /**
     * @brief Включает/выключает очередь сообщений.
     *
     * Эта функция устанавливает/снимает флаг в дескрипторе поиска.
     * Очередь сообщений предназначена для синхронизации обработки сообщений.
     *
     * @remark Алгоритм синхронизации: при возникновении события в очередь добавляется сообщение и
     * вызывается функция обратного вызова, установленная функцией @ref SetMessageCallback, из
     * которой посылается сигнал потоку, обрабатывающему сообщения, этот поток при получении сигнала
     * циклично вызывает @ref GetMessage, чтобы получить и обработать все сообщения.
     *
     * @param[in] fEnable   =`true`, включает очередь, иначе - выключает.
     *
     * @warning Если не извлекать сообщения из очереди функцией @ref GetMessage, то
     * она будет расти пока не закончится память.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline void EnableMessageQueue(bool fEnable = true);

    /**
     * @brief Извлекает следующее сообщение из очереди.
     *
     * @param[out] nMsg     Тип сообщения.
     * @param[out] pMsgData Указатель на данные пользователя, которые были установлены функцией
     * @ref SetMessageCallback.
     *
     * @return true если сообщение успешно извлечено, иначе - очередь пуста.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline bool GetMessage(ilr_search_msg& nMsg, const void*& pMsgData) const;

    /**
     * @brief Устанавливает параметры поиска считывателей.
     *
     * @remark Параметры поиска считывателей общие для всех дескрипторов.
     * Параметры не теряются при закрытии всех дескрипторов поиска.
     *
     * @param[in] rOptions  Опции поиска считывателей.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline void SetOptions(const ilr_search_options& rOptions);

    /**
     * @brief Возвращает параметры поиска считывателей.
     *
     * @param[out] rOptions  Опции поиска считывателей.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline void GetOptions(ilr_search_options& rOptions) const;

    /**
     * @brief Устанавливает список портов для прослушки конвертеров к режиме "Клиент".
     *
     * Устанавливает список TCP-портов, к которым будут подключаться IP-конвертеры
     * в режиме "Клиент".
     *
     * @remark Список портов общий для всех дескрипторов.
     *
     * @param[in] pPorts  Массив портов. Если =`nullptr`, то очищает список.
     * @param[in] nCount  Количество элементов массива портов.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`, `ILR_E_INVALIDARG`,
     * `ILR_E_OUTOFMEMORY`.
     */
    inline void SetListenPorts(const uint16_t* pPorts, size_t nCount);

    /**
     * @brief Возвращает список портов для прослушки конвертеров к режиме "Клиент".
     *
     * Возвращает список TCP-портов, который был установлен функцией @ref SetListenPorts.
     *
     * @remark Список портов общий для всех дескрипторов.
     *
     * @param[in out] oPorts  Список портов.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`, `ILR_E_OUTOFMEMORY`.
     */
    void GetListenPorts(std::vector<uint16_t>& oPorts) const;

    /**
     * @brief Возвращает состояние Tcp-порта, открытого для прослушки конвертеров в режиме "Клиент".
     *
     * @param[in]  nTcpPort  Номер TCP порта.
     *
     * @remark Чтобы открыть порт для прослушки нужно добавить его в список портов с помощью функции
     * @ref ilr_search_set_listen_ports, и включить поиск конвертеров в режиме "Клиент" с помощью
     * @ref ilr_search_set_options (установить pOptions->nReaderTypes |= ILR_READER_CLIENT).
     *
     * @return Состояние порта: `ILR_OK` порт открыт успешно, `ILR_E_PENDING` порт в процессе
     * открытия или его нет в списке портов для прослушки, иначе не удалось открыть.
     */
    inline ilr_status GetListenStatus(uint16_t nTcpPort) const;

    /**
     * @brief Ищет считыватели.
     *
     * @param[in] fReset  =`true` очищает список найденных считывателей перед началом поиска.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
     * считывателей.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUT_OF_RESOURCES`, `ILR_E_OUTOFMEMORY`.
     */
    inline void Scan(bool fReset = false);

    /**
     * @brief Запускает асинхронную команду поиска считывателей.
     *
     * @param[in]  fReset =`true` очистить список найденных перед поиском.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_OUT_OF_RESOURCES`, `ILR_E_OUTOFMEMORY`.
     */
    inline CAsyncCommand Begin_Scan(bool fReset = false);

    /**
     * @brief Возвращает количество найденных считывателей.
     *
     * @warning Эта функция копирует список из потока поиска, поэтому её лучше не вызывать в цикле.
     *
     * @return Количество найденных считывателей.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`, `ILR_E_OUTOFMEMORY`.
     */
    inline size_t GetReaderCount() const;

    /**
     * @brief Возвращает информацию о найденном считывателе.
     *
     * @param[in]  nIdx    Позиция в списке найденных считывателей.
     * @param[out] rInfo   Информация о считывателе.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`, `ILR_E_BOUNDS`.
     */
    inline void GetReaderInfo(size_t nIdx, ilr_reader_info& rInfo) const;

    /**
     * @brief Включает/выключает авто поиск считывателей.
     *
     * @param[in]  fEnable =`true`, включает авто поиск, иначе - выключает.
     * @param[in]  fWait   =`true`, ждёт полного завершения команды, иначе только устанавливает
     * флаг.
     *
     * @remark Если @p fWait =`true`, то функция не возвращает управление пока ждёт выполнение
     * команды в потоке поиска считывателей.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline void SetAutoScan(bool fEnable = true, bool fWait = true);

    /**
     * @brief Запускает асинхронную команду вкл/выкл режим авто поиска считывателей.
     *
     * @param[in]  fEnable =`true` включает поиск в реальном времени, иначе - выключает.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`, `ILR_E_OUTOFMEMORY`,
     * `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_SetAutoScan(bool fEnable);

    /**
     * @brief Возвращает флаг авто поиска считывателей.
     *
     * @return true, авто поиск включен, иначе - выключен.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline bool GetAutoScan() const;

    /**
     * @brief Открывает порт и возвращает его дескриптор.
     *
     * @param[in]  nPortType   Тип порта.
     * @param[in]  pszPortName Имя порта.
     * @param[out] pInfo       Информация о считывателе. Может быть равно `nullptr`.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
     * считывателей.
     *
     * @return Системный дескриптор порта.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_PORT_ACCESS_DENIED`, `ILR_E_PORT_NOT_EXIST`, `ILR_E_PORT_OPEN_FAILED`,
     * `ILR_E_CONNECTION_ERROR`, `ILR_E_NOTIMPL`.
     */
    inline int OpenPort(ilr_port_type nPortType, const char* pszPortName,
                        ilr_reader_info* pInfo = nullptr);

    /**
     * @brief Запускает асинхронную команду открытия порта.
     *
     * @param[in]  nPortType   Тип порта.
     * @param[in]  pszPortName Имя порта.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_OpenPort(ilr_port_type nPortType, const char* pszPortName);

    /**
     * @brief Возвращает результат открытия порта.
     *
     * @param[in]  hCommand Дескриптор команды.
     * @param[out] PortFD   Системный дескриптор порта.
     * @param[out] rInfo    Информация о конвертере (если известно).
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    static inline void End_OpenPort(ilr_handle hCommand, int& PortFD, ilr_reader_info& rInfo);

    /**
     * @brief Закрывает порт
     *
     * @param[in]  nPortType   Тип порта.
     * @param[in]  pszPortName Имя порта.
     * @param[in]  hPort       Системный дескриптор порта.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
     * считывателей.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline void ClosePort(ilr_port_type nPortType, const char* pszPortName, int hPort);

    /**
     * @brief Запускает асинхронную команду закрытия порта.
     *
     * @param[in]  nPortType   Тип порта
     * @param[in]  pszPortName Имя порта
     * @param[in]  hPortFD     Системный дескриптор порта.
     *
     * @return Класс дескриптора команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`, `ILR_E_OUTOFMEMORY`,
     * `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_ClosePort(ilr_port_type nPortType, const char* pszPortName,
                                         int hPortFD);

    /**
     * @brief Клонирует дескриптор поиска считывателей.
     * @return Класс дескриптора поиска считывателей.
     */
    inline CReaderSearch Clone() const;
};

inline void CReaderSearch::SetMessageCallback(ilr_search_message_callback pCallback,
                                              void* pUserData) {
    ILRCheck(ilr_search_set_message_callback(m_h, pCallback, pUserData));
}

inline void CReaderSearch::EnableMessageQueue(bool fEnable) {
    ILRCheck(ilr_search_enable_message_queue(m_h, (ilr_bool)fEnable));
}

inline bool CReaderSearch::GetMessage(ilr_search_msg& nMsg, const void*& pMsgData) const {
    ilr_bool f;
    ILRCheck(ilr_search_get_message(m_h, &nMsg, &pMsgData, &f));
    return f;
}

inline void CReaderSearch::SetOptions(const ilr_search_options& rOptions) {
    ILRCheck(ilr_search_set_options(m_h, &rOptions));
}

inline void CReaderSearch::GetOptions(ilr_search_options& rOptions) const {
    ILRCheck(ilr_search_get_options(m_h, &rOptions));
}

inline void CReaderSearch::SetListenPorts(const uint16_t* pPorts, size_t nCount) {
    ILRCheck(ilr_search_set_listen_ports(m_h, pPorts, nCount));
}

ilr_status CReaderSearch::GetListenStatus(uint16_t nTcpPort) const {
    ilr_status res;
    ILRCheck(ilr_search_get_listen_status(m_h, nTcpPort, &res));
    return res;
}

inline void CReaderSearch::Scan(bool fReset) {
    ILRCheck(ilr_search_scan(m_h, (ilr_bool)fReset));
}

inline size_t CReaderSearch::GetReaderCount() const {
    size_t res;
    ILRCheck(ilr_search_get_reader_count(m_h, &res));
    return res;
}

inline void CReaderSearch::GetReaderInfo(size_t nIdx, ilr_reader_info& rInfo) const {
    ILRCheck(ilr_search_get_reader_info(m_h, nIdx, &rInfo));
}

inline void CReaderSearch::SetAutoScan(bool fEnable, bool fWait) {
    ILRCheck(ilr_search_set_auto_scan(m_h, (ilr_bool)fEnable, (ilr_bool)fWait));
}

inline bool CReaderSearch::GetAutoScan() const {
    ilr_bool f;
    ILRCheck(ilr_search_get_auto_scan(m_h, &f));
    return static_cast<bool>(f);
}

inline int CReaderSearch::OpenPort(ilr_port_type nPortType, const char* pszPortName,
                                   ilr_reader_info* pInfo) {
    int fd;
    ILRCheck(ilr_search_open_port(m_h, nPortType, pszPortName, pInfo, &fd));
    return fd;
}

inline void CReaderSearch::ClosePort(ilr_port_type nPortType, const char* pszPortName, int hPort) {
    ILRCheck(ilr_search_close_port(m_h, nPortType, pszPortName, hPort));
}

inline CAsyncCommand CReaderSearch::Begin_Scan(bool fReset) {
    CAsyncCommand res;
    ILRCheck(ilr_search_begin_scan(m_h, (ilr_bool)fReset, &res.m_h));
    return res;
}

inline CAsyncCommand CReaderSearch::Begin_SetAutoScan(bool fEnable) {
    CAsyncCommand res;
    ILRCheck(ilr_search_begin_set_auto_scan(m_h, (ilr_bool)fEnable, &res.m_h));
    return res;
}

inline CAsyncCommand CReaderSearch::Begin_OpenPort(ilr_port_type nPortType,
                                                   const char* pszPortName) {
    CAsyncCommand res;
    ILRCheck(ilr_search_begin_open_port(m_h, nPortType, pszPortName, &res.m_h));
    return res;
}

inline void CReaderSearch::End_OpenPort(ilr_handle hCommand, int& PortFD, ilr_reader_info& rInfo) {
    ILRCheck(ilr_search_end_open_port(hCommand, &PortFD, &rInfo));
}

inline CAsyncCommand CReaderSearch::Begin_ClosePort(ilr_port_type nPortType,
                                                    const char* pszPortName, int hPortFD) {
    CAsyncCommand res;
    ILRCheck(ilr_search_begin_close_port(m_h, nPortType, pszPortName, hPortFD, &res.m_h));
    return res;
}

inline CReaderSearch CReaderSearch::Clone() const {
    ilr_handle hNewHandle;
    ILRCheck(ilr_clone_handle(m_h, &hNewHandle));
    return CReaderSearch(hNewHandle);
}

/**
 * @brief Класс считывателя.
 *
 * Класс-обёртка для дескриптора считывателя.
 */
class CReader : public CILRHandle {
public:
    /// Конструктор по умолчанию.
    CReader();

    /**
     * @brief Конструктор класса из дескриптора считывателя.
     *
     * @param[in] h Значение дескриптора SDK.
     */
    CReader(ilr_handle h);

    /**
     * @brief Конструктор перемещения.
     *
     * @param[in out] other Другой считыватель.
     */
    CReader(CReader&& other);

    /// Деструктор
    virtual ~CReader();

    /// Оператор перемещения
    CReader& operator=(CReader&& other);

    /**
     * @brief Устанавливает функцию обратного вызова для уведомлений считывателя.
     *
     * Устанавливает функцию для получения сообщений от дескриптора считывателя.
     *
     * @param[in] pCallback Указатель на функцию, которую библиотека будет вызывать при
     * возникновении события считывателя.
     * @param[in] pUserData Указатель на данные пользователя, который будет передаваться в функцию.
     *
     * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в try catch.
     * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
     * из которого вызвана эта callback-функция, иначе вернёт ошибку
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline void SetMessageCallback(ilr_reader_message_callback pCallback,
                                   void* pUserData = nullptr);

    /**
     * @brief Включает/выключает очередь сообщений.
     *
     * Эта функция устанавливает/снимает флаг в дескрипторе считывателя.
     * Очередь сообщений предназначена для синхронизации обработки сообщений.
     *
     * @remark Алгоритм синхронизации: при возникновении события в очередь добавляется сообщение и
     * вызывается функция обратного вызова, установленная функцией SetMessageCallback,
     * из которой посылается сигнал потоку, обрабатывающему сообщения, этот поток при получении
     * сигнала циклично вызывает GetMessage, чтобы получить и обработать все сообщения.
     *
     * @param[in]  fEnable  =`true`, включает очередь, иначе - выключает.
     *
     * @warning Если не извлекать сообщения из очереди функцией GetMessage, то
     * она будет расти пока не закончится память.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline void EnableMessageQueue(bool fEnable = true);

    /**
     * @brief Извлекает следующее сообщение из очереди.
     *
     * @param[out] nMsg     Тип сообщения.
     * @param[out] pMsgData Указатель на данные пользователя, которые были установлены функцией
     * SetMessageCallback.
     *
     * @return true если сообщение успешно извлечено, иначе - очередь пуста.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline bool GetMessage(ilr_reader_msg& nMsg, const void*& pMsgData) const;

    /**
     * @brief Устанавливает параметры считывателя.
     *
     * @param[in]  rOptions Параметры считывателя.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline void SetOptions(const ilr_reader_options& rOptions);

    /**
     * @brief Возвращает параметры считывателя.
     *
     * @param[out]  rOptions Параметры считывателя.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline void GetOptions(ilr_reader_options& rOptions) const;

    /**
     * @brief Подключается к считывателю.
     *
     * @param[in]  fReconnect =`true` Отключается перед подключением.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_PORT_ACCESS_DENIED`, `ILR_E_PORT_NOT_EXIST`, `ILR_E_PORT_OPEN_FAILED`,
     * `ILR_E_CONNECTION_ERROR`.
     */
    inline void Connect(bool fReconnect = false);

    /**
     * @brief Запускает асинхронную команду подключения к считывателю.
     *
     * @param[in]  fReconnect =`true` переподключиться.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`, `ILR_E_OUTOFMEMORY`,
     * `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_Connect(bool fReconnect = false);

    /**
     * @brief Отключается от считывателя.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline void Disconnect();

    /**
     * @brief Запускает асинхронную команду отключения от считывателя.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`, `ILR_E_OUTOFMEMORY`,
     * `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_Disconnect();

    /**
     * @brief Возвращает состояние подключения к считывателю
     *
     * @return Состояние подключения к считывателю.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline ilr_connection_status GetConnectionStatus() const;

    /**
     * @brief Возвращает информацию о считывателе.
     *
     * @param[out] rInfo    Информация о считывателе.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline void GetReaderInfo(ilr_reader_info& rInfo) const;

    /**
     * @brief Ищет карты.
     *
     * @param[in]  fReset    =`true` сбросить старые результаты поиска
     * @param[in]  fPowerOff =`true` выключает RF поле после сканирования
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES` и другие.
     */
    inline void Scan(bool fReset = false, bool fPowerOff = true);

    /**
     * @brief Запускает асинхронную команду поиска карты.
     *
     * @param[in]  fReset      =`true` сбросить старые результаты поиска.
     * @param[in]  fPowerOff   =`true` выключает RF поле после сканирования.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`, `ILR_E_OUTOFMEMORY`,
     * `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_Scan(bool fReset = false, bool fPowerOff = true);

    /**
     * @brief Возвращает информацию о карте в поле считывателя.
     *
     * @param[out] rInfo    Информация о карте.
     *
     * @return True, то карта найдена, иначе - нет карты.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline bool GetCardInfo(ilr_card_info& rInfo) const;

    /**
     * @brief Включает/выключает автоматическое сканирование карт.
     *
     * @param[in]  fEnable  =`true` включить авто сканирование, иначе - выключить.
     * @param[in]  fWait    =`true` ждать полного завершения команды, иначе только установить флаг.
     *
     * @remark Если fWait = true, то функция не возвращает управление пока ждёт
     * выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline void SetAutoScan(bool fEnable = true, bool fWait = true);

    /**
     * @brief Запускает асинхронную команду вкл/выкл автоматического сканирования карт.
     *
     * @param[in]  fEnable     =`true` включает авто сканирование, иначе - выключает.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`, `ILR_E_OUTOFMEMORY`,
     * `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_SetAutoScan(bool fEnable = true);

    /**
     * @brief Возвращает флаг автоматическое сканирование карт.
     *
     * @return true если авто сканирование включено.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline bool GetAutoScan() const;

    /**
     * @brief Читает данные карты Mifare Ultralight.
     *
     * @param[in]  nPageIdx   Номер первой читаемой страницы (0..15).
     * @param[out] pBuf       Буфер для прочитанных страниц.
     * @param[in]  nPageCount Количество страниц, которые нужно прочитать.
     * @param[out] pRead      Количество прочитанных страниц. Может быть = `nullptr`.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_READER_ERROR`, `ILR_E_BAD_RESPONSE`,
     * `ILR_E_REQUEST_TIMEOUT`, `ILR_E_SCARD_ERROR`, `ILR_E_FAIL`, `ILR_E_ABORT`.
     */
    inline void ReadMfUltralight(size_t nPageIdx, uint32_t* pBuf, size_t nPageCount,
                                 size_t* pRead = nullptr);

    /**
     * @brief Запускает асинхронную команду чтения данных из карты Mifare Ultralight.
     *
     * @param[in]  nPageIdx   Номер первой читаемой страницы (начиная от 0).
     * @param[in]  nPageCount Количество читаемых страниц.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`, `ILR_E_INVALIDARG`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_ReadMfUltralight(size_t nPageIdx, size_t nPageCount);

    /**
     * @brief Возвращает результат чтения данных из карты Mifare Ultralight.
     *
     * @param[in]  hCommand  Дескриптор команды.
     * @param[out] pList     Ссылка на список прочитанных страниц. Ссылка действительна до закрытия
     * дескриптора команды @p hCommand.
     * @param[out] nRead     Количество прочитанных страниц.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_POINTER`,
     * `ILR_E_HANDLE`.
     */
    static inline void End_ReadMfUltralight(ilr_handle hCommand, const uint32_t*& pList,
                                            size_t& nRead);

    /**
     * @brief Пишет данные карты Mifare Ultralight.
     *
     * @param[in]  nPageIdx   Номер первой записываемой страницы (0..15).
     * @param[in]  pData      Данные страниц.
     * @param[in]  nPageCount Количество страниц, которые нужно записать.
     * @param[out] pWritten   Количество записанных страниц. Может быть = `nullptr`.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_CARD_PAGE_LOCK`, `ILR_E_READER_ERROR`,
     * `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`, `ILR_E_SCARD_ERROR`, `ILR_E_FAIL`,
     * `ILR_E_ABORT`.
     */
    inline void WriteMfUltralight(size_t nPageIdx, const uint32_t* pData, size_t nPageCount,
                                  size_t* pWritten = nullptr);

    /**
     * @brief Запускает асинхронную команду записи данных в карту Mifare Ultralight.
     *
     * @param[in]  nPageIdx   Номер первой записываемой страницы (начиная от 0).
     * @param[in]  pData      Данные страниц.
     * @param[in]  nPageCount Количество записываемых страниц.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_WriteMfUltralight(size_t nPageIdx, const uint32_t* pData,
                                                 size_t nPageCount);

    /**
     * @brief Возвращает результат записи данных в карту Mifare Ultralight.
     *
     * @param[in]  hCommand  Дескриптор команды.
     * @param[out] nWritten  Количество прочитанных страниц
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    static inline void End_WriteMfUltralight(ilr_handle hCommand, size_t& nWritten);

    /**
     * @brief Загружает ключ для авторизации сектора Mifare Classic / Plus SL1.
     *
     * @param[in]  nKey  Ключ аутентификации Mifare Classic.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`, `ILR_E_INVALIDARG`.
     */
    inline void LoadMfCKey(const ilr_mf_classic_key& nKey);

    /**
     * @brief Загружает ключ для авторизации сектора Mifare Plus SL3.
     *
     * @param[in]  rKey   Ключ аутентификации Mifare Plus.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline void LoadMfPKey(const ilr_mf_plus_key& rKey);

    /**
     * @brief Авторизует сектор карты Mifare Classic/Plus.
     *
     * Авторизует сектор карты Mifare Classic/Plus, используя ключ, загруженный
     * функцией LoadMfCKey / LoadMfPKey.
     *
     * @param[in]  nAddress Номер блока (0..255) или адрес Mifare Plus.
     * @param[in]  fKeyB    =`true` авторизовать по ключу Б, иначе - по ключу А.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @return true сектор успешно авторизован.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_MIFARE_ADDRESS`, `ILR_E_READER_ERROR`,
     * `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`, `ILR_E_SCARD_ERROR`, `ILR_E_FAIL`,
     * `ILR_E_ABORT`.
     */
    inline bool AuthMfCard(size_t nAddress, bool fKeyB);

    /**
     * @brief Запускает асинхронную команду авторизации сектора карты Mifare Classic / Plus
     * используя ключ, загруженный функцией LoadMfAuthKey / LoadMfPlusAuthKey.
     *
     * @param[in]  nAddress Номер блока или адрес Mifare Plus.
     * @param[in]  fKeyB    =`true` авторизовать по ключу B, иначе A.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_AuthMfCard(size_t nAddress, bool fKeyB);

    /**
     * @brief Возвращает результат авторизации сектора карты.
     *
     * @param[in]  hCommand Дескриптор команды.
     * @param[out] fAuthOk  =`true`  авторизация прошла успешно.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    static inline void End_AuthMfCard(ilr_handle hCommand, bool& fAuthOk);

    /**
     * @brief Авторизует сектор карты Mifare Classic / Plus, используя ключи считывателя.
     *
     * @param[in]  nAddress Номер блока (0..255) или адрес Mifare Plus.
     * @param[in]  fKeyB    =`true` авторизовать по ключу Б, иначе - по ключу А.
     * @param[in]  nKeys    Биты (0..15) ключей в памяти считывателя.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @return Позиция найденного ключа в памяти считывателя, =-1 ключ не найден.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_READER_ERROR`, `ILR_E_BAD_RESPONSE`,
     * `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline int AuthMfCard2(size_t nAddress, bool fKeyB, uint32_t nKeys = 0xFFFF);

    /**
     * @brief Запускает асинхронную команду авторизации сектора карты Mifare Classic / Plus.
     * используя ключи в памяти считывателя
     *
     * @param[in]  nAddress  Номер блока или адрес Mifare Plus,
     * @param[in]  fKeyB     =`true` авторизовать по ключу B, иначе A.
     * @param[in]  nKeys     Биты номеров ключей в памяти считывателя.
     *
     * @return Класс дескриптор команды `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     *
     * @exception CILRException если произошла ошибка.
     */
    inline CAsyncCommand Begin_AuthMfCard2(size_t nAddress, bool fKeyB, uint32_t nKeys = 0xFFFF);

    /**
     * @brief Возвращает результат авторизации сектора карты.
     *
     * @param[in]  hCommand  Дескриптор команды.
     * @param[out] nFoundIdx Позиция найденного ключа в памяти считывателя, =-1 ключ не найден.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    static inline void End_AuthMfCard2(ilr_handle hCommand, int& nFoundIdx);

    /**
     * @brief Читает данные карты Mifare Classic или Mifare Plus SL1.
     *
     * @param[in]  nBlockIdx   Номер первого читаемого блока (0..255).
     * @param[out] pBuf        Буфер для прочитанных блоков.
     * @param[in]  nBlockCount Количество блоков, которые нужно прочитать.
     * @param[out] pRead       Количество прочитанных блоков. Может быть =`nullptr`.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_POINTER`, `ILR_E_INVALIDARG`,
     * `ILR_E_HANDLE`, `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`,
     * `ILR_E_OUT_OF_RESOURCES`, `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_CARD_AUTH`,
     * `ILR_E_READER_ERROR`, `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`, `ILR_E_SCARD_ERROR`,
     * `ILE_E_ABORT`.
     */
    inline void ReadMfClassic(size_t nBlockIdx, ilr_mf_block_data* pBuf, size_t nBlockCount,
                              size_t* pRead = nullptr);

    /**
     * @brief Запускает асинхронную команду чтения данных из карты Mifare Classic или Mifare Plus
     * SL1.
     *
     * @param[in]  nBlockIdx   Номер блока Mifare Classic.
     * @param[in]  nBlockCount Количество блоков.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_ReadMfClassic(size_t nBlockIdx, size_t nBlockCount);

    /**
     * @brief Возвращает результат чтения данных из карты Mifare Classic или Mifare Plus SL1.
     *
     * @param[in]  hCommand  Дескриптор команды.
     * @param[out] pList     Ссылка на список прочитанных блоков. Ссылка действительна до закрытия
     * дескриптора команды @p hCommand.
     * @param[out] nRead     Количество прочитанных блоков.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_POINTER`,
     * `ILR_E_HANDLE`.
     */
    static inline void End_ReadMfClassic(ilr_handle hCommand, const ilr_mf_block_data*& pList,
                                         size_t& nRead);

    /**
     * @brief Пишет данные карты Mifare Classic или Mifare Plus SL1.
     *
     * @param[in]  nBlockIdx   Номер первого записываемого блока (0..255).
     * @param[in]  pData       Данные записываемых блоков.
     * @param[in]  nBlockCount Количество блоков, которые нужно записать.
     * @param[out] pWritten    Количество записанных блоков. Может быть = `nullptr`.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_CARD_AUTH`, `ILR_E_READER_ERROR`,
     * `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`, `ILR_E_SCARD_ERROR`, `ILR_E_ABORT`.
     */
    inline void WriteMfClassic(size_t nBlockIdx, const ilr_mf_block_data* pData, size_t nBlockCount,
                               size_t* pWritten = nullptr);

    /**
     * @brief Запускает асинхронную команду записи данных в карту Mifare Classic или Mifare Plus
     * SL1.
     *
     * @param[in]  nBlockIdx   Номер блока Mifare Classic.
     * @param[in]  pData       Данные блоков для записи.
     * @param[in]  nBlockCount Количество записываемых блоков.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_WriteMfClassic(size_t nBlockIdx, const ilr_mf_block_data* pData,
                                              size_t nBlockCount);

    /**
     * @brief Возвращает результат записи данных в карту Mifare Classic или Mifare Plus SL1.
     *
     * @param[in]  hCommand  Дескриптор команды.
     * @param[out] nWritten Количество записанных блоков.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    static inline void End_WriteMfClassic(ilr_handle hCommand, size_t& nWritten);

    /**
     * @brief Читает данные карты Mifare Plus SL3.
     *
     * @param[in]  nAddress    Номер первого читаемого блока (0..255).
     * @param[out] pBuf        Буфер для прочитанных блоков
     * @param[in]  nBlockCount Количество блоков, которые нужно прочитать.
     * @param[in]  fOpenText   =`true` открытая передача, иначе - зашифрованная
     * @param[out] pRead       Количество прочитанных блоков. Может быть = `nullptr`.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_POINTER`,
     * `ILR_E_HANDLE`, `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`,
     * `ILR_E_OUT_OF_RESOURCES`, `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_CARD_AUTH`,
     * `ILR_E_READER_ERROR`, `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline void ReadMfPlus(size_t nAddress, ilr_mf_block_data* pBuf, size_t nBlockCount,
                           bool fOpenText = true, size_t* pRead = nullptr);

    /**
     * @brief Запускает асинхронную команду чтения данных из карты Mifare Plus SL3.
     *
     * @param[in]  nAddress    Номер блока карты Mifare.
     * @param[in]  nBlockCount Количество блоков, которые нужно прочитать.
     * @param[in]  fOpenText   =`true` открытая передача, иначе зашифрованная.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_ReadMfPlus(size_t nAddress, size_t nBlockCount, bool fOpenText);

    /**
     * @brief Возвращает результат чтения данных из карты Mifare Plus SL3.
     *
     * @param[in]  hCommand  Дескриптор команды.
     * @param[out] pList     Ссылка на список прочитанных блоков. Ссылка действительна до закрытия
     * дескриптора команды @p hCommand.
     * @param[out] nRead     Количество прочитанных блоков. Может быть равно `nullptr`.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_POINTER`,
     * `ILR_E_HANDLE`.
     */
    static inline void End_ReadMfPlus(ilr_handle hCommand, const ilr_mf_block_data*& pList,
                                      size_t& nRead);

    /**
     * @brief Пишет данные карты Mifare Plus SL3.
     *
     * @param[in]  nAddress    Номер первого записываемого блока (0..255) или адрес Mifare Plus.
     * @param[in]  pData       Данные записываемых блоков.
     * @param[in]  nBlockCount Количество блоков, которые нужно записать.
     * @param[in]  fOpenText   =`true` открытая передача, иначе - зашифрованная.
     * @param[out] pWritten    Количество записанных блоков. Может быть = `nullptr`.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_CARD_AUTH`, `ILR_E_READER_ERROR`,
     * `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline void WriteMfPlus(size_t nAddress, const ilr_mf_block_data* pData, size_t nBlockCount,
                            bool fOpenText = true, size_t* pWritten = nullptr);

    /**
     * @brief Запускает асинхронную команду записи данных в карту Mifare Plus SL3.
     *
     * @param[in]  nAddress    Номер блока или адрес Mifare Plus.
     * @param[in]  pData       Данные блоков для записи.
     * @param[in]  nBlockCount Количество блоков.
     * @param[in]  fOpenText   =`true` открытая передача, иначе зашифрованная.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_WriteMfPlus(size_t nAddress, const ilr_mf_block_data* pData,
                                           size_t nBlockCount, bool fOpenText);

    /**
     * @brief Возвращает результат записи данных в карту Mifare Plus SL3.
     *
     * @param[in]  hCommand Дескриптор команды.
     * @param[out] nWritten Количество записанных блоков.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    static inline void End_WriteMfPlus(ilr_handle hCommand, size_t& nWritten);

    /**
     * @brief Увеличивает содержимое блока-значения карты Mifare и сохраняет результат во временном
     * регистре данных.
     *
     * @param[in]  nBlockIdx Номер блока (0..255)
     * @param[in]  nValue    Величина инкремента.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_CARD_AUTH`, `ILR_E_READER_ERROR`,
     * `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline void MfIncrement(size_t nBlockIdx, uint32_t nValue);

    /**
     * @brief Увеличивает содержимое блока-значения карты Mifare и сохраняет результат во временном
     * регистре данных.
     *
     * @param[in]  nBlockIdx Номер блока Mifare Classic/Plus.
     * @param[in]  nValue    Величина инкремента.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_MfIncrement(size_t nBlockIdx, uint32_t nValue);

    /**
     * @brief Уменьшает содержимое блока-значения карты Mifare и сохраняет результат во временном
     * регистре данных.
     *
     * @param[in]  nBlockIdx Номер блока (0..255)
     * @param[in]  nValue    Величина декремента.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_CARD_AUTH`, `ILR_E_READER_ERROR`,
     * `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline void MfDecrement(size_t nBlockIdx, uint32_t nValue);

    /**
     * @brief Уменьшает содержимое блока-значения карты Mifare и сохраняет результат во временном
     * регистре данных.
     *
     * @param[in]  nBlockIdx Номер блока Mifare Classic/Plus.
     * @param[in]  nValue    Величина инкремента.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_MfDecrement(size_t nBlockIdx, uint32_t nValue);

    /**
     * @brief Записывает содержимое во временном регистре данных в блок-значение.
     *
     * @param[in]  nBlockIdx Номер блока (0..255)
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_CARD_AUTH`, `ILR_E_READER_ERROR`,
     * `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline void MfTransfer(size_t nBlockIdx);

    /**
     * @brief Записывает содержимое во временном регистре данных в блок-значение.
     *
     * @param[in]  nBlockIdx Номер блока Mifare Classic/Plus.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_MfTransfer(size_t nBlockIdx);

    /**
     * @brief Перемещает содержимое блока в регистр данных Mifare.
     *
     * @param[in]  nBlockIdx Номер блока (0..255)
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_CARD_AUTH`, `ILR_E_READER_ERROR`,
     * `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline void MfRestore(size_t nBlockIdx);

    /**
     * @brief Перемещает содержимое блока в регистр данных Mifare.
     *
     * @param[in]  nBlockIdx Номер блока Mifare Classic/Plus.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_MfRestore(size_t nBlockIdx);

    /**
     * @brief Выключает RF поле считывателя (после выключения нужно подождать 10 мс).
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_READER_ERROR`, `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`,
     * `ILR_E_ABORT`.
     */
    inline void MfPowerOff();

    /**
     * @brief R+A+S(Request+Anticollision+Select)
     *
     * @param[in]  fWakeUp  =`true` разбудить карту.
     * @param[out] nSAK     Код `SAK` карты.
     * @param[out] nATQ     Код `ATQ` карты.
     * @param[out] rUID     Номер карты.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @return true карта найдена, иначе нет карты.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_READER_ERROR`, `ILR_E_BAD_RESPONSE`,
     * `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline bool MfRas(bool fWakeUp, uint8_t& nSAK, uint16_t& nATQ, ilr_card_uid& rUID);

    /**
     * @brief R+R(Request+Reselect(по известному номеру)).
     *
     * @param[in]  fWakeUp  =`true` разбудить карту.
     * @param[in]  rUID     Номер карты.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @return true карта найдена, иначе нет карты.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_READER_ERROR`, `ILR_E_BAD_RESPONSE`,
     * `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline bool MfRR(bool fWakeUp, const ilr_card_uid& rUID);

    /**
     * @brief Halt.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_CARD_NACK`, `ILR_E_READER_ERROR`, `ILR_E_BAD_RESPONSE`,
     * `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline void MfHalt();

    /**
     * @brief Переходит на ISO 14443-4.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_READER_ERROR`, `ILR_E_BAD_RESPONSE`,
     * `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline void MfRats();

    /**
     * @brief Переходит на ISO 14443-4.
     *
     * @param[in out] oAts  Данные ATS.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_READER_ERROR`, `ILR_E_BAD_RESPONSE`,
     * `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    void MfRats(std::vector<uint8_t>& oAts);

    /**
     * @brief Записывает ключи AES и всех блоков.
     *
     * @param[in]  nAddress Адрес ключа.
     * @param[in]  rKey     Значение ключа.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_READER_ERROR`, `ILR_E_BAD_RESPONSE`,
     * `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline void MfWritePerso(uint32_t nAddress, const ilr_mf_plus_key& rKey);

    /**
     * @brief Переключает Mifare Plus в SL1 или SL3(если SL1 нет).
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_READER_ERROR`, `ILR_E_BAD_RESPONSE`,
     * `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline void MfCommitPerso();

    /**
     * @brief Записывает ключи аутентификации Mifare Classic в память считывателя.
     *
     * @param[in]  nIdx     Номер ячейки в памяти считывателя, в которую записывается первый ключ.
     * @param[in]  fKeyB    =`true` ключ Б, иначе - ключ А.
     * @param[in]  pKeys    Список записываемых ключей.
     * @param[in]  nCount   Количество ключей, которые нужно записать.
     * @param[out] pWritten Количество записанных ключей. Может быть равно `nullptr`.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_READER_ERROR`, `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`,
     * `ILR_E_ABORT`.
     */
    inline void WriteMfCKey(size_t nIdx, bool fKeyB, const ilr_mf_classic_key* pKeys, size_t nCount,
                            size_t* pWritten = nullptr);

    /**
     * @brief Запускает асинхронную команду записи ключей аутентификации Mifare Classic в память
     * считывателя.
     *
     * @param[in]  nIdx     Позиция первого ключа в памяти считывателя.
     * @param[in]  fKeyB    =`true` ключ B, иначе ключ A.
     * @param[in]  pKeys    Список ключей аутентификации.
     * @param[in]  nCount   Количество ключей аутентификации.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_WriteMfCKey(size_t nIdx, bool fKeyB, const ilr_mf_classic_key* pKeys,
                                           size_t nCount);

    /**
     * @brief Возвращает результат записи ключей аутентификации Mifare Classic в память
     * считывателя.
     *
     * @param[in]  hCommand Дескриптор команды.
     * @param[out] nWritten Количество записанных ключей.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    static inline void End_WriteMfCKey(ilr_handle hCommand, size_t& nWritten);

    /**
     * @brief Записывает ключи аутентификации Mifare Plus в память считывателя
     *
     * @param[in]  nIdx     Номер ячейки в памяти считывателя, в которую записывается первый ключ.
     * @param[in]  fKeyB    =`true` ключ Б, иначе - ключ А.
     * @param[in]  pKeys    Список записываемых ключей.
     * @param[in]  nCount   Количество ключей, которые нужно записать.
     * @param[out] pWritten Количество записанных ключей. Может быть равно `nullptr`.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_READER_ERROR`, `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`,
     * `ILR_E_ABORT`.
     */
    inline void WriteMfPKey(size_t nIdx, bool fKeyB, const ilr_mf_plus_key* pKeys, size_t nCount,
                            size_t* pWritten = nullptr);

    /**
     * @brief Запускает асинхронную команду записи ключей аутентификации Mifare Plus в память
     * считывателя.
     *
     * @param[in]  nIdx     Позиция первого ключа в памяти считывателя.
     * @param[in]  fKeyB    =`true` ключ B, иначе ключ A.
     * @param[in]  pKeys    Список ключей аутентификации.
     * @param[in]  nCount   Количество ключей аутентификации.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_WriteMfPKey(size_t nIdx, bool fKeyB, const ilr_mf_plus_key* pKeys,
                                           size_t nCount);

    /**
     * @brief Возвращает результат записи ключей аутентификации Mifare Plus в память считывателя.
     *
     * @param[in]  hCommand Дескриптор команды.
     * @param[out] nWritten Количество записанных ключей.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    static inline void End_WriteMfPKey(ilr_handle hCommand, size_t& nWritten);

    /**
     * @brief Загружает пароль Temic в память объекта считывателя.
     *
     * @param[in]  nPassword Пароль Temic. Если =-1, то нет пароля.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline void LoadTemicPassword(const int64_t nPassword);

    /**
     * @brief Ищет карту Temic в поле считывателя.
     *
     * @param[in]  nScanParam Параметры сканирования Temic. =-1 авто определение.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_READER_ERROR`, `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`,
     * `ILR_E_ABORT`.
     */
    inline void ScanTemic(int nScanParam = -1);

    /**
     * @brief Запускает асинхронную команду поиска карты Temic в поле считывателя.
     *
     * @param[in]  nScanParam Параметры сканирования Temic. =-1 авто определение.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_ScanTemic(int nScanParam = -1);

    /**
     * @brief Вкл/выкл сканирование карт Temic (для Z-2 Rd-All и Z-2 EHR).
     *
     * Если включено, то ищет карту Temic при ручном или автоматическом поиске карт (
     * @ref ilr_reader_scan, @ref ilr_reader_set_auto_scan).
     *
     * @param[in]  fEnable  =`true` включает сканирование Temic.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline void SetScanTemic(bool fEnable = true);

    /**
     * @brief Возвращает True если сканирование Temic включено.
     *
     * @return True если сканирование Temic включено.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    inline bool GetScanTemic() const;

    /**
     * @brief Читает данные карты Temic.
     *
     * @param[in]  nBlockIdx   Номер первого блока, который нужно причитать (0..9).
     * @param[out] pBuf        Буфер для прочитанных данных.
     * @param[in]  nBlockCount Количество блоков, которые нужно прочитать.
     * @param[in]  nScanParam  Параметры сканирования Temic. =-1 авто определение.
     * @param[out] pRead       Количество прочитанных блоков. Может быть равно `nullptr`.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_POINTER`, `ILR_E_INVALIDARG`,
     * `ILR_E_HANDLE`, `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`,
     * `ILR_E_OUT_OF_RESOURCES`, `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_READER_ERROR`,
     * `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline void ReadTemic(size_t nBlockIdx, uint32_t* pBuf, size_t nBlockCount, int nScanParam = -1,
                          size_t* pRead = nullptr);

    /**
     * @brief Запускает асинхронную команду чтения данных из карты Temic.
     *
     * @param[in]  nBlockIdx   Номер блока карты.
     * @param[in]  nBlockCount Количество блоков.
     * @param[in]  nScanParam  Параметр сканирования Temic. =-1 авто определение.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_ReadTemic(size_t nBlockIdx, size_t nBlockCount, int nScanParam = -1);

    /**
     * @brief Возвращает результат чтения данных из карты Temic.
     *
     * @param[in]  hCommand  Дескриптор команды.
     * @param[out] pList     Ссылка на список прочитанных блоков. Ссылка действительна до закрытия
     * дескриптора команды @p hCommand.
     * @param[out] nRead     Количество прочитанных блоков. Может быть равно `nullptr`.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_POINTER`,
     * `ILR_E_HANDLE`.
     */
    static inline void End_ReadTemic(ilr_handle hCommand, const uint*& pList, size_t& nRead);

    /**
     * @brief Пишет данные карты Temic.
     *
     * @param[in]  nBlockIdx   Номер первого блока, в который нужно записать (0..7).
     * @param[in]  pData       Данные блоков для записи.
     * @param[in]  nBlockCount Количество блоков, которые нужно записать.
     * @param[in]  fLock       =`true` заблокировать блоки от перезаписи.
     * @param[in]  nScanParam  Параметры сканирования Temic. =-1 авто определение.
     * @param[out] pWritten    Количество записанных блоков. Может быть равно `nullptr`.
     *
     * @remark Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`,
     * `ILR_E_NOTIMPL`, `ILR_E_NO_CARD`, `ILR_E_WRITE_T57`, `ILR_E_READER_ERROR`,
     * `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline void WriteTemic(size_t nBlockIdx, const uint32_t* pData, size_t nBlockCount,
                           bool fLock = false, int nScanParam = -1, size_t* pWritten = nullptr);

    /**
     * @brief Запускает асинхронную команду записи данных в карту Temic.
     *
     * @param[in]  nBlockIdx   Номер блока карты.
     * @param[in]  pData       Данные для записи.
     * @param[in]  nBlockCount Количество блоков.
     * @param[in]  fLock       =`true` блокировать от перезаписи.
     * @param[in]  nScanParam  Параметр сканирования Temic. =-1 авто определение.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CAsyncCommand Begin_WriteTemic(size_t nBlockIdx, const uint32_t* pData,
                                          size_t nBlockCount, bool fLock = false,
                                          int nScanParam = -1);

    /**
     * @brief Возвращает результат записи данных в карту Temic.
     *
     * @param[in]  hCommand Дескриптор команды.
     * @param[out] nWritten Количество записанных блоков.
     *
     * @exception CILRException(ILR_E_HANDLE) если дескриптор неправильный.
     */
    static inline void End_WriteTemic(ilr_handle hCommand, size_t& nWritten);

    /**
     * @brief Сброс TRES.
     *
     * @param[in]  fWait    =`true` ждать завершения команды.
     *
     * @remark Если fWait = true, то функция не возвращает управление пока ждёт выполнение команды
     * в потоке считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline void ResetTemic(bool fWait = true);

    /**
     * @brief Запускает асинхронную команду сброса TRES.
     *
     * @return Класс команды.
     *
     * @exception CILRException если произошла ошибка `ILR_E_HANDLE`,
     * `ILR_E_OUTOFMEMORY`, `ILR_E_OUT_OF_RESOURCES`, `ILR_E_NOTIMPL`, `ILR_E_READER_ERROR`,
     * `ILR_E_BAD_RESPONSE`, `ILR_E_REQUEST_TIMEOUT`, `ILR_E_ABORT`.
     */
    inline CAsyncCommand Begin_ResetTemic();

    /**
     * @brief Кодирует данные для эмуляции Em-Marine, для записи в блоки 0..2.
     *
     * @param[in]  rUID        Номер Em-Marine, который нужно эмулировать.
     * @param[out] pBuf3       Буфер для данных 3-х блоков.
     * @param[in]  nBlockCount Размер буфера в блоках.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_POINTER`,
     * `ILR_E_BUFFER_TOO_SMALL`.
     */
    static inline void EncodeTemicEmMarine(const ilr_card_uid& rUID, uint32_t* pBuf3,
                                           size_t nBlockCount);

    /**
     * @brief Декодирует номер Em-Marine из данных блоков 0..2 карты Temic.
     *
     * @param[in]  pData3      Данные блоков 0..2.
     * @param[in]  nBlockCount Количество блоков. Должно быть не меньше 3.
     * @param[out] rUID        Номер Em-Marine. Если Em-Marine не обнаружен, то пустой номер.
     * @param[out] pConfigOk   =`true` конфигурация Temic для эмуляции Em-Marine правильная. Может
     * быть равно `nullptr`.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`.
     */
    static inline void DecodeTemicEmMarine(const uint32_t* pData3, size_t nBlockCount,
                                           ilr_card_uid& rUID, bool* pConfigOk = nullptr);

    /**
     * @brief Кодирует данные для эмуляции HID, для записи в блоки 0..3.
     *
     * @param[in]  rUID        Номер HID, который нужно эмулировать.
     * @param[out] pBuf4       Буфер для данных 4-х блоков.
     * @param[in]  nBlockCount Размер буфера в блоках.
     * @param[in]  nWiegand    Номер кодировки Wiegand 18..37.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`, `ILR_E_POINTER`,
     * `ILR_E_BUFFER_TOO_SMALL`.
     */
    static inline void EncodeTemicHid(const ilr_card_uid& rUID, uint32_t* pBuf4, size_t nBlockCount,
                                      int nWiegand);

    /**
     * @brief Декодирует номер HID из данных блоков 0..3 карты Temic.
     *
     * @param[in]  pData4      Данные блоков 0..3.
     * @param[in]  nBlockCount Количество блоков. Должно быть не меньше 4.
     * @param[out] rUID        Номер HID. Если HID не обнаружен, то пустой номер.
     * @param[out] nWiegand    Номер кодировки Wiegand.
     * @param[out] pConfigOk   =`true` конфигурация Temic правильная. Может быть равно `nullptr`.
     *
     * @exception CILRException если произошла ошибка `ILR_E_INVALIDARG`.
     */
    static inline void DecodeTemicHid(const uint32_t* pData4, size_t nBlockCount,
                                      ilr_card_uid& rUID, int& nWiegand, bool* pConfigOk = nullptr);

    /**
     * @brief Возвращает биты типов карт, UID которых умеет читать считыватель.
     *
     * @param[in]  nReaderModel   Модель считывателя.
     */
    static uint32_t GetSupportedCardTypes(ilr_reader_model nReaderModel);

    /**
     * @brief Возвращает типы карт, данные которых умеет читать/писать считыватель
     *
     * @param[in]  nReaderModel   Модель считывателя.
     * @param[in]  nFwVersion     Версия прошивки считывателя.
     */
    static uint32_t GetSupportedRewrittenCardTypes(ilr_reader_model nReaderModel,
                                                   uint32_t nFwVersion);

    /**
     * @brief Клонирует дескриптор считывателя.
     * @return Дескриптор считывателя.
     */
    inline CReader Clone() const;
};

inline void CReader::SetMessageCallback(ilr_reader_message_callback pCallback, void* pUserData) {
    ILRCheck(ilr_reader_set_message_callback(m_h, pCallback, pUserData));
}

inline void CReader::EnableMessageQueue(bool fEnable) {
    ILRCheck(ilr_reader_enable_message_queue(m_h, (ilr_bool)fEnable));
}

inline bool CReader::GetMessage(ilr_reader_msg& nMsg, const void*& pMsgData) const {
    ilr_bool f;
    ILRCheck(ilr_reader_get_message(m_h, &nMsg, &pMsgData, &f));
    return f;
}

inline void CReader::SetOptions(const ilr_reader_options& rOptions) {
    ILRCheck(ilr_reader_set_options(m_h, &rOptions));
}

inline void CReader::GetOptions(ilr_reader_options& rOptions) const {
    ILRCheck(ilr_reader_get_options(m_h, &rOptions));
}

inline void CReader::Connect(bool fReconnect) {
    ILRCheck(ilr_reader_connect(m_h, (ilr_bool)fReconnect));
}

inline CAsyncCommand CReader::Begin_Connect(bool fReconnect) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_connect(m_h, (ilr_bool)fReconnect, &res.m_h));
    return res;
}

inline void CReader::Disconnect() {
    ILRCheck(ilr_reader_disconnect(m_h));
}

inline CAsyncCommand CReader::Begin_Disconnect() {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_disconnect(m_h, &res.m_h));
    return res;
}

inline ilr_connection_status CReader::GetConnectionStatus() const {
    ilr_connection_status res;
    ILRCheck(ilr_reader_get_connection_status(m_h, &res));
    return res;
}

inline void CReader::GetReaderInfo(ilr_reader_info& rInfo) const {
    ILRCheck(ilr_reader_get_info(m_h, &rInfo));
}

inline void CReader::Scan(bool fReset, bool fPowerOff) {
    ILRCheck(ilr_reader_scan(m_h, (ilr_bool)fReset, (ilr_bool)fPowerOff));
}

inline CAsyncCommand CReader::Begin_Scan(bool fReset, bool fPowerOff) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_scan(m_h, (ilr_bool)fReset, (ilr_bool)fPowerOff, &res.m_h));
    return res;
}

inline bool CReader::GetCardInfo(ilr_card_info& rInfo) const {
    ILRCheck(ilr_reader_get_card_info(m_h, &rInfo));
    return (rInfo.rUID.nLength != 0);
}

inline void CReader::SetAutoScan(bool fEnable, bool fWait) {
    ILRCheck(ilr_reader_set_auto_scan(m_h, (ilr_bool)fEnable, (ilr_bool)fWait));
}

inline CAsyncCommand CReader::Begin_SetAutoScan(bool fEnable) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_set_auto_scan(m_h, (ilr_bool)fEnable, &res.m_h));
    return res;
}

inline bool CReader::GetAutoScan() const {
    ilr_bool res;
    ILRCheck(ilr_reader_get_auto_scan(m_h, &res));
    return static_cast<bool>(res);
}

inline void CReader::ReadMfUltralight(size_t nPageIdx, uint32_t* pBuf, size_t nPageCount,
                                      size_t* pRead) {
    ILRCheck(ilr_reader_read_mf_ultralight(m_h, nPageIdx, pBuf, nPageCount, pRead));
}

inline CAsyncCommand CReader::Begin_ReadMfUltralight(size_t nPageIdx, size_t nPageCount) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_read_mf_ultralight(m_h, nPageIdx, nPageCount, &res.m_h));
    return res;
}

inline void CReader::End_ReadMfUltralight(ilr_handle hCommand, const uint32_t*& pList,
                                          size_t& nRead) {
    ILRCheck(ilr_reader_end_read_mf_ultralight(hCommand, &pList, &nRead));
}

inline void CReader::WriteMfUltralight(size_t nPageIdx, const uint32_t* pData, size_t nPageCount,
                                       size_t* pWritten) {
    ILRCheck(ilr_reader_write_mf_ultralight(m_h, nPageIdx, pData, nPageCount, pWritten));
}

inline CAsyncCommand CReader::Begin_WriteMfUltralight(size_t nPageIdx, const uint32_t* pData,
                                                      size_t nPageCount) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_write_mf_ultralight(m_h, nPageIdx, pData, nPageCount, &res.m_h));
    return res;
}

inline void CReader::End_WriteMfUltralight(ilr_handle hCommand, size_t& nWritten) {
    ILRCheck(ilr_reader_end_write_mf_ultralight(hCommand, &nWritten));
}

inline void CReader::LoadMfCKey(const ilr_mf_classic_key& nKey) {
    ILRCheck(ilr_reader_load_mf_ckey(m_h, nKey));
}

inline void CReader::LoadMfPKey(const ilr_mf_plus_key& rKey) {
    ILRCheck(ilr_reader_load_mf_pkey(m_h, rKey));
}

inline bool CReader::AuthMfCard(size_t nAddress, bool fKeyB) {
    ilr_bool res;
    ILRCheck(ilr_reader_auth_mf_card(m_h, nAddress, (ilr_bool)fKeyB, &res));
    return static_cast<bool>(res);
}

inline CAsyncCommand CReader::Begin_AuthMfCard(size_t nAddress, bool fKeyB) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_auth_mf_card(m_h, nAddress, (ilr_bool)fKeyB, &res.m_h));
    return res;
}

inline void CReader::End_AuthMfCard(ilr_handle hCommand, bool& fAuthOk) {
    ilr_bool f;
    ILRCheck(ilr_reader_end_auth_mf_card(hCommand, &f));
    fAuthOk = static_cast<bool>(f);
}

inline int CReader::AuthMfCard2(size_t nAddress, bool fKeyB, uint32_t nKeys) {
    int res;
    ILRCheck(ilr_reader_auth_mf_card2(m_h, nAddress, (ilr_bool)fKeyB, nKeys, &res));
    return res;
}

inline CAsyncCommand CReader::Begin_AuthMfCard2(size_t nAddress, bool fKeyB, uint32_t nKeys) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_auth_mf_card2(m_h, nAddress, (ilr_bool)fKeyB, nKeys, &res.m_h));
    return res;
}

inline void CReader::End_AuthMfCard2(ilr_handle hCommand, int& nFoundIdx) {
    ILRCheck(ilr_reader_end_auth_mf_card2(hCommand, &nFoundIdx));
}

inline void CReader::ReadMfClassic(size_t nBlockIdx, ilr_mf_block_data* pBuf, size_t nBlockCount,
                                   size_t* pRead) {
    ILRCheck(ilr_reader_read_mf_classic(m_h, nBlockIdx, pBuf, nBlockCount, pRead));
}

inline CAsyncCommand CReader::Begin_ReadMfClassic(size_t nBlockIdx, size_t nBlockCount) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_read_mf_classic(m_h, nBlockIdx, nBlockCount, &res.m_h));
    return res;
}

inline void CReader::End_ReadMfClassic(ilr_handle hCommand, const ilr_mf_block_data*& pList,
                                       size_t& nRead) {
    ILRCheck(ilr_reader_end_read_mf_classic(hCommand, &pList, &nRead));
}

inline void CReader::WriteMfClassic(size_t nBlockIdx, const ilr_mf_block_data* pData,
                                    size_t nBlockCount, size_t* pWritten) {
    ILRCheck(ilr_reader_write_mf_classic(m_h, nBlockIdx, pData, nBlockCount, pWritten));
}

inline CAsyncCommand CReader::Begin_WriteMfClassic(size_t nBlockIdx, const ilr_mf_block_data* pData,
                                                   size_t nBlockCount) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_write_mf_classic(m_h, nBlockIdx, pData, nBlockCount, &res.m_h));
    return res;
}

inline void CReader::End_WriteMfClassic(ilr_handle hCommand, size_t& nWritten) {
    ILRCheck(ilr_reader_end_write_mf_classic(hCommand, &nWritten));
}

inline void CReader::ReadMfPlus(size_t nAddress, ilr_mf_block_data* pBuf, size_t nBlockCount,
                                bool fOpenText, size_t* pRead) {
    ILRCheck(ilr_reader_read_mf_plus(m_h, nAddress, pBuf, nBlockCount, (ilr_bool)fOpenText, pRead));
}

inline CAsyncCommand CReader::Begin_ReadMfPlus(size_t nAddress, size_t nBlockCount,
                                               bool fOpenText) {
    CAsyncCommand res;
    ILRCheck(
        ilr_reader_begin_read_mf_plus(m_h, nAddress, nBlockCount, (ilr_bool)fOpenText, &res.m_h));
    return res;
}

inline void CReader::End_ReadMfPlus(ilr_handle hCommand, const ilr_mf_block_data*& pList,
                                    size_t& nRead) {
    ILRCheck(ilr_reader_end_read_mf_plus(hCommand, &pList, &nRead));
}

inline void CReader::WriteMfPlus(size_t nAddress, const ilr_mf_block_data* pData,
                                 size_t nBlockCount, bool fOpenText, size_t* pWritten) {
    ILRCheck(
        ilr_reader_write_mf_plus(m_h, nAddress, pData, nBlockCount, (ilr_bool)fOpenText, pWritten));
}

inline CAsyncCommand CReader::Begin_WriteMfPlus(size_t nAddress, const ilr_mf_block_data* pData,
                                                size_t nBlockCount, bool fOpenText) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_write_mf_plus(m_h, nAddress, pData, nBlockCount, (ilr_bool)fOpenText,
                                            &res.m_h));
    return res;
}

inline void CReader::End_WriteMfPlus(ilr_handle hCommand, size_t& nWritten) {
    ILRCheck(ilr_reader_end_write_mf_plus(hCommand, &nWritten));
}

inline void CReader::MfIncrement(size_t nBlockIdx, uint32_t nValue) {
    ILRCheck(ilr_reader_mf_increment(m_h, nBlockIdx, nValue));
}

inline CAsyncCommand CReader::Begin_MfIncrement(size_t nBlockIdx, uint32_t nValue) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_mf_increment(m_h, nBlockIdx, nValue, &res.m_h));
    return res;
}

inline void CReader::MfDecrement(size_t nBlockIdx, uint32_t nValue) {
    ILRCheck(ilr_reader_mf_decrement(m_h, nBlockIdx, nValue));
}

inline CAsyncCommand CReader::Begin_MfDecrement(size_t nBlockIdx, uint32_t nValue) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_mf_decrement(m_h, nBlockIdx, nValue, &res.m_h));
    return res;
}

inline void CReader::MfTransfer(size_t nBlockIdx) {
    ILRCheck(ilr_reader_mf_transfer(m_h, nBlockIdx));
}

inline CAsyncCommand CReader::Begin_MfTransfer(size_t nBlockIdx) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_mf_transfer(m_h, nBlockIdx, &res.m_h));
    return res;
}

inline void CReader::MfRestore(size_t nBlockIdx) {
    ILRCheck(ilr_reader_mf_restore(m_h, nBlockIdx));
}

inline CAsyncCommand CReader::Begin_MfRestore(size_t nBlockIdx) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_mf_restore(m_h, nBlockIdx, &res.m_h));
    return res;
}

inline void CReader::MfPowerOff() {
    ILRCheck(ilr_reader_mf_power_off(m_h));
}

inline bool CReader::MfRas(bool fWakeUp, uint8_t& nSAK, uint16_t& nATQ, ilr_card_uid& rUID) {
    ilr_bool res;
    ILRCheck(ilr_reader_mf_ras(m_h, (ilr_bool)fWakeUp, &nSAK, &nATQ, &rUID, &res));
    return static_cast<bool>(res);
}

inline bool CReader::MfRR(bool fWakeUp, const ilr_card_uid& rUID) {
    ilr_bool res;
    ILRCheck(ilr_reader_mf_rr(m_h, (ilr_bool)fWakeUp, rUID, &res));
    return static_cast<bool>(res);
}

inline void CReader::MfHalt() {
    ILRCheck(ilr_reader_mf_halt(m_h));
}

inline void CReader::MfRats() {
    ILRCheck(ilr_reader_mf_rats(m_h));
}

inline void CReader::MfWritePerso(uint32_t nAddress, const ilr_mf_plus_key& rKey) {
    ILRCheck(ilr_reader_mf_write_perso(m_h, nAddress, rKey));
}

inline void CReader::MfCommitPerso() {
    ILRCheck(ilr_reader_mf_commit_perso(m_h));
}

inline void CReader::WriteMfCKey(size_t nIdx, bool fKeyB, const ilr_mf_classic_key* pKeys,
                                 size_t nCount, size_t* pWritten) {
    ILRCheck(ilr_reader_write_mf_ckey(m_h, nIdx, (ilr_bool)fKeyB, pKeys, nCount, pWritten));
}

inline CAsyncCommand CReader::Begin_WriteMfCKey(size_t nIdx, bool fKeyB,
                                                const ilr_mf_classic_key* pKeys, size_t nCount) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_write_mf_ckey(m_h, nIdx, (ilr_bool)fKeyB, pKeys, nCount, &res.m_h));
    return res;
}

inline void CReader::End_WriteMfCKey(ilr_handle hCommand, size_t& nWritten) {
    ILRCheck(ilr_reader_end_write_mf_ckey(hCommand, &nWritten));
}

inline void CReader::WriteMfPKey(size_t nIdx, bool fKeyB, const ilr_mf_plus_key* pKeys,
                                 size_t nCount, size_t* pWritten) {
    ILRCheck(ilr_reader_write_mf_pkey(m_h, nIdx, (ilr_bool)fKeyB, pKeys, nCount, pWritten));
}

inline CAsyncCommand CReader::Begin_WriteMfPKey(size_t nIdx, bool fKeyB,
                                                const ilr_mf_plus_key* pKeys, size_t nCount) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_write_mf_pkey(m_h, nIdx, (ilr_bool)fKeyB, pKeys, nCount, &res.m_h));
    return res;
}

inline void CReader::End_WriteMfPKey(ilr_handle hCommand, size_t& nWritten) {
    ILRCheck(ilr_reader_end_write_mf_pkey(hCommand, &nWritten));
}

inline void CReader::LoadTemicPassword(const int64_t nPassword) {
    ILRCheck(ilr_reader_load_temic_password(m_h, nPassword));
}

inline void CReader::ScanTemic(int nScanParam) {
    ILRCheck(ilr_reader_scan_temic(m_h, nScanParam));
}

inline CAsyncCommand CReader::Begin_ScanTemic(int nScanParam) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_scan_temic(m_h, nScanParam, &res.m_h));
    return res;
}

inline void CReader::SetScanTemic(bool fEnable) {
    ILRCheck(ilr_reader_set_scan_temic(m_h, (ilr_bool)fEnable));
}

inline bool CReader::GetScanTemic() const {
    ilr_bool res;
    ILRCheck(ilr_reader_get_scan_temic(m_h, &res));
    return static_cast<bool>(res);
}

inline void CReader::ReadTemic(size_t nBlockIdx, uint32_t* pBuf, size_t nBlockCount, int nScanParam,
                               size_t* pRead) {
    ILRCheck(ilr_reader_read_temic(m_h, nBlockIdx, pBuf, nBlockCount, nScanParam, pRead));
}

inline CAsyncCommand CReader::Begin_ReadTemic(size_t nBlockIdx, size_t nBlockCount,
                                              int nScanParam) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_read_temic(m_h, nBlockIdx, nBlockCount, nScanParam, &res.m_h));
    return res;
}

inline void CReader::End_ReadTemic(ilr_handle hCommand, const uint*& pList, size_t& nRead) {
    ILRCheck(ilr_reader_end_read_temic(hCommand, &pList, &nRead));
}

inline void CReader::WriteTemic(size_t nBlockIdx, const uint32_t* pData, size_t nBlockCount,
                                bool fLock, int nScanParam, size_t* pWritten) {
    ILRCheck(ilr_reader_write_temic(m_h, nBlockIdx, pData, nBlockCount, (ilr_bool)fLock, nScanParam,
                                    pWritten));
}

inline CAsyncCommand CReader::Begin_WriteTemic(size_t nBlockIdx, const uint32_t* pData,
                                               size_t nBlockCount, bool fLock, int nScanParam) {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_write_temic(m_h, nBlockIdx, pData, nBlockCount, (ilr_bool)fLock,
                                          nScanParam, &res.m_h));
    return res;
}

inline void CReader::End_WriteTemic(ilr_handle hCommand, size_t& nWritten) {
    ILRCheck(ilr_reader_end_write_temic(hCommand, &nWritten));
}

inline void CReader::ResetTemic(bool fWait) {
    ILRCheck(ilr_reader_reset_temic(m_h, (ilr_bool)fWait));
}

inline CAsyncCommand CReader::Begin_ResetTemic() {
    CAsyncCommand res;
    ILRCheck(ilr_reader_begin_reset_temic(m_h, &res.m_h));
    return res;
}

inline void CReader::EncodeTemicEmMarine(const ilr_card_uid& rUID, uint32_t* pBuf3,
                                         size_t nBlockCount) {
    ILRCheck(ilr_encode_temic_emmarine(rUID, pBuf3, nBlockCount));
}

inline void CReader::DecodeTemicEmMarine(const uint32_t* pData3, size_t nBlockCount,
                                         ilr_card_uid& rUID, bool* pConfigOk) {
    ilr_bool fConfigOk;
    ILRCheck(ilr_decode_temic_emmarine(pData3, nBlockCount, &rUID, &fConfigOk));
    if (pConfigOk)
        *pConfigOk = static_cast<bool>(fConfigOk);
}

inline void CReader::EncodeTemicHid(const ilr_card_uid& rUID, uint32_t* pBuf4, size_t nBlockCount,
                                    int nWiegand) {
    ILRCheck(ilr_encode_temic_hid(rUID, pBuf4, nBlockCount, nWiegand));
}

inline void CReader::DecodeTemicHid(const uint32_t* pData4, size_t nBlockCount, ilr_card_uid& rUID,
                                    int& nWiegand, bool* pConfigOk) {
    ilr_bool fConfigOk;
    ILRCheck(ilr_decode_temic_hid(pData4, nBlockCount, &rUID, &nWiegand, &fConfigOk));
    if (pConfigOk)
        *pConfigOk = static_cast<bool>(fConfigOk);
}

inline CReader CReader::Clone() const {
    ilr_handle hNewHandle;
    ILRCheck(ilr_clone_handle(m_h, &hNewHandle));
    return CReader(hNewHandle);
}

/**
 * \brief Класс для инициализации/финализации библиотеки SDK.
 */
class CILR final {
public:
    /**
     * @brief Конструктор по умолчанию
     *
     * Инициализирует библиотеку с помощью функции @ref ilr_init.
     */
    CILR(bool fInit = true);

    /**
     * @brief Деструктор
     *
     * Завершает работу с библиотекой с помощью @ref ilr_cleanup.
     */
    ~CILR();

    /** Инициализирует библиотеку. */
    void Init();

    /** Завершает работу библиотеки. */
    void Finalize();

    /**
     * @brief Возвращает номер версии библиотеки.
     */
    static inline uint32_t GetVersion();

    /**
     * @brief Проверяет версию SDK.
     *
     * @return True, если версия библиотеки совместима с заголовочным файлом.
     */
    static inline bool CheckVersion();

    /**
     * @brief Устанавливает уровень лога отладки.
     *
     * @param[in] nLevel Уровень лога отладки.
     */
    static inline void SetLogLevel(ilr_log_level nLevel);

    /**
     * @brief Устанавливает функцию обратного вызова для лога отладки.
     *
     * @param[in] pCallback Указатель на функцию, которую библиотека будет вызывать для передачи
     * сообщений лога отладки.
     * @param[in] pUserData Указатель на данные пользователя, который будет передаваться в функцию.
     *
     * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в try catch.
     * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
     * из которого вызвана эта callback-функция, иначе вернёт ошибку
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`.
     */
    static inline void SetLogCallback(ilr_logging_callback pCallback, void* pUserData = nullptr);

    /**
     * @brief Устанавливает функцию обратного вызова для фильтрации портов при поиске считывателей.
     *
     * @param[in] pCallback Указатель на функцию, которую поток поиска считывателей будет вызывать
     * при нахождении порта.
     * @param[in] pUserData Указатель на данные пользователя, который будет передаваться в функцию.
     *
     * @warning Не позволяйте исключениям выйти из callback-функции, оберните код в `try catch`.
     * Нельзя из callback-функции вызывать функции, которые ждут выполнение команды в потоке,
     * из которого вызвана эта callback-функция, иначе вернёт ошибку
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`.
     *
     * @exception CILRException если произошла ошибка `ILR_E_OUTOFMEMORY`, `ILR_E_FAIL`.
     */
    inline void SetFilterPortCallback(ilr_filter_port_callback pCallback,
                                      void* pUserData = nullptr);

    /**
     * @brief Устанавливает глобальные настройки библиотеки.
     *
     * @param[in] rOptions Опции библиотеки.
     *
     * @exception CILRException если произошла ошибка `ILR_E_OUTOFMEMORY`, `ILR_E_FAIL`.
     */
    inline void SetOptions(const ilr_options& rOptions);

    /**
     * @brief Возвращает глобальные настройки библиотеки.
     *
     * @param[out] rOptions Опции библиотеки.
     *
     * @exception CILRException если произошла ошибка `ILR_E_OUTOFMEMORY`, `ILR_E_FAIL`.
     */
    inline void GetOptions(ilr_options& rOptions) const;

    /**
     * @brief Создаёт дескриптор поиска считывателей.
     *
     * @remark Эта функция создаёт поток поиска считывателей, если ещё не создан (один поток на
     * библиотеку). Функция не возвращает управление пока ждёт выполнение команды в потоке поиска
     * считывателей.
     *
     * @return Класс поиска считывателей.
     *
     * @exception CILRException если произошла ошибка `ILR_E_OUTOFMEMORY`, `ILR_E_FAIL`,
     * `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CReaderSearch GetSearch();

    /**
     * @brief Создаёт дескриптор считывателя.
     *
     * @param[in]  nPortType   Тип порта.
     * @param[in]  pszPortName Имя порта.
     *
     * @remark Эта функция создаёт поток считывателя, если ещё не создан (один поток на порт).
     * Функция не возвращает управление пока ждёт выполнение команды в потоке считывателя.
     * Функция позволяет создать более одного дескриптора для одного порта.
     *
     * @return Класс считывателя.
     *
     * @exception CILRException если произошла ошибка `ILR_E_OUTOFMEMORY`, `ILR_E_FAIL`,
     * `ILR_E_INVALIDARG`, `ILR_E_BLOCKING_CALL_NOT_ALLOWED`, `ILR_E_OUT_OF_RESOURCES`.
     */
    inline CReader GetReader(ilr_port_type nPortType, const char* pszPortName);

protected:
    /** true, библиотека инициализирована. */
    bool m_fInit = false;
};

inline uint32_t CILR::GetVersion() {
    return ilr_get_version();
}

inline bool CILR::CheckVersion() {
    return ilr_check_version();
}

inline void CILR::SetLogLevel(ilr_log_level nLevel) {
    ILRCheck(ilr_set_log_level(nLevel));
}

inline void CILR::SetLogCallback(ilr_logging_callback pCallback, void* pUserData) {
    ILRCheck(ilr_set_log_callback(pCallback, pUserData));
}

inline void CILR::SetFilterPortCallback(ilr_filter_port_callback pCallback, void* pUserData) {
    ILRCheck(ilr_set_filter_port_callback(pCallback, pUserData));
}

inline void CILR::SetOptions(const ilr_options& rOptions) {
    ILRCheck(ilr_set_options(&rOptions));
}

inline void CILR::GetOptions(ilr_options& rOptions) const {
    ILRCheck(ilr_get_options(&rOptions));
}

inline CReaderSearch CILR::GetSearch() {
    CReaderSearch res;
    ILRCheck(ilr_get_search(&res.m_h));
    return res;
}

inline CReader CILR::GetReader(ilr_port_type nPortType, const char* pszPortName) {
    CReader res;
    ILRCheck(ilr_get_reader(nPortType, pszPortName, &res.m_h));
    return res;
}

/**
 * @brief Возвращает true если бит bitN установлен в числе val.
 *
 * @param[in] val  Число, в котором проверяется бит.
 * @param[in] bitN Номер бита (отсчёт от 0).
 */
#define GET_BIT(val, bitN) (bool)((val >> (bitN)) & 1)
/**
 * @brief Устанавливает/снимает бит bitN в числе val.
 *
 * @param[in] val  Число, в котором изменяется бит.
 * @param[in] bitN Номер бита (отсчёт от 0).
 * @param[in] On   true, установить бит, иначе - снять.
 */
#define SET_BIT(val, bitN, On) \
    { (On) ? ((val) |= (1 << (bitN))) : ((val) &= ~(1 << (bitN))); }

/** Возвращает текущее время. */
inline std::chrono::steady_clock::time_point now() {
    return std::chrono::steady_clock::now();
}

/**
 * @brief Возвращает интервал времени в миллисекундах от времени start до текущего времени.
 *
 * @param[in] start Начальное время.
 */
template <class result_t = std::chrono::milliseconds, class clock_t = std::chrono::steady_clock,
          class duration_t = std::chrono::milliseconds>
inline auto since(std::chrono::time_point<clock_t, duration_t> const& start) {
    return std::chrono::duration_cast<result_t>(clock_t::now() - start);
}

/// Названия типов портов считывателей
extern const char* kPortTypeNames[ILR_PORT_TYPE_SIZE];
/// Названия моделей считывателей
extern const char* kReaderModelNames[ILR_READER_MODEL_SIZE];
/// Названия типов карт
extern const char* kCardTypeNames[ILR_CARD_TYPE_SIZE];
/// Название типов Mifare Plus
extern const char* kMpTypeNames[ILR_MF_PLUS_TYPE_SIZE];

/**
 * @brief Преобразует версию Sdk Readers в строку.
 *
 * @param[in] nVersion Версия SDK.
 */
std::string SdkVersionToStr(uint32_t nVersion);

/**
 * @brief Преобразует версию считывателя в строку.
 *
 * @param[in] nVersion Версия считывателя.
 */
std::string ReaderVersionToStr(uint32_t nVersion);

/**
 * @brief Преобразует время в строку.
 *
 * @param[in] tTime  Время.
 */
std::string TimeToStr(const int64_t& tTime);

/**
 * @brief Проверяет является ли карта картой Mifare Classic.
 *
 * @param[in]  nType   Тип карты.
 *
 * @return True это Mifare Classic.
 */
bool IsMfClassic(ilr_card_type nType);

/**
 * @brief Проверяет является ли карта картой Mifare Classic или её эмуляцией.
 *
 * @param[in]  nType   Тип карты.
 * @param[in]  nSL     Уровень безопасности Mifare Plus.
 *
 * @return True это Mifare Classic или Mifare Plus SL1.
 */
bool IsMfClassicMode(ilr_card_type nType, ilr_mf_plus_sl nSL);

/**
 * @brief Проверяет является ли карта картой Mifare Plus.
 *
 * @param[in]  nType   Тип карты.
 *
 * @return True это Mifare Plus.
 */
bool IsMfPlus(ilr_card_type nType);

/**
 * @brief Проверяет является ли карта картой Mifare Plus с уровнем безопасности SL3.
 *
 *  @param[in]  nType   Тип карты.
 *  @param[in]  nSL     Уровень безопасности Mifare Plus.
 *
 *  @return True это Mifare Plus SL3.
 */
bool IsMfPlusSL3Mode(ilr_card_type nType, ilr_mf_plus_sl nSL);

/**
 * @brief Вычисляет количество блоков карты Mifare Classic/Plus по размеру её памяти.
 *
 * @param[in] nMemSize  Размер памяти карты.
 *
 * @return Количество блоков Mifare.
 */
inline size_t GetNumberOfMfBlocks(uint32_t nMemSize) {
    return nMemSize / 16;
}

/**
 * @brief Возвращает номер сектора по номеру блока карты Mifare Classic/Plus.
 *
 * @param[in] nBlockIdx  Номер блока.
 *
 * @return Номер сектора.
 */
inline size_t GetMfSectorByBlock(size_t nBlockIdx) {
    return (nBlockIdx < 128) ? (nBlockIdx / 4) : (32 + ((nBlockIdx - 128) / 16));
}

/**
 * @brief Возвращает номер блока по номеру сектора.
 *
 * @param[in] nSectorIdx  Номер сектора.
 *
 * @return Номер блока.
 */
inline size_t GetMfBlockBySector(size_t nSectorIdx) {
    return (nSectorIdx < 32) ? (nSectorIdx * 4) : (128 + ((nSectorIdx - 32) * 16));
}

/**
 * @brief Возвращает количество блоков в секторе.
 *
 * @param[in] nSectorIdx  Номер сектора.
 *
 * @return Количество блоков в секторе.
 */
inline size_t GetMfSectorBlockCount(size_t nSectorIdx) {
    return (nSectorIdx < 32) ? 4 : 16;
}

/**
 * @brief Возвращает информацию о блоке карты Mifare Classic/Plus.
 *
 * @param[in]  nBlockIdx     Номер блока.
 * @param[out] nSectorIdx    Номер сектора.
 * @param[out] nSBlockIdx    Номер блока в секторе.
 * @param[out] nSBlockCount  Количество блоков в секторе.
 */
void GetMfBlockInfo(size_t nBlockIdx, size_t& nSectorIdx, size_t& nSBlockIdx, size_t& nSBlockCount);

/**
 * @brief Преобразует номер карты в строку.
 *
 *  @param[in] nType Тип карты.
 *  @param[in] rUID  Номер карты.
 *
 *  @return Строка с номером карты.
 */
std::string CardUIDToStr(ilr_card_type nType, const CCardUID& rUID);

/**
 * @brief Возвращает биты доступа сектора Mifare Classic/Plus.
 *
 * @param[in] rTrailerData  Данные блока-прицепа Mifare.
 */
uint32_t GetMfAccessBits(const ilr_mf_block_data& rTrailerData);

/**
 * @brief Устанавливает биты доступа сектора Mifare Classic/Plus.
 *
 * @param[in]  nSectorAccess  Биты доступа сектора Mifare Classic/Plus.
 * @param[out] rTrailerData   Данные блока-прицепа Mifare.
 */
void SetMfAccessBits(uint32_t nSectorAccess, ilr_mf_block_data& rTrailerData);

/**
 * @brief Возвращает номер области сектора Mifare Classic/Plus по номеру блока.
 *
 * @param[in] nBlockIdx  Номер блока (0..255).
 *
 * @return Номер области (0..3).
 */
size_t GetMfAreaByBlockIdx(size_t nBlockIdx);

/**
 * @brief Возвращает биты доступа (3 бита) для области сектора Mifare Classic/Plus.
 *
 * @param[in]  nSectorAccess  Биты доступа сектора Mifare Classic/Plus.
 * @param[in]  nArea          Номер области 0..3 (=0..2 область данных, =3 прицеп сектора).
 * @param[out] nAreaAccess    Биты доступа (3 бита) для области сектора Mifare.
 */
bool TryGetMfAreaAccess(uint32_t nSectorAccess, size_t nArea, uint32_t& nAreaAccess);

/**
 * @brief Устанавливает биты доступа для области сектора Mifare (3 бита).
 *
 * @param[in]  nArea          Номер области 0..3 (=0..2 область данных, =3 прицеп сектора).
 * @param[in]  nAreaAccess    Биты доступа (3 бита) для области сектора Mifare.
 * @param[out] nSectorAccess  Биты доступа сектора Mifare Classic/Plus.
 */
void SetMfAreaAccess(size_t nArea, uint32_t nAreaAccess, uint32_t& nSectorAccess);

/**
 * @brief Извлекает ключ аутентификации из блока-прицепа Mifare Classic.
 *
 * @param[in] rTrailerData Данные блока-прицепа Mifare Classic.
 * @param[in] fKeyB        =true извлечь ключ Б, иначе - ключ А.
 *
 * @remark Из памяти карты Mifare нельзя прочитать ключ аутентификации, исключением является ключ Б
 * когда он не может быть использован для авторизации.
 */
inline CMifareClassicKey GetMfClassicKey(const ilr_mf_block_data& rTrailerData, bool fKeyB) {
    return CMifareClassicKey(fKeyB ? &rTrailerData.a[10] : rTrailerData.a, 6);
}

/**
 * @brief Устанавливает ключ аутентификации в блок-прицеп Mifare Classic.
 *
 * @param[out] rTrailerData Данные блока-прицепа Mifare Classic.
 * @param[in]  fKeyB        =true установить ключ Б, иначе - ключ А.
 * @param[in]  nKey         Ключ аутентификации Mifare Classic.
 */
inline void SetMfClassicKey(ilr_mf_block_data& rTrailerData, bool fKeyB,
                            const ilr_mf_classic_key& nKey) {
    memcpy(fKeyB ? &rTrailerData.a[10] : rTrailerData.a, &nKey, 6);
}

/**
 * @brief Пытается получить значение из блока-значения.
 *
 * @param[in]  rData   Данные блока-значения.
 * @param[out] nValue  Значение.
 *
 * @return True, значение успешно получено.
 */
bool TryGetMfValue(const ilr_mf_block_data& rData, int& nValue);

/**
 * @brief Устанавливает значение для блока-значения.
 *
 * @param[out] rData   Данные блока-значения.
 * @param[in]  nValue  Значение.
 */
void SetMfValue(ilr_mf_block_data& rData, int nValue);

/**
 * @brief Пытается получить адрес из блока-значения.
 *
 * @param[in]  rData     Данные блока-значения.
 * @param[out] nAddress  Адрес.
 */
bool TryGetMfValueAddress(const ilr_mf_block_data& rData, uint8_t& nAddress);

/**
 * @brief Устанавливает адрес для блока-значения.
 *
 * @param[out] rData     Данные блока-значения.
 * @param[in]  nAddress  Адрес.
 */
void SetMfValueAddress(ilr_mf_block_data& rData, uint8_t nAddress);

};  // namespace ilr
