#include <stdio.h>

#include <iomanip>  // для std::put_time
#include <iostream>

#include "ilr_cpp_helpers.h"
#include "ilreaders.h"

// #define ILR_LOG  // Раскомментируйте, чтобы включить отладочные сообщения
#define ILR_LOG_FILE  // Писать лог в файл

using namespace ilr;

#ifdef ILR_LOG
const char kLogLevelChars[] = {'-', 'E', 'W', 'I', 'D'};
const char kLogFileName[] = "ilreaders.log";  // Путь к лог файлу

void ILR_CALL LogCallback(ilr_log_level level, const char* pContext, const char* pMessage, void*) {
#ifdef ILR_LOG_FILE  // Запись в файл
    std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::app);
    auto& out = file;
#else  // иначе в консоль
    auto& out = std::cout;
#endif
    auto t = std::time(nullptr);
    auto tmb = std::localtime(&t);
    out << std::put_time(tmb, "%d-%m-%Y %H:%M:%S") << " [" << kLogLevelChars[level] << ' '
        << pContext << "] " << pMessage << std::endl;
}
#endif

void DoReadUltralight(CReader& oReader) {
    size_t nRead = 0;
    try {
        uint aPages[16];  // 16 страниц по 4 байта
        std::cout << "Чтение данных карты..." << std::endl;
        auto tStartTime = now();
        oReader.ReadMfUltralight(0, aPages, std::size(aPages), &nRead);
        std::cout << "Прочитано за " << since(tStartTime).count() << " мс" << std::endl;

        const char* kLocked[2] = {"", "Заблокировано"};
        uint nPageData, n;
        uint8_t* pBytes;
        std::cout << "Страница. Байты 0..3" << std::endl;
        for (size_t i = 0; i < 16; i++) {  // цикл по страницам
            nPageData = aPages[i];
            pBytes = (uint8_t*)&nPageData;
            std::cout << std::dec << std::setw(4) << i << ". " << std::hex << std::setw(3)
                      << (uint)pBytes[0] << std::setw(3) << (uint)pBytes[1] << std::setw(3)
                      << (uint)pBytes[2] << std::setw(3) << (uint)pBytes[3];
            switch (i) {
            case 0:  // Serial Number
            case 1:
                std::cout << " Серийный номер" << std::endl;
                break;

            case 2:  // Internal / Lock
                {
                    std::cout << " Внутреннее / Блокировка" << std::endl;
                    n = (nPageData >> 16);
                    std::cout << "Lock0[" << std::hex << std::setw(2) << (n & 0xff)
                              << "] BOTP:" << std::dec << (n & 1) << ", BL9-4:" << ((n >> 1) & 1)
                              << ", BL15-10:" << ((n >> 2) & 1) << ", OTP:" << ((n >> 3) & 1)
                              << ", L4:" << ((n >> 4) & 1) << ", L5:" << ((n >> 5) & 1)
                              << ", L6:" << ((n >> 6) & 1) << ", L7:" << ((n >> 7) & 1)
                              << std::endl;
                    printf(
                        "Lock1[%.2X] L8:%d, L9:%d, L10:%d, L11:%d, L12:%d, L13:%d, L14:%d, "
                        "L15:%d\n",
                        n >> 8, (n >> 8) & 1, (n >> 9) & 1, (n >> 10) & 1, (n >> 11) & 1,
                        (n >> 12) & 1, (n >> 13) & 1, (n >> 14) & 1, (n >> 15) & 1);
                    break;
                }

            case 3:  // OTP
                {
                    std::string s("00000000 00000000 00000000 00000000");
                    n = nPageData;
                    for (size_t j = 0; j < s.length(); ++j) {
                        if (' ' == s[j])
                            continue;
                        if (n & 1)
                            s[j] = '1';
                        n >>= 1;
                    }
                    std::cout << " OTP (" << std::dec << nPageData << ") " << s << ' '
                              << kLocked[GET_BIT(aPages[2] >> 16, 3)] << std::endl;
                    break;
                }

            default:
                std::cout << " Данные (" << std::dec << nPageData << ") "
                          << kLocked[GET_BIT(aPages[2] >> 16, i)] << std::endl;
                break;
            }
        }
        std::cout << "-----" << std::endl;
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
        std::cout << "Прочитано " << std::dec << nRead << " из 16 страниц" << std::endl;
    }
}

void DoWriteUltralight(CReader& oReader) {
    try {
        std::cout << "Введите номер страницы (10-тичное), байты 0 1 2 3 (16-ричные):" << std::endl;

        size_t nPageN;
        uint a[4];
        std::cin >> std::dec >> nPageN >> std::hex >> a[0] >> a[1] >> a[2] >> a[3];
        std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
        if (std::cin.fail()) {
            std::cin.clear();
            std::cout << "Неправильный ввод" << std::endl;
            return;
        }

        uint nPageData;
        for (size_t i = 0; i < std::size(a); ++i)
            ((uint8_t*)&nPageData)[i] = (uint8_t)a[i];

        std::cout << "Запись..." << std::endl;
        auto tStartTime = now();
        oReader.WriteMfUltralight(nPageN, &nPageData, 1);
        std::cout << "Записано за " << since(tStartTime).count() << " мс" << std::endl;
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
}

bool DoConnectTo(CILR& oILR, CReader& oReader) {
    CReaderSearch oSearch(oILR.GetSearch());
    {
        // Включаем поиск считывателя Z-2 MF CCID (по умолчанию выключен)
        ilr_search_options rOptions;
        oSearch.GetOptions(rOptions);
        rOptions.nReaderTypes |= ILR_READER_CCID;
        oSearch.SetOptions(rOptions);
    }
    // Ищем считыватели
    std::cout << "Поиск считывателей..." << std::endl;
    oSearch.Scan();
    auto nCount = oSearch.GetReaderCount();
    if (0 == nCount) {
        std::cout << "Считыватель не найден" << std::endl;
        return false;
    }
    std::cout << "Найдено " << nCount << ':' << std::endl;
    bool fFound = false;
    ilr_reader_info rInfo;
    for (size_t i = 0; i < nCount; i++) {
        oSearch.GetReaderInfo(i, rInfo);
        // Если порт занят, пропускаем его
        if (*rInfo.pszConnect != '\0')
            continue;
        if (CReader::GetSupportedRewrittenCardTypes(rInfo.nModel, rInfo.nFwVersion) &
            ILR_RWCT_F_MF_ULTRALIGHT) {
            fFound = true;
            break;
        }
    }
    if (!fFound) {
        std::cout << "Считыватель не поддерживает чтение/запись карт Mifare Ultralight"
                  << std::endl;
        return false;
    }

    // Подключаемся к считывателю
    std::cout << "Подключение к считывателю [" << kPortTypeNames[rInfo.nPortType] << ": "
              << rInfo.pszPortName << "]..." << std::endl;
    oReader = oILR.GetReader(rInfo.nPortType, rInfo.pszPortName);
    // Отключаем авто поиск карт
    oReader.SetAutoScan(false);
    if (rInfo.nModel != ILR_READER_MODEL_UNKNOWN) {
        ilr_reader_options rOptions;
        oReader.GetOptions(rOptions);
        rOptions.nConnectModel = rInfo.nModel;
        oReader.SetOptions(rOptions);
    }
    oReader.Connect();
    oReader.GetReaderInfo(rInfo);
    std::stringstream ss;
    if (rInfo.nModel != ILR_READER_MODEL_UNKNOWN)
        ss << kReaderModelNames[rInfo.nModel];
    if (rInfo.nSn != -1)
        ss << " с/н:" << rInfo.nSn;
    if (rInfo.nFwVersion != 0)
        ss << " прошивка:" << ReaderVersionToStr(rInfo.nFwVersion);
    if (rInfo.nFwBuildDate != 0)
        ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
    std::cout << "Считыватель успешно подключён [" << ss.str() << ']' << std::endl;
    return true;
}

int main() {
    try {
#ifdef ILR_LOG
#ifdef ILR_LOG_FILE
        // Очищаем лог файл
        std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::trunc);
        file.close();
#endif
        CILR::SetLogCallback(LogCallback);
        CILR::SetLogLevel(ILR_LOG_LEVEL_DEBUG);
#endif

        // Ищем считыватель и подключаемся к нему
        CILR oILR;
        CReader oReader;
        if (!DoConnectTo(oILR, oReader))
            return 0;

        // Основной цикл программы
        while (true) {
            std::cout << "Поиск карты Mifare Ultralight..." << std::endl;
            oReader.Scan();
            ilr_card_info rCI;
            oReader.GetCardInfo(rCI);
            bool fCardFound = (ILR_CARD_MF_ULTRALIGHT == rCI.nType);
            // Если карта Mifare Ultralight найдена,
            if (fCardFound)
                std::cout << kCardTypeNames[rCI.nType] << ' ' << CardUIDToStr(rCI.nType, rCI.rUID)
                          << std::endl;
            else  // Mifare Ultralight не найдена
                std::cout << "Карта не найдена" << std::endl;

            std::cout << "-----" << std::endl;
            std::cout << "Введите номер команды:" << std::endl;
            std::cout << "1 - Искать снова" << std::endl;
            if (fCardFound) {
                std::cout << "2 - Прочитать данные из карты" << std::endl;
                std::cout << "3 - Записать данные на карту..." << std::endl;
            }
            std::cout << "0 - Выйти из программы" << std::endl;

            int nCommand;
            std::cin >> nCommand;
            if (std::cin.fail()) {
                std::cin.clear();
                nCommand = -1;
            }
            std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');

            switch (nCommand) {
            case 1:
                break;

            case 2:
                if (fCardFound)
                    DoReadUltralight(oReader);
                break;

            case 3:
                if (fCardFound)
                    DoWriteUltralight(oReader);
                break;

            case 0:
                return 0;

            default:
                std::cout << "Неправильный ввод" << std::endl;
                break;
            }
        }
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
    return 0;
}
