#include "CMfUltralightDialog.h"

#include "CAppSettings.h"
#include "utils.h"

CMfUltralightDialog::CMfUltralightDialog() :
    m_VBox(Gtk::Orientation::VERTICAL),
    m_RightBox(Gtk::Orientation::VERTICAL),
    m_SerialButton(Gdk::RGBA(0.65, 0.79, 0.94)),
    m_SerialLabel("Серийный номер"),
    m_LockButton(Gdk::RGBA(1.0, 1.0, 0.6)),
    m_LockLabel("Lock / OTP"),
    m_LockRoButton(Gdk::RGBA(0.87, 0.87, 0.74)),
    m_LockRoLabel("Lock / OTP (только чтение)"),
    m_UserDataButton(Gdk::RGBA(0.66, 0.94, 0.66)),
    m_UserDataLabel("Данные пользователя"),
    m_UserDataRoButton(Gdk::RGBA(0.74, 0.86, 0.74)),
    m_UserDataRoLabel("Данные (только чтение)"),
    m_FormatFrame("Формат данных"),
    m_FormatBox(Gtk::Orientation::VERTICAL),
    m_HexCheckButton("Шестнадцатеричный"),
    m_DecCheckButton("Десятичный"),
    m_OctCheckButton("Восьмеричный"),
    m_BinCheckButton("Двоичный"),
    m_ReadButton("Читать"),
    m_WriteButton("Записать") {
    set_destroy_with_parent(true);

    set_title("Mifare Ultralight");
    set_default_size(600, 460);

    m_VBox.set_expand();
    set_child(m_VBox);

    m_VBox.append(m_DataBox);
    m_VBox.append(m_BottomBox);

    m_DataBox.append(m_DataScrolledWindow);
    m_DataBox.append(m_RightBox);

    m_DataScrolledWindow.set_policy(Gtk::PolicyType::AUTOMATIC, Gtk::PolicyType::AUTOMATIC);
    m_DataScrolledWindow.set_expand();
    m_DataScrolledWindow.set_child(m_DataGrid);

    m_SerialButton.set_rgba(g_AppSet.m_MfUlSerialColor);
    m_LockButton.set_rgba(g_AppSet.m_MfUlLockColor);
    m_LockRoButton.set_rgba(g_AppSet.m_MfUlLockRoColor);
    m_UserDataButton.set_rgba(g_AppSet.m_MfUlUserDataColor);
    m_UserDataRoButton.set_rgba(g_AppSet.m_MfUlUserDataRoColor);

    m_DataGrid.set_data_pointer(&m_aNewPages[0][0], sizeof(m_aNewPages));
    m_DataGrid.set_old_data_pointer(&m_aOldPages[0][0], sizeof(m_aOldPages));
    m_DataGrid.set_block_size(4);
    const char* kColTitles[1] = {"Страница"};
    m_DataGrid.set_fix_col_titles(kColTitles, std::size(kColTitles));
    // Серийный номер
    m_DataGrid.add_or_set_byte_attrs(0, 8, CCardDataGrid::BYTE_FORMAT_UNDEF, Gdk::RGBA(0, 0, 0),
                                     m_SerialButton.get_rgba());
    // Lock 0, Lock 1
    for (size_t i = 10; i < 12; i++)
        m_DataGrid.add_or_set_byte_attrs(i, 1, CCardDataGrid::BYTE_FORMAT_BIN, Gdk::RGBA(0, 0, 0),
                                         m_LockButton.get_rgba());

    // OTP
    for (size_t i = 12; i < 16; i++)
        m_DataGrid.add_or_set_byte_attrs(i, 1, CCardDataGrid::BYTE_FORMAT_UNDEF, Gdk::RGBA(0, 0, 0),
                                         m_LockButton.get_rgba());

    // Data
    for (size_t i = 16; i < 64; i += 4)
        m_DataGrid.add_or_set_byte_attrs(i, 4, CCardDataGrid::BYTE_FORMAT_UNDEF, Gdk::RGBA(0, 0, 0),
                                         m_UserDataButton.get_rgba());

    m_DataGrid.set_range_checkbox(2, 14, true, false, true);
    m_DataGrid.update();
    m_DataGrid.signal_get_byte_access().connect(
        sigc::mem_fun(*this, &CMfUltralightDialog::on_data_get_byte_access));
    m_DataGrid.signal_validate_byte().connect(
        sigc::mem_fun(*this, &CMfUltralightDialog::on_data_validate_byte));
    m_DataGrid.signal_checkbox_toggle().connect(
        sigc::mem_fun(*this, &CMfUltralightDialog::on_checkbox_toggle));

    m_RightBox.set_margin(5);
    m_RightBox.set_spacing(5);
    m_RightBox.append(m_LegendFrame);
    m_RightBox.append(m_FormatFrame);

    m_LegendFrame.set_child(m_LegendGrid);
    m_LegendGrid.set_margin(5);
    m_LegendGrid.set_column_spacing(5);
    m_LegendGrid.set_row_spacing(5);
    m_LegendGrid.attach(m_SerialButton, 0, 0);
    m_LegendGrid.attach(m_SerialLabel, 1, 0);
    m_LegendGrid.attach(m_LockButton, 0, 1);
    m_LegendGrid.attach(m_LockLabel, 1, 1);
    m_LegendGrid.attach(m_LockRoButton, 0, 2);
    m_LegendGrid.attach(m_LockRoLabel, 1, 2);
    m_LegendGrid.attach(m_UserDataButton, 0, 3);
    m_LegendGrid.attach(m_UserDataLabel, 1, 3);
    m_LegendGrid.attach(m_UserDataRoButton, 0, 4);
    m_LegendGrid.attach(m_UserDataRoLabel, 1, 4);
    m_SerialLabel.set_halign(Gtk::Align::START);
    m_SerialLabel.set_mnemonic_widget(m_SerialButton);
    m_SerialButton.signal_color_set().connect(
        sigc::mem_fun(*this, &CMfUltralightDialog::on_serial_color_set));
    m_LockLabel.set_halign(Gtk::Align::START);
    m_LockLabel.set_mnemonic_widget(m_LockButton);
    m_LockButton.signal_color_set().connect(
        sigc::mem_fun(*this, &CMfUltralightDialog::on_lock_color_set));
    m_LockRoLabel.set_halign(Gtk::Align::START);
    m_LockRoLabel.set_mnemonic_widget(m_LockRoButton);
    m_LockRoButton.signal_color_set().connect(
        sigc::mem_fun(*this, &CMfUltralightDialog::on_lock_ro_color_set));
    m_UserDataLabel.set_halign(Gtk::Align::START);
    m_UserDataLabel.set_mnemonic_widget(m_UserDataButton);
    m_UserDataButton.signal_color_set().connect(
        sigc::mem_fun(*this, &CMfUltralightDialog::on_user_data_color_set));
    m_UserDataRoLabel.set_halign(Gtk::Align::START);
    m_UserDataRoLabel.set_mnemonic_widget(m_UserDataRoButton);
    m_UserDataRoButton.signal_color_set().connect(
        sigc::mem_fun(*this, &CMfUltralightDialog::on_user_data_ro_color_set));

    m_FormatFrame.set_child(m_FormatBox);
    m_FormatBox.set_margin(5);
    m_FormatBox.set_spacing(5);
    m_FormatBox.append(m_HexCheckButton);
    m_FormatBox.append(m_DecCheckButton);
    m_FormatBox.append(m_OctCheckButton);
    m_FormatBox.append(m_BinCheckButton);

    m_DecCheckButton.set_group(m_HexCheckButton);
    m_OctCheckButton.set_group(m_DecCheckButton);
    m_BinCheckButton.set_group(m_OctCheckButton);
    m_HexCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CMfUltralightDialog::on_hex_toggled));
    m_DecCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CMfUltralightDialog::on_dec_toggled));
    m_OctCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CMfUltralightDialog::on_oct_toggled));
    m_BinCheckButton.signal_toggled().connect(
        sigc::mem_fun(*this, &CMfUltralightDialog::on_bin_toggled));

    m_BottomBox.set_margin(5);
    m_BottomBox.set_spacing(5);
    m_BottomBox.set_hexpand();
    m_BottomBox.append(m_ReadButton);
    m_BottomBox.append(m_WriteButton);

    m_ReadButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CMfUltralightDialog::on_button_read));
    m_WriteButton.signal_clicked().connect(
        sigc::mem_fun(*this, &CMfUltralightDialog::on_button_write));

    signal_hide().connect(sigc::mem_fun(*this, &CMfUltralightDialog::on_hide));
}

CMfUltralightDialog::~CMfUltralightDialog() {
}

void CMfUltralightDialog::Init(const ilr::CReader& oReader) {
    m_oReader = oReader.Clone();
    ilr_card_info rInfo;
    m_oReader.GetCardInfo(rInfo);
    set_title(
        Glib::ustring::format("Mifare Ultralight ", ilr::CardUIDToStr(rInfo.nType, rInfo.rUID)));

    m_DataGrid.set_byte_format(g_AppSet.m_nDataFormat);
    switch (m_DataGrid.get_byte_format()) {
    case byte_format::BYTE_FORMAT_HEX:
        m_HexCheckButton.set_active();
        break;

    case byte_format::BYTE_FORMAT_DEC:
        m_DecCheckButton.set_active();
        break;

    case byte_format::BYTE_FORMAT_OCT:
        m_OctCheckButton.set_active();
        break;

    case byte_format::BYTE_FORMAT_BIN:
        m_BinCheckButton.set_active();
        break;
    }
    ReadCardData();
}

uint CMfUltralightDialog::on_data_get_byte_access(uint nAddress) {
    return GetByteAccess(nAddress);
}

void CMfUltralightDialog::on_data_validate_byte(uint nAddress, uint8_t& b) {
    auto nPageIdx = (nAddress / 4);
    auto nByteIdx = (nAddress % 4);
    uint nOldValue = m_aOldPages[nPageIdx][nByteIdx];
    uint nValue = b;
    auto fUpdateColors = false;
    auto fUpdateCheckboxes = false;
    if (nAddress < 10)
        nValue = nOldValue;
    else if (nAddress < 11) {  // Lock0
        // Если какой-либо бит в Lock-байте был установлен в 1, то он больше не может быть сброшен
        nValue |= nOldValue;
        uint nChanges = nOldValue ^ b;
        // Если установлен бит BL OTP и изменён бит L OTP,
        if ((nOldValue & 1) && (nChanges & 8))
            // Возвращаем старое значение бита L OTP
            SET_BIT(nValue, 3, nOldValue & 8);
        // Если (установлен бит BL 9-4) и (изменены биты L 4 .. L 7)
        if ((nOldValue & 2) && (nChanges & 0xF0))
            // Возвращаем старые значения бит L 4 .. L 7
            nValue = (nValue & ~0xF0) | (nOldValue & 0xF0);
        fUpdateColors = true;
        fUpdateCheckboxes = true;
    }
    else if (nAddress < 12) {  // Lock1
        // Если какой-либо бит в Lock-байте был установлен в 1, то он больше не может быть сброшен
        nValue |= nOldValue;
        uint nChanges = nOldValue ^ b;
        // Если (установлен бит BL 9-4) и (изменены биты L 0 .. L 1)
        if ((m_aOldPages[2][2] & 2) && (nChanges & 3))
            // Возвращаем старые значения бит L 0 .. L 1
            nValue = (nValue & ~3) | (nOldValue & 3);
        // Если (установлен бит BL 15-10) и (изменены биты L 2 .. L 7)
        if ((m_aOldPages[2][2] & 4) && (nChanges & 0xFC))
            // Возвращаем старые значения бит L 2 .. L 7
            nValue = (nValue & ~0xFC) | (nOldValue & 0xFC);
        fUpdateColors = true;
        fUpdateCheckboxes = true;
    }
    else if (nAddress < 16) {  // OTP
        // При установке 1 в каком-либо бите из области OTP, его становится невозможно сбросить в
        // ноль
        nValue |= nOldValue;
        // Если (OTP область заблокирована) и (значение изменилось),
        if ((m_aOldPages[2][2] & 8) && (nOldValue != b))
            nValue = nOldValue;
        fUpdateColors = true;
    }
    else {  // Data
        // Если (страница заблокирована) и (значение изменилось),
        if (GET_BIT(*(const uint16_t*)&m_aOldPages[2][2], nPageIdx) && (nOldValue != b))
            nValue = nOldValue;
    }
    b = (uint8_t)nValue;
    if (m_aNewPages[nPageIdx][nByteIdx] != b) {
        m_aNewPages[nPageIdx][nByteIdx] = b;
        if (fUpdateColors)
            UpdateDataGridColors();
        if (fUpdateCheckboxes)
            UpdateDataCheckboxes();
    }
}

void CMfUltralightDialog::on_checkbox_toggle(uint nBlockIdx) {
    auto fCheck = !m_DataGrid.get_checkbox_active(nBlockIdx);
    if (CanCheckPage(nBlockIdx))
        SetPageChecked(nBlockIdx, fCheck);
}

void CMfUltralightDialog::on_serial_color_set() {
    g_AppSet.SetMfUlSerialColor(m_SerialButton.get_rgba());
    m_DataGrid.add_or_set_byte_attrs(0, 8, CCardDataGrid::BYTE_FORMAT_UNDEF, Gdk::RGBA(0, 0, 0),
                                     m_SerialButton.get_rgba());
}

void CMfUltralightDialog::on_lock_color_set() {
    g_AppSet.SetMfUlLockColor(m_LockButton.get_rgba());
    UpdateDataGridColors();
}

void CMfUltralightDialog::on_lock_ro_color_set() {
    g_AppSet.SetMfUlLockRoColor(m_LockRoButton.get_rgba());
    UpdateDataGridColors();
}

void CMfUltralightDialog::on_user_data_color_set() {
    g_AppSet.SetMfUlUserDataColor(m_UserDataButton.get_rgba());
    UpdateDataGridColors();
}

void CMfUltralightDialog::on_user_data_ro_color_set() {
    g_AppSet.SetMfUlUserDataRoColor(m_UserDataRoButton.get_rgba());
    UpdateDataGridColors();
}

void CMfUltralightDialog::on_hex_toggled() {
    m_DataGrid.set_byte_format(CCardDataGrid::BYTE_FORMAT_HEX);
}

void CMfUltralightDialog::on_dec_toggled() {
    m_DataGrid.set_byte_format(CCardDataGrid::BYTE_FORMAT_DEC);
}

void CMfUltralightDialog::on_oct_toggled() {
    m_DataGrid.set_byte_format(CCardDataGrid::BYTE_FORMAT_OCT);
}

void CMfUltralightDialog::on_bin_toggled() {
    m_DataGrid.set_byte_format(CCardDataGrid::BYTE_FORMAT_BIN);
}

void CMfUltralightDialog::on_button_read() {
    ReadCardData();
}

void CMfUltralightDialog::on_button_write() {
    try {
        for (size_t i = 0; i < std::size(m_aNewPages); i++) {
            if (memcmp(&m_aNewPages[i][0], &m_aOldPages[i][0], 4) != 0) {
                m_oReader.WriteMfUltralight(i, (uint32_t*)&m_aNewPages[i][0], 1);
                memcpy(&m_aOldPages[i], &m_aNewPages[i], 4);
            }
        }
    }
    catch (const std::exception& e) {
        // std::cerr << e.what() << '\n';
        ShowMessage(e.what(), Gtk::MessageType::ERROR);
    }
    UpdateDataGridColors();
    UpdateDataCheckboxes();
}

void CMfUltralightDialog::on_dialog_response(int) {
    m_refDialog = nullptr;
}

void CMfUltralightDialog::on_hide() {
    g_AppSet.SetDataFormat(m_DataGrid.get_byte_format());
    m_oReader.Close();
}

void CMfUltralightDialog::ReadCardData() {
    try {
        m_oReader.ReadMfUltralight(0, (uint32_t*)&m_aOldPages[0][0], std::size(m_aOldPages));
    }
    catch (const std::exception& e) {
        // std::cerr << e.what() << '\n';
        ShowMessage(e.what(), Gtk::MessageType::ERROR);
    }
    memcpy(m_aNewPages, m_aOldPages, sizeof(m_aNewPages));
    UpdateDataGridColors();
    UpdateDataCheckboxes();
}

void CMfUltralightDialog::UpdateDataGridColors() {
    auto nLock = *(uint16_t*)&m_aOldPages[2][2];

    // Lock 0
    auto fLocked = (((nLock & 1) || (nLock & 8)) && ((nLock & 2) || ((nLock & 0xF0) == 0xF0)));
    m_DataGrid.add_or_set_byte_attrs(10, 1, CCardDataGrid::BYTE_FORMAT_BIN, Gdk::RGBA(0, 0, 0),
                                     fLocked ? m_LockRoButton.get_rgba() : m_LockButton.get_rgba(),
                                     false);

    // Lock 1
    fLocked = ((nLock & 2) || ((nLock & 3) == 3)) && ((nLock & 4) || ((nLock & 0xFC) == 0xFC));
    m_DataGrid.add_or_set_byte_attrs(11, 1, CCardDataGrid::BYTE_FORMAT_BIN, Gdk::RGBA(0, 0, 0),
                                     fLocked ? m_LockRoButton.get_rgba() : m_LockButton.get_rgba(),
                                     false);

    // OTP
    uint nPageIdx = 3;
    uint nByteIdx = 0;
    fLocked = (nLock & 8) != 0;
    for (size_t i = 12; i < 16; i++, nByteIdx++)
        m_DataGrid.add_or_set_byte_attrs(i, 1, CCardDataGrid::BYTE_FORMAT_UNDEF, Gdk::RGBA(0, 0, 0),
                                         (fLocked || (0xff == m_aOldPages[nPageIdx][nByteIdx])) ?
                                             m_LockRoButton.get_rgba() :
                                             m_LockButton.get_rgba(),
                                         false);

    // Data
    nPageIdx = 4;
    for (size_t i = 16; i < 64; i += 4, nPageIdx++)
        m_DataGrid.add_or_set_byte_attrs(
            i, 4, CCardDataGrid::BYTE_FORMAT_UNDEF, Gdk::RGBA(0, 0, 0),
            GET_BIT(nLock, nPageIdx) ? m_UserDataRoButton.get_rgba() : m_UserDataButton.get_rgba(),
            false);

    m_DataGrid.redraw();
}

void CMfUltralightDialog::UpdateDataCheckboxes() {
    auto& nLock = *(const uint16_t*)&m_aNewPages[2][2];
    auto& nOldLock = *(const uint16_t*)&m_aOldPages[2][2];
    bool fActive, fSensitive;
    for (size_t i = 2; i < std::size(m_aNewPages); i++) {
        if (2 == i) {
            fActive = ((nLock & 7) == 7);
            fSensitive = ((nOldLock & 7) != 7);
        }
        else {
            fActive = GET_BIT(nLock, i);
            fSensitive = !GET_BIT(nOldLock, i);
        }
        m_DataGrid.set_checkbox_active(i, fActive);
        m_DataGrid.set_checkbox_sensitive(i, fSensitive);
    }
}

void CMfUltralightDialog::SetPageChecked(int nPageIdx, bool fChecked) {
    auto& nLock = *(uint16_t*)&m_aNewPages[2][2];
    uint nNewLock = nLock;
    if (2 == nPageIdx) {
        if (fChecked)
            nNewLock |= 7;
        else
            nNewLock = (nNewLock & ~7) | (m_aOldPages[2][2] & 7);
    }
    else if (nPageIdx > 2)
        SET_BIT(nNewLock, nPageIdx, fChecked);

    if (nLock != nNewLock) {
        nLock = (uint16_t)nNewLock;
        m_DataGrid.set_checkbox_active(nPageIdx, fChecked);
        m_DataGrid.set_checkbox_sensitive(nPageIdx, CanCheckPage(nPageIdx));
    }
}

bool CMfUltralightDialog::CanCheckPage(int nPageIdx) const {
    auto nLock = *(const uint16_t*)&m_aOldPages[2][2];
    switch (nPageIdx) {
    case 2:  // Internal / Lock
        return (nLock & 7) != 7;

    case 3:  // OTP
        return !GET_BIT(nLock, 0);

    case 4:  // Data
    case 5:
    case 6:
    case 7:
    case 8:
    case 9:
        return !GET_BIT(nLock, 1);

    case 10:  // Data
    case 11:
    case 12:
    case 13:
    case 14:
    case 15:
        return !GET_BIT(nLock, 2);
    }
    return false;
}

uint CMfUltralightDialog::GetByteAccess(uint nAddress) const {
    auto nLock = *(const uint16_t*)&m_aOldPages[2][2];
    if (nAddress < 10)  // Serial Number, BCC0, BCC1, Internal
        return 0;
    if (nAddress < 12)  // Lock0, Lock1
        return nLock ^ 0xffff;
    if (nAddress < 16)  // OTP
        return (nLock & 8) ? 0 : (m_aOldPages[3][nAddress % 4] ^ 0xff);
    return GET_BIT(nLock, nAddress / 4) ? 0 : 0xff;
}

void CMfUltralightDialog::ShowMessage(const std::string& sMessage, Gtk::MessageType nType) {
    if (nullptr == m_refDialog)
        m_refDialog.reset(new Gtk::MessageDialog("Демо", false, nType));
    m_refDialog->set_transient_for(*this);
    m_refDialog->set_message(sMessage);
    m_refDialog->show();
    m_refDialog->signal_response().connect(
        sigc::bind(sigc::mem_fun(*this, &CMfUltralightDialog::on_dialog_response)));
}
