#include <stdio.h>

#include <fstream>
#include <iomanip>  // для std::put_time
#include <iostream>

#include "ilr_cpp_helpers.h"
#include "ilreaders.h"

// #define ILR_LOG  // Раскомментируйте, чтобы включить отладочные сообщения
#define ILR_LOG_FILE  // Писать лог в файл

using namespace ilr;

#ifdef ILR_LOG
const char kLogLevelChars[] = {'-', 'E', 'W', 'I', 'D'};
const char kLogFileName[] = "ilreaders.log";  // Путь к лог файлу

void ILR_CALL LogCallback(ilr_log_level level, const char* pContext, const char* pMessage, void*) {
#ifdef ILR_LOG_FILE  // Запись в файл
    std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::app);
    auto& out = file;
#else  // иначе в консоль
    auto& out = std::cout;
#endif
    auto t = std::time(nullptr);
    auto tmb = std::localtime(&t);
    out << std::put_time(tmb, "%d-%m-%Y %H:%M:%S") << " [" << kLogLevelChars[level] << ' '
        << pContext << "] " << pMessage << std::endl;
}
#endif

void ILR_CALL MessageCallback(ilr_search_msg nMsg, const void* pMsgData, void*) {
    try {
        switch (nMsg) {
        case ILR_SEARCH_MSG_READER_FOUND:  // Считыватель найден
        case ILR_SEARCH_MSG_READER_LOST:   // Считыватель потерян
            {
                const ilr_reader_info* pInfo = (const ilr_reader_info*)pMsgData;
                std::stringstream ss;
                if (pInfo->nModel != ILR_READER_MODEL_UNKNOWN)
                    ss << kReaderModelNames[pInfo->nModel];
                if (pInfo->nSn != -1)
                    ss << " с/н:" << pInfo->nSn;
                if (pInfo->nFwVersion != 0)
                    ss << " прошивка:" << ReaderVersionToStr(pInfo->nFwVersion);
                if (pInfo->nFwBuildDate != 0)
                    ss << " сборка " << TimeToStr(pInfo->nFwBuildDate);

                if (ILR_SEARCH_MSG_READER_FOUND == nMsg)
                    std::cout << "{!} Считыватель найден (" << pInfo->pszPortName
                              << "): " << ss.str() << std::endl;
                else
                    std::cout << "{!} Считыватель потерян (" << pInfo->pszPortName
                              << "): " << ss.str() << std::endl;
                break;
            }

        case ILR_SEARCH_MSG_LIST_CHANGED:  // Список считывателей изменился
            std::cout << "{!} Список считыватель изменился" << std::endl;
            break;

        default:
            break;
        }
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
}

int main() {
    try {
#ifdef ILR_LOG
#ifdef ILR_LOG_FILE
        // Очищаем лог файл
        std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::trunc);
        file.close();
#endif
        CILR::SetLogCallback(LogCallback);
        CILR::SetLogLevel(ILR_LOG_LEVEL_DEBUG);
#endif

        CILR oILR;
        CReaderSearch oSearch(oILR.GetSearch());
#if 0
        // Выбираем типы считывателей для поиска
        ilr_search_options rOptions;
        oSearch.GetOptions(rOptions);
        rOptions.nReaderTypes =
            ILR_READER_ILUSB | ILR_READER_CCID | ILR_READER_SERVER | ILR_READER_CLIENT;
        oSearch.SetOptions(rOptions);
        // Включаем поиск считывателей, подключённых к конвертеру в режиме "Клиент"
        const uint16_t aListenPorts[] = {25000};
        oSearch.SetListenPorts(aListenPorts, std::size(aListenPorts));
#endif
        // Подписываемся на уведомления о нахождении/потери считывателей
        oSearch.SetMessageCallback(MessageCallback);
        // Включаем авто поиск считывателей (по умолчанию выключен)
        oSearch.SetAutoScan();
        std::cout << "Ожидание подключения/отключения считывателей..." << std::endl;

        std::cout << "Нажмите <Enter> для выхода..." << std::endl;
        std::cin.get();
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }

    return 0;
}
