#include <iomanip>   // для std::put_time
#include <iostream>  // для std::cout и std::cin

#include "ilr_cpp_helpers.h"
#include "ilreaders.h"

// #define ILR_LOG  // Раскомментируйте, чтобы включить отладочные сообщения
#define ILR_LOG_FILE  // Писать лог в файл

using namespace ilr;

#ifdef ILR_LOG
const char kLogLevelChars[] = {'-', 'E', 'W', 'I', 'D'};
const char kLogFileName[] = "ilreaders.log";  // Путь к лог файлу

void ILR_CALL LogCallback(ilr_log_level level, const char* pContext, const char* pMessage, void*) {
#ifdef ILR_LOG_FILE  // Запись в файл
    std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::app);
    auto& out = file;
#else  // иначе в консоль
    auto& out = std::cout;
#endif
    auto t = std::time(nullptr);
    auto tmb = std::localtime(&t);
    out << std::put_time(tmb, "%d-%m-%Y %H:%M:%S") << " [" << kLogLevelChars[level] << ' '
        << pContext << "] " << pMessage << std::endl;
}
#endif

void ILR_CALL MessageCallback(ilr_reader_msg nMsg, const void*, void* pUserData) {
    if (ILR_READER_MSG_CONNECTION_CHANGED == nMsg) {
        try {
            auto pReader = static_cast<CReader*>(pUserData);
            auto nStatus = pReader->GetConnectionStatus();
            switch (nStatus) {
            case ILR_CONNECTION_DISCONNECTED:
                std::cout << "{!} Считыватель отключён" << std::endl;
                break;

            case ILR_CONNECTION_CONNECTED:
                std::cout << "{!} Считыватель подключён" << std::endl;
                break;

            case ILR_CONNECTION_CONNECTING:
                std::cout << "{!} Идёт подключение к считывателю" << std::endl;
                break;
            }
        }
        catch (const std::exception& e) {
            std::cerr << e.what() << std::endl;
        }
    }
}

ilr_port_type GetPortTypeByName(const char* pszPortName) {
    if (*pszPortName == '\0')
        return ILR_PORT_TYPE_UNKNOWN;
    if (strncmp(pszPortName, "/dev/tty", 8) == 0)
        return ILR_PORT_TYPE_COMPORT;
    auto p = strchr(pszPortName, '@');
    if (p != nullptr) {
        int nSn;
        uint nTcpPort;
        if (sscanf(pszPortName, "%d@%u", &nSn, &nTcpPort) == 2)
            return ILR_PORT_TYPE_CLIENT;
        // 30:00000000@192.168.1.98:25001
        return ILR_PORT_TYPE_PROXY;
    }
    uint a[4];
    if (sscanf(pszPortName, "%u.%u.%u.%u", &a[0], &a[1], &a[2], &a[3]) == 4)
        return ILR_PORT_TYPE_SERVER;
    return ILR_PORT_TYPE_CCID;
}

bool DoConnectTo(CILR& oILR, CReader& oReader) {
    CReaderSearch oSearch(oILR.GetSearch());
    // Ищем считыватели
    std::cout << "Поиск считывателей..." << std::endl;
    oSearch.Scan();
    auto nCount = oSearch.GetReaderCount();
    std::cout << "Найдено " << nCount << ':' << std::endl;

    constexpr int kEnterPortNameCommand = 999;
    int nCommand;
    if (nCount > 0) {
        std::cout << "Выберите считыватель:" << std::endl;
        for (size_t i = 0; i < nCount; i++) {
            ilr_reader_info rInfo;
            oSearch.GetReaderInfo(i, rInfo);

            std::stringstream ss;
            if (rInfo.nModel != ILR_READER_MODEL_UNKNOWN)
                ss << kReaderModelNames[rInfo.nModel];
            if (rInfo.nSn != -1)
                ss << " с/н:" << rInfo.nSn;
            if (rInfo.nFwVersion != 0)
                ss << " прошивка:" << ReaderVersionToStr(rInfo.nFwVersion);
            if (rInfo.nFwBuildDate != 0)
                ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
            std::cout << 1 + i << ". " << rInfo.pszPortName << " []" << rInfo.pszConnect
                      << "]: " << ss.str() << std::endl;
        }
        std::cout << kEnterPortNameCommand << " - Ввести имя порта..." << std::endl;
        std::cout << "0 - Выйти из программы" << std::endl;

        std::cin >> nCommand;
        if (std::cin.fail()) {
            std::cin.clear();
            nCommand = -1;
        }
        std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
    }
    else
        nCommand = kEnterPortNameCommand;
    auto nPortType = ILR_PORT_TYPE_UNKNOWN;
    std::string sPortName;
    auto nConnectModel = ILR_READER_MODEL_UNKNOWN;
    switch (nCommand) {
    case 0:
        return false;

    case kEnterPortNameCommand:
        {
            std::cout << "Введите имя порта считывателя:" << std::endl;
            std::getline(std::cin, sPortName);
            nPortType = GetPortTypeByName(sPortName.c_str());
        }
        break;

    default:
        if ((nCommand > 0) && (static_cast<size_t>(nCommand) <= nCount)) {
            ilr_reader_info rInfo;
            oSearch.GetReaderInfo(static_cast<size_t>(nCommand - 1), rInfo);
            nPortType = rInfo.nPortType;
            sPortName = rInfo.pszPortName;
            nConnectModel = rInfo.nModel;
        }
        break;
    }
    if (nPortType != ILR_PORT_TYPE_UNKNOWN) {
        // Подключаемся к считывателю
        std::cout << "Подключение к считывателю [" << kPortTypeNames[nPortType] << ": " << sPortName
                  << "]..." << std::endl;
        oReader = oILR.GetReader(nPortType, sPortName.c_str());
        // Подписываемся на уведомления от считывателя
        oReader.SetMessageCallback(MessageCallback, &oReader);
        if (nConnectModel != ILR_READER_MODEL_UNKNOWN) {
            ilr_reader_options rOptions;
            oReader.GetOptions(rOptions);
            rOptions.nConnectModel = nConnectModel;
            oReader.SetOptions(rOptions);
        }
        oReader.Connect();
        // Получаем информацию о конвертере
        ilr_reader_info rInfo;
        oReader.GetReaderInfo(rInfo);
        std::stringstream ss;
        if (rInfo.nModel != ILR_READER_MODEL_UNKNOWN)
            ss << kReaderModelNames[rInfo.nModel];
        if (rInfo.nSn != -1)
            ss << " с/н:" << rInfo.nSn;
        if (rInfo.nFwVersion != 0)
            ss << " прошивка:" << ReaderVersionToStr(rInfo.nFwVersion);
        if (rInfo.nFwBuildDate != 0)
            ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
        std::cout << "Считыватель успешно подключён [" << ss.str() << ']' << std::endl;
    }
    return true;
}

int main() {
    try {
#ifdef ILR_LOG
#ifdef ILR_LOG_FILE
        // Очищаем лог файл
        std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::trunc);
        file.close();
#endif
        CILR::SetLogCallback(LogCallback);
        CILR::SetLogLevel(ILR_LOG_LEVEL_DEBUG);
#endif

        CILR oILR;
        CReader oReader;

        if (!DoConnectTo(oILR, oReader))
            return 0;

        while (true) {
            std::cout << "-----" << std::endl;
            std::cout << "Введите номер команды:" << std::endl;
            std::cout << "1 - Подключиться к..." << std::endl;
            std::cout << "2 - Подключиться" << std::endl;
            std::cout << "3 - Отключиться" << std::endl;
            std::cout << "0 - Выйти из программы" << std::endl;

            int nCommand;
            std::cin >> nCommand;
            if (std::cin.fail()) {
                std::cin.clear();
                nCommand = -1;
            }
            std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');

            switch (nCommand) {
            case 0:
                return 0;

            case 1:
                if (!DoConnectTo(oILR, oReader))
                    return 0;
                break;

            case 2:
                oReader.Connect();
                break;

            case 3:
                oReader.Disconnect();
                break;

            default:
                std::cout << "Неправильный ввод" << std::endl;
                break;
            }
        }
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
    return 0;
}
