#include <stdio.h>

#include <iomanip>  // для std::put_time
#include <iostream>

#include "ilr_cpp_helpers.h"
#include "ilreaders.h"

// #define ILR_LOG  // Раскомментируйте, чтобы включить отладочные сообщения
#define ILR_LOG_FILE  // Писать лог в файл

using namespace ilr;

#ifdef ILR_LOG
const char kLogLevelChars[] = {'-', 'E', 'W', 'I', 'D'};
const char kLogFileName[] = "ilreaders.log";  // Путь к лог файлу

void ILR_CALL LogCallback(ilr_log_level level, const char* pContext, const char* pMessage, void*) {
#ifdef ILR_LOG_FILE  // Запись в файл
    std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::app);
    auto& out = file;
#else  // иначе в консоль
    auto& out = std::cout;
#endif
    auto t = std::time(nullptr);
    auto tmb = std::localtime(&t);
    out << std::put_time(tmb, "%d-%m-%Y %H:%M:%S") << " [" << kLogLevelChars[level] << ' '
        << pContext << "] " << pMessage << std::endl;
}
#endif

void ILR_CALL MessageCallback(ilr_reader_msg nMsg, const void* pMsgData, void*) {
    if ((ILR_READER_MSG_CARD_FOUND == nMsg) || (ILR_READER_MSG_CARD_LOST == nMsg)) {
        try {
            const ilr_card_info* pInfo = (const ilr_card_info*)pMsgData;
            std::stringstream ss;
            if (pInfo->nMpType != ILR_MF_PLUS_UNKNOWN)
                ss << " " << kMpTypeNames[pInfo->nMpType];
            if (pInfo->nSL != ILR_MF_PLUS_SL_UNKNOWN)
                ss << " SL" << static_cast<int>(pInfo->nSL);

            if (ILR_READER_MSG_CARD_FOUND == nMsg)
                std::cout << "{!} Карта найдена " << kCardTypeNames[pInfo->nType] << ' '
                          << CardUIDToStr(pInfo->nType, pInfo->rUID) << ss.str() << std::endl;
            else
                std::cout << "{!} Карта потеряна " << kCardTypeNames[pInfo->nType] << ' '
                          << CardUIDToStr(pInfo->nType, pInfo->rUID) << ss.str() << std::endl;
        }
        catch (const std::exception& e) {
            std::cerr << e.what() << std::endl;
        }
    }
}

int main() {
    try {
#ifdef ILR_LOG
#ifdef ILR_LOG_FILE
        // Очищаем лог файл
        std::ofstream file(kLogFileName, std::ios_base::out | std::ios_base::trunc);
        file.close();
#endif
        CILR::SetLogCallback(LogCallback);
        CILR::SetLogLevel(ILR_LOG_LEVEL_DEBUG);
#endif

        CILR oILR;

        // Ищем считыватель
        CReader oReader;
        {
            std::cout << "Поиск считывателя..." << std::endl;
            CReaderSearch oSearch(oILR.GetSearch());
#if 0
            ilr_search_options rOptions;
            oSearch.GetOptions(rOptions);
            rOptions.nReaderTypes |= ILR_READER_CCID;
            oSearch.SetOptions(rOptions);
#endif
            oSearch.Scan();
            auto nCount = oSearch.GetReaderCount();
            bool fFound = false;
            ilr_reader_info rInfo;
            for (size_t i = 0; i < nCount; i++) {
                oSearch.GetReaderInfo(i, rInfo);
                // Если порт занят, то пропускаем
                if (*rInfo.pszConnect != '\0')
                    continue;
                fFound = true;
                break;
            }

            if (!fFound) {
                std::cout << "Считыватель не найден" << std::endl;
                return 0;
            }
            oReader = oILR.GetReader(rInfo.nPortType, rInfo.pszPortName);
            // Подключаемся к считывателю
            std::cout << "Подключение к считывателю [" << kPortTypeNames[rInfo.nPortType] << ": "
                      << rInfo.pszPortName << "]..." << std::endl;
            oReader.Connect();
        }
        // Получаем информацию о считывателе
        ilr_reader_info rInfo;
        oReader.GetReaderInfo(rInfo);
        std::stringstream ss;
        if (rInfo.nModel != ILR_READER_MODEL_UNKNOWN)
            ss << kReaderModelNames[rInfo.nModel];
        if (rInfo.nSn != -1)
            ss << " с/н:" << rInfo.nSn;
        if (rInfo.nFwVersion != 0)
            ss << " прошивка:" << ReaderVersionToStr(rInfo.nFwVersion);
        if (rInfo.nFwBuildDate != 0)
            ss << " сборка " << TimeToStr(rInfo.nFwBuildDate);
        std::cout << "Считыватель успешно подключён [" << ss.str() << ']' << std::endl;

        // Подписываемся на уведомления о поднесении/удалении карты
        oReader.SetMessageCallback(MessageCallback, &oReader);
        // Включаем авто поиск карт (нет необходимости, т.к. включен по умолчанию)
        oReader.SetAutoScan(true, false);
        // Ожидаем поднесение/удаление карты
        std::cout << "Ожидание поднесения карты..." << std::endl;

        std::cout << "Нажмите <Enter> для выхода..." << std::endl;
        std::cin.get();
    }
    catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }
    return 0;
}
